﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h> // for nn::sf::NativeHandle
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/tcap/server/tcap_IManager.sfdl.h> // for IManager
#include <nn/tcap/tcap_OperationModeTypes.h>
#include <nn/tcap/tcap_ServiceName.h>

namespace nn { namespace tcap {

namespace {

const int NumberOfClients = 1;
sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<server::IManager> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

nn::os::Mutex g_ManagerAccessMutex(false);

int g_InitializationCount = 0;

} // namespace

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_SDK_ASSERT(g_InitializationCount >= 0);
    if ( g_InitializationCount == 0 )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::tcap::ServiceName));
    }
    g_InitializationCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_SDK_ASSERT(g_InitializationCount > 0);
    g_InitializationCount--;
    if ( g_InitializationCount == 0 )
    {
        g_Holder.FinalizeHolder();
    }
}

void InitializeWith(nn::sf::SharedPointer<server::IManager>&& manager) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_SDK_ASSERT(g_InitializationCount >= 0);
    if ( g_InitializationCount == 0 )
    {
        g_Holder.InitializeHolderDirectly(std::move(manager));
    }
    g_InitializationCount++;
}

void GetContinuousHighSkinTemperatureEvent(nn::os::SystemEventType* pOutEvent, nn::os::EventClearMode mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    nn::sf::NativeHandle sfHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder.GetObject()->GetContinuousHighSkinTemperatureEvent(&sfHandle));
    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfHandle.GetOsHandle(), sfHandle.IsManaged(), mode);
    sfHandle.Detach();
}

void SetOperationMode(OperationMode operationMode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    g_Holder.GetObject()->SetOperationMode(operationMode);
}

void LoadAndApplySettings() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ManagerAccessMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    g_Holder.GetObject()->LoadAndApplySettings();
}

}} // namespace nn::tcap
