﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/hid/system/hid_InputDetection.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/tc/tc.h>
#include <nn/tcap/tcap_OperationModeTypes.h>

#include "tcap_ThermalCoordinatorAdvancedPolicy.h"

namespace nn { namespace tcap { namespace server {

namespace {

nn::os::Mutex g_ExternalLibraryInitializationMutex(false);

int g_InitializationCount = 0;

} // namespace

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_SDK_ASSERT(g_InitializationCount >= 0);
    if ( g_InitializationCount == 0 )
    {
        nn::tc::Initialize();
        nn::hid::system::InitializeInputDetector();
    }
    g_InitializationCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_SDK_ASSERT(g_InitializationCount > 0);
    g_InitializationCount--;
    if ( g_InitializationCount == 0 )
    {
        // FinalizeInputDetector は存在しない
        nn::tc::Finalize();
    }
}

void Update(nn::TimeSpan elapsedTime) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    GetThermalCoordinatorAdvancedPolicy().Update(elapsedTime);
}

void ClearElapsedTime() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    GetThermalCoordinatorAdvancedPolicy().ClearElapsedTime();
}

nn::os::SystemEventType* GetContinuousHighSkinTemperatureEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    return GetThermalCoordinatorAdvancedPolicy().GetContinuousHighSkinTemperatureEvent();
}

void SetOperationMode(OperationMode operationMode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    GetThermalCoordinatorAdvancedPolicy().SetOperationMode(operationMode);
}

void LoadAndApplySettings() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_ExternalLibraryInitializationMutex);

    NN_ABORT_UNLESS(g_InitializationCount > 0);
    GetThermalCoordinatorAdvancedPolicy().LoadAndApplySettings();
}

}}} // namespace nn::tcap::server
