﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/tc/impl/tc.h>
#include <nn/tc/impl/tc_PowerModeTypes.h>
#include "detail/tc_ThermalCoordinator.h"

namespace nn { namespace tc { namespace impl {

namespace {

// Initialize と Finalize は守る。
int g_InitializeCount = 0;

struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

} // namespace

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {
        detail::GetThermalCoordinator().Initialize();
    }

    g_InitializeCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    g_InitializeCount--;

    if ( g_InitializeCount == 0 )
    {
        detail::GetThermalCoordinator().Finalize();
    }
}

nn::Result SetOperatingMode(OperatingMode operatingMode) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetOperatingMode(operatingMode));
    NN_RESULT_SUCCESS;
}

nn::Result GetEventPtr(nn::os::SystemEventType** pOutEventPtr, EventTarget target) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().GetEventPtr(pOutEventPtr, target));
    NN_RESULT_SUCCESS;
}

nn::Result IsActionAllowed(bool* pOutPossible, Action action) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().IsActionAllowed(pOutPossible, action));
    NN_RESULT_SUCCESS;
}

nn::Result EnableVirtualTemperature(Location location) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetVirtualTemperatureEnabled(location, true));
    NN_RESULT_SUCCESS;
}

nn::Result DisableVirtualTemperature(Location location) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetVirtualTemperatureEnabled(location, false));
    NN_RESULT_SUCCESS;
}

nn::Result SetVirtualTemperature(Location location, TemperatureMilliC temperature) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetVirtualTemperature(location, temperature));
    NN_RESULT_SUCCESS;
}

void Update() NN_NOEXCEPT
{
    detail::GetThermalCoordinator().Update();
}

void SetPowerMode(PowerMode powerMode) NN_NOEXCEPT
{
    detail::GetThermalCoordinator().SetPowerMode(powerMode);
}

nn::Result EnableFanControl() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetFanControlEnabled(true));
    NN_RESULT_SUCCESS;
}

nn::Result DisableFanControl() NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().SetFanControlEnabled(false));
    NN_RESULT_SUCCESS;
}

nn::Result IsFanControlEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetThermalCoordinator().GetFanControlEnabled(pOutEnabled));
    NN_RESULT_SUCCESS;
}

TemperatureMilliC GetSkinTemperatureMilliC() NN_NOEXCEPT
{
    return detail::GetThermalCoordinator().GetSkinTemperatureMilliC();
}

}}} // namespace nn::tc::impl
