﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fan/fan.h>
#include <nn/tc/tc_Types.h>

#include "tc_FirmwareDebugSettingsAccessor.h"

namespace nn { namespace tc { namespace impl { namespace detail {

struct TskinCoefficients
{
    uint64_t a;
    uint64_t b;
};

enum class TskinSelect
{
    Soc,
    Pcb,
    Both,
};

struct TemperatureLevel
{
    TemperatureMilliC minTemperature;
    TemperatureMilliC maxTemperature;
    nn::fan::RotationSpeedLevel minSpeedLevel;
    nn::fan::RotationSpeedLevel maxSpeedLevel;
};

enum RateSelect
{
    Table,
    TableHandheld,
    Prev,
    Both,
};

class SettingsHolder
{
    NN_DISALLOW_COPY(SettingsHolder);
    NN_DISALLOW_MOVE(SettingsHolder);

private:
    static constexpr int MaxNumberOfTemperatureLevel = 0x10;

public:
    SettingsHolder() NN_NOEXCEPT;

    //! settings から設定を取得します。
    void LoadSettings() NN_NOEXCEPT;

    //! SoC 側センサ用 IIR フィルタのゲイン値を取得します。
    int GetIirFilterGainSoc() const NN_NOEXCEPT;

    //! PCB 側センサ用 IIR フィルタのゲイン値を取得します。
    int GetIirFilterGainPcb() const NN_NOEXCEPT;

    //! 携帯モードで SoC 温度から表面温度を導出する為の係数を取得します。
    const TskinCoefficients& GetTskinCoefficientsSocHandheld() const NN_NOEXCEPT;

    //! 携帯モードで PCB 温度から表面温度を導出する為の係数を取得します。
    const TskinCoefficients& GetTskinCoefficientsPcbHandheld() const NN_NOEXCEPT;

    //! 据置モードで SoC 温度から表面温度を導出する為の係数を取得します。
    const TskinCoefficients& GetTskinCoefficientsSocConsole() const NN_NOEXCEPT;

    //! 据置モードで PCB 温度から表面温度を導出する為の係数を取得します。
    const TskinCoefficients& GetTskinCoefficientsPcbConsole() const NN_NOEXCEPT;

    //! どの表面温度を採用するかの設定を取得します。
    TskinSelect GetTskinSelect() const NN_NOEXCEPT;

    //! 携帯モードの温度レベルテーブルのレベル数を取得します。
    int GetNumberOfTemperatureLevelHandheld() const NN_NOEXCEPT;

    //! 携帯モードの温度レベルテーブルのレベル構造体を取得します。
    const TemperatureLevel& GetTemperatureLevelHandheld(int index) const NN_NOEXCEPT;

    //! 据置モードの温度レベルテーブルのレベル数を取得します。
    int GetNumberOfTemperatureLevelConsole() const NN_NOEXCEPT;

    //! 据置モードの温度レベルテーブルのレベル構造体を取得します。
    const TemperatureLevel& GetTemperatureLevelConsole(int index) const NN_NOEXCEPT;

    //! どのポリシーから導出した回転レートを採用するかの設定を取得します。
    RateSelect GetRateSelect() const NN_NOEXCEPT;

    //! 評価用ログ出力が有るか取得します。
    bool IsLogEnabled() const NN_NOEXCEPT;

    //! スリープの有効・無効を取得します。
    bool IsSleepEnabled() const NN_NOEXCEPT;

private:
    template <typename T>
    void ReadSetting(T* pOutValue, const char* pKey) NN_NOEXCEPT
    {
        m_FirmwareDebugSettingsAccessor.ReadSetting(pOutValue, pKey);
        NN_DETAIL_TC_TRACE("%s : %d\n", pKey, *pOutValue);
    }

    void LoadTskinCoefficients(TskinCoefficients* pOutTskinCoefficients, const char* pKey) NN_NOEXCEPT;

    void LoadTskinSelect() NN_NOEXCEPT;

    void LoadTemperatureLevelTable(TemperatureLevel* pOutTemperatureLevelTable, int* pOutNumberOfValiLevel, const char* pKey) NN_NOEXCEPT;

    void LoadRateSelect() NN_NOEXCEPT;

private:
    //! SoC 側センサ用 IIR フィルタのゲイン値
    int m_IirFilterGainSoc;

    //! PCB 側センサ用 IIR フィルタのゲイン値
    int m_IirFilterGainPcb;

    //! 携帯モードで SoC 温度から表面温度を導出する為の係数
    TskinCoefficients m_TskinCoefficientsSocHandheld;

    //! 携帯モードで PCB 温度から表面温度を導出する為の係数
    TskinCoefficients m_TskinCoefficientsPcbHandheld;

    //! 据置モードで SoC 温度から表面温度を導出する為の係数
    TskinCoefficients m_TskinCoefficientsSocConsole;

    //! 据置モードで PCB 温度から表面温度を導出する為の係数
    TskinCoefficients m_TskinCoefficientsPcbConsole;

    //! どの表面温度を採用するかの設定
    TskinSelect m_TskinSelect;

    //! 携帯モードの温度レベルテーブルのレベル数
    int m_NumberOfTemperatureLevelHandheld;

    //! 携帯モードの温度レベルテーブル
    TemperatureLevel m_TemperatureLevelTableHandheld[MaxNumberOfTemperatureLevel];

    //! 据置モードの温度レベルテーブルのレベル数
    int m_NumberOfTemperatureLevelConsole;

    //! 据置モードの温度レベルテーブル
    TemperatureLevel m_TemperatureLevelTableConsole[MaxNumberOfTemperatureLevel];

    //! どのポリシーから導出した回転レートを採用するかの設定。
    RateSelect m_RateSelect;

    //! 評価用ログ出力の有無
    bool m_LogEnabled;

    //! スリープの有効・無効
    bool m_SleepEnabled;

    //! 設定値へのアクセサ
    FirmwareDebugSettingsAccessor m_FirmwareDebugSettingsAccessor;
};

}}}} // namespace nn::tc::impl::detail
