﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <string>
#include <utility>

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>

#include <nn/fan/fan.h>
#include <nn/tc/detail/tc_Log.h>
#include <nn/tc/tc_Types.h>

#include "tc_Parser.h"
#include "tc_SettingsHolder.h"

namespace nn { namespace tc { namespace impl { namespace detail {

const TemperatureMilliC MinTemperatureMilliC = static_cast<TemperatureMilliC>(std::numeric_limits<int>::min());
const TemperatureMilliC MaxTemperatureMilliC = static_cast<TemperatureMilliC>(std::numeric_limits<int>::max());
const size_t TemperatureLevelTableBufferSize = 0x400;

// tskin_rate_table_handheld および tskin_rate_table_console 読み込み用のバッファ。
char g_TemperatureLevelTableBuffer[TemperatureLevelTableBufferSize];

SettingsHolder::SettingsHolder() NN_NOEXCEPT
    : m_IirFilterGainSoc(100)
    , m_IirFilterGainPcb(100)
    , m_TskinCoefficientsSocHandheld{5464LL, 174190LL}
    , m_TskinCoefficientsPcbHandheld{5817LL, 129580LL}
    , m_TskinCoefficientsSocConsole{6182LL, 112480LL}
    , m_TskinCoefficientsPcbConsole{6396LL, 119440LL}
    , m_TskinSelect(TskinSelect::Both)
    , m_NumberOfTemperatureLevelHandheld(6)
    , m_TemperatureLevelTableHandheld
    // 温度範囲と対応する nn::fan::RotationSpeedLevel を低温側から記述します。
    {
        { MinTemperatureMilliC, 40000,   0,   0}, // 0%
        { 36000,                43000,  51,  51}, // 20%
        { 43000,                48000,  51, 102}, // 20% - 40%
        { 48000,                53000, 102, 153}, // 40% - 60%
        { 53000, MaxTemperatureMilliC, 153, 153}, // 60%
        { 48000, MaxTemperatureMilliC, 153, 153}, // 60%
    }
    , m_NumberOfTemperatureLevelConsole(5)
    , m_TemperatureLevelTableConsole
    {
        // SIGLO-42182: 冷却が不要な時もファンが回転し続けることは、一般的にファンの寿命の観点で望ましくありません。
        // NX では想定する使用方法でファンの寿命が問題にならないことは確認できていますが、方針を残すために修正を投入します。
        { MinTemperatureMilliC, 40000,   0,   0}, // 0%
        { 36000,                43000,  51,  51}, // 20%
        { 43000,                53000,  51, 153}, // 20% - 60%
        { 53000,                58000, 153, 255}, // 60% - 100%
        { 58000, MaxTemperatureMilliC, 255, 255}, // 100%
    }
    , m_RateSelect(RateSelect::Both)
    , m_LogEnabled(false)
    , m_SleepEnabled(true)
{
    // 何も処理しません。
}

void SettingsHolder::LoadTskinCoefficients(TskinCoefficients* pOutTskinCoefficients, const char* pKey) NN_NOEXCEPT
{
    const size_t BufferSize = 0x40;
    char buffer[BufferSize];

    memset(&(buffer[0]), '\0', sizeof(buffer));
    auto size = m_FirmwareDebugSettingsAccessor.ReadSettingString(&(buffer[0]), sizeof(buffer), pKey);

    if ( size <= 0 )
    {
        NN_DETAIL_TC_WARN("%s : field is not readable. tc uses default coefficients.\n", pKey);
        return;
    }

    NN_DETAIL_TC_TRACE("%s : %s.\n", pKey, buffer);

    const int NestTarget = 1;
    const int NumberOfCoefficients = 2;
    int coefficients[NumberOfCoefficients];
    auto parsedNumber = ParseStringToSequence(coefficients, NumberOfCoefficients, buffer, sizeof(buffer), NestTarget);

    if ( NumberOfCoefficients != parsedNumber )
    {
        NN_DETAIL_TC_WARN("%s : field is incorrect. tc uses default coefficients.\n", pKey);
    }
    else
    {
        pOutTskinCoefficients->a = static_cast<int64_t>(coefficients[0]);
        pOutTskinCoefficients->b = static_cast<int64_t>(coefficients[1]);
    }
}

void SettingsHolder::LoadTskinSelect() NN_NOEXCEPT
{
    const size_t BufferSize = 0x10;
    char buffer[BufferSize];

    memset(&(buffer[0]), '\0', sizeof(buffer));
    auto size = m_FirmwareDebugSettingsAccessor.ReadSettingString(&(buffer[0]), sizeof(buffer), "tskin_select");

    if ( size <= 0 )
    {
        NN_DETAIL_TC_WARN("tskin_select : field is not readable. tc uses both temperatures.\n");
        return;
    }

    NN_DETAIL_TC_TRACE("tskin_select : %s.\n", buffer);

    const std::pair<const char*, TskinSelect> TskinSelectMappingArray[] =
    {
        { "soc",  TskinSelect::Soc  },
        { "pcb",  TskinSelect::Pcb  },
        { "both", TskinSelect::Both },
    };

    for ( const std::pair<const char*, TskinSelect> tskinSelectMapping : TskinSelectMappingArray )
    {
        if ( tskinSelectMapping.first == std::string(buffer) )
        {
            m_TskinSelect = tskinSelectMapping.second;
            return;
        }
    }

    NN_DETAIL_TC_WARN("tskin_select : field is incorrect. tc uses both temperatures.\n");
}

void SettingsHolder::LoadTemperatureLevelTable(TemperatureLevel* pOutTemperatureLevelTable, int* pOutNumberOfValidLevel, const char* pKey) NN_NOEXCEPT
{
    const int NestTarget = 2;

    std::memset(g_TemperatureLevelTableBuffer, '\0', TemperatureLevelTableBufferSize);
    size_t size = settings::fwdbg::GetSettingsItemValue(g_TemperatureLevelTableBuffer,
        TemperatureLevelTableBufferSize, "tc", pKey);

    if ( size <= 0 )
    {
        NN_DETAIL_TC_WARN("%s : field is not readable. tc uses default table.\n", pKey);
    }
    else
    {
        NN_DETAIL_TC_TRACE("%s : %s.\n", pKey, g_TemperatureLevelTableBuffer);

        const int ValuePerLevel = 4;
        int list[MaxNumberOfTemperatureLevel * ValuePerLevel];
        int numberOfValidLevel = ParseStringToSequence(list,
            MaxNumberOfTemperatureLevel * ValuePerLevel, g_TemperatureLevelTableBuffer, size, NestTarget) / ValuePerLevel;

        // リストが存在しないか要素が不十分な長さであればデフォルトのテーブルを使用する。
        if ( numberOfValidLevel == 0 )
        {
            NN_DETAIL_TC_WARN("%s : field is incorrect. tc uses default table.\n", pKey);
        }
        else
        {
            for ( int index = 0; index < numberOfValidLevel; index ++ )
            {
                (pOutTemperatureLevelTable + index)->minTemperature = static_cast<TemperatureMilliC>(list[index * ValuePerLevel]);
                (pOutTemperatureLevelTable + index)->maxTemperature = static_cast<TemperatureMilliC>(list[index * ValuePerLevel + 1]);
                (pOutTemperatureLevelTable + index)->minSpeedLevel = static_cast<nn::fan::RotationSpeedLevel>(list[index * ValuePerLevel + 2]);
                (pOutTemperatureLevelTable + index)->maxSpeedLevel = static_cast<nn::fan::RotationSpeedLevel>(list[index * ValuePerLevel + 3]);
            }

            *pOutNumberOfValidLevel = numberOfValidLevel;

            NN_DETAIL_TC_TRACE("Table has %d TemperatureLevels.\n", numberOfValidLevel);

            for ( int index = 0; index < numberOfValidLevel; index ++ )
            {
                NN_DETAIL_TC_TRACE("TemperatureLevel[%d] = {%d, %d, %d, %d};\n", index,
                    (pOutTemperatureLevelTable + index)->minTemperature,
                    (pOutTemperatureLevelTable + index)->maxTemperature,
                    (pOutTemperatureLevelTable + index)->minSpeedLevel,
                    (pOutTemperatureLevelTable + index)->maxSpeedLevel);
            }
        }
    }
}

void SettingsHolder::LoadRateSelect() NN_NOEXCEPT
{
    const size_t BufferSize = 0x10;
    char buffer[BufferSize];

    memset(&(buffer[0]), '\0', sizeof(buffer));
    auto size = m_FirmwareDebugSettingsAccessor.ReadSettingString(&(buffer[0]), sizeof(buffer), "rate_select");

    if ( size <= 0 )
    {
        NN_DETAIL_TC_WARN("rate_select : field is not readable. tc uses both temperatures.\n");
        return;
    }

    NN_DETAIL_TC_TRACE("rate_select : %s.\n", buffer);

    const std::pair<const char*, RateSelect> RateSelectMappingArray[] =
    {
        { "table",          RateSelect::Table         },
        { "table-handheld", RateSelect::TableHandheld },
        { "prev",           RateSelect::Prev          },
        { "both",           RateSelect::Both          },
    };

    for ( const std::pair<const char*, RateSelect> rateSelectMapping : RateSelectMappingArray )
    {
        if ( rateSelectMapping.first == std::string(buffer) )
        {
            m_RateSelect = rateSelectMapping.second;
            return;
        }
    }

    NN_DETAIL_TC_WARN("rate_select : field is incorrect. tc uses both temperatures.\n");
}

void SettingsHolder::LoadSettings() NN_NOEXCEPT
{
    ReadSetting(&m_IirFilterGainSoc, "iir_filter_gain_soc");
    ReadSetting(&m_IirFilterGainPcb, "iir_filter_gain_pcb");

    LoadTskinCoefficients(&m_TskinCoefficientsSocHandheld, "tskin_soc_coefficients_handheld");
    LoadTskinCoefficients(&m_TskinCoefficientsPcbHandheld, "tskin_pcb_coefficients_handheld");
    LoadTskinCoefficients(&m_TskinCoefficientsSocConsole, "tskin_soc_coefficients_console");
    LoadTskinCoefficients(&m_TskinCoefficientsPcbConsole, "tskin_pcb_coefficients_console");

    LoadTskinSelect();

    LoadTemperatureLevelTable(&(m_TemperatureLevelTableHandheld[0]), &m_NumberOfTemperatureLevelHandheld, "tskin_rate_table_handheld");
    LoadTemperatureLevelTable(&(m_TemperatureLevelTableConsole[0]), &m_NumberOfTemperatureLevelConsole, "tskin_rate_table_console");

    LoadRateSelect();

    ReadSetting(&m_LogEnabled, "log_enabled");
    ReadSetting(&m_SleepEnabled, "sleep_enabled");
}

int SettingsHolder::GetIirFilterGainSoc() const NN_NOEXCEPT
{
    return m_IirFilterGainSoc;
}

int SettingsHolder::GetIirFilterGainPcb() const NN_NOEXCEPT
{
    return m_IirFilterGainPcb;
}

const TskinCoefficients& SettingsHolder::GetTskinCoefficientsSocHandheld() const NN_NOEXCEPT
{
    return m_TskinCoefficientsSocHandheld;
}

const TskinCoefficients& SettingsHolder::GetTskinCoefficientsPcbHandheld() const NN_NOEXCEPT
{
    return m_TskinCoefficientsPcbHandheld;
}

const TskinCoefficients& SettingsHolder::GetTskinCoefficientsSocConsole() const NN_NOEXCEPT
{
    return m_TskinCoefficientsSocConsole;
}

const TskinCoefficients& SettingsHolder::GetTskinCoefficientsPcbConsole() const NN_NOEXCEPT
{
    return m_TskinCoefficientsPcbConsole;
}

TskinSelect SettingsHolder::GetTskinSelect() const NN_NOEXCEPT
{
    return m_TskinSelect;
}

int SettingsHolder::GetNumberOfTemperatureLevelHandheld() const NN_NOEXCEPT
{
    return m_NumberOfTemperatureLevelHandheld;
}

const TemperatureLevel& SettingsHolder::GetTemperatureLevelHandheld(int index) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(index < m_NumberOfTemperatureLevelHandheld);
    NN_ABORT_UNLESS(index >= 0);
    return m_TemperatureLevelTableHandheld[index];
}

int SettingsHolder::GetNumberOfTemperatureLevelConsole() const NN_NOEXCEPT
{
    return m_NumberOfTemperatureLevelConsole;
}

const TemperatureLevel& SettingsHolder::GetTemperatureLevelConsole(int index) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(index < m_NumberOfTemperatureLevelConsole);
    NN_ABORT_UNLESS(index >= 0);
    return m_TemperatureLevelTableConsole[index];
}

RateSelect SettingsHolder::GetRateSelect() const NN_NOEXCEPT
{
    return m_RateSelect;
}

bool SettingsHolder::IsLogEnabled() const NN_NOEXCEPT
{
    return m_LogEnabled;
}

bool SettingsHolder::IsSleepEnabled() const NN_NOEXCEPT
{
    return m_SleepEnabled;
}

}}}} // namespace nn::tc::impl::detail
