﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_Abort.h>

#include <nn/pcv/pcv.h>
#include <nn/tc/detail/tc_Log.h>
#include <nn/tc/tc_Types.h>

#include "tc_SettingHandler.h"

namespace nn { namespace tc { namespace impl { namespace detail {

int SettingHandler::CheckThreshold(TemperatureMilliC temperature) NN_NOEXCEPT
{
    nn::pcv::MilliC rangeMax = 0;
    nn::pcv::MilliC rangeMin = 0;

    if ( m_CurrentThresholdIndex == 0 )
    {
        rangeMin = std::numeric_limits<nn::pcv::MilliC>::min();
    }
    else
    {
        rangeMin = m_TemperatureThresholds[m_CurrentThresholdIndex - 1].minMilliC;
    }

    if ( m_CurrentThresholdIndex == m_NumberOfThresholds )
    {
        rangeMax = std::numeric_limits<nn::pcv::MilliC>::max();
    }
    else
    {
        rangeMax = m_TemperatureThresholds[m_CurrentThresholdIndex].maxMilliC;
    }

    // 等号で処理しているのでヒステリシス 0 は非推奨。
    // temperature のアラインに沿って同じ上限値と下限値が設定された場合振動する。
    if ( static_cast<nn::pcv::MilliC>(temperature) <= rangeMin )
    {
        return -1;
    }
    else if ( static_cast<nn::pcv::MilliC>(temperature) >= rangeMax )
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

void SettingHandler::Initialize() NN_NOEXCEPT
{
    nn::pcv::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::pcv::GetTemperatureThresholds(m_TemperatureThresholds, &m_NumberOfThresholds, nn::pcv::MaxNumTemperatureThresholds));

    for ( int index = 0; index < m_NumberOfThresholds; index++ )
    {
        NN_DETAIL_TC_TRACE("TemperatureThresholdsForPcv[%d] = {%d, %d};\n", index,
            m_TemperatureThresholds[index].minMilliC, m_TemperatureThresholds[index].maxMilliC);
    }

    m_ForceUpdateTemperature = true;
}

void SettingHandler::Finalize() NN_NOEXCEPT
{
    m_ForceUpdateTemperature = false;

    nn::pcv::Finalize();
}

void SettingHandler::SetTemperature(TemperatureMilliC temperature) NN_NOEXCEPT
{
    bool updateTemperature = false;

    // Index を変更する必要があるか確認する。
    int moveIndex = CheckThreshold(temperature);

    // 遷移回数の上限は m_NumberOfThresholds
    for ( int count = 0; count < m_NumberOfThresholds && moveIndex != 0; count++ )
    {
        updateTemperature = true;

        // 上限、下限値が適切にハンドリングされていない場合 ABORT に落としている。
        NN_ABORT_UNLESS(m_CurrentThresholdIndex <= m_NumberOfThresholds,  "Incorrect TemperatureThresholds!");
        NN_ABORT_UNLESS(m_CurrentThresholdIndex >= 0, "Incorrect TemperatureThresholds!");

        m_CurrentThresholdIndex += moveIndex;

        //NN_DETAIL_TC_TRACE_V1("Current threshold index: %d(%d MilliC).\n", m_CurrentThresholdIndex, temperature);

        moveIndex = CheckThreshold(temperature);
    }

    if ( updateTemperature || m_ForceUpdateTemperature )
    {
        //NN_DETAIL_TC_TRACE_V1("Update PCV temperature(%d MilliC).\n", temperature);

        nn::pcv::SetTemperature(static_cast<nn::pcv::MilliC>(temperature));
        m_LastDvfsTemperature = temperature;
        m_ForceUpdateTemperature = false;
    }
}

void SettingHandler::SetPowerMode(PowerMode powerMode) NN_NOEXCEPT
{
    PowerMode prePowerMode = m_PowerMode;
    m_PowerMode = powerMode;

    // Sleep から出るときに温度を更新する。
    if ( m_PowerMode == PowerMode_MinimumAwake && prePowerMode == PowerMode_SleepReady )
    {
        //NN_DETAIL_TC_TRACE_V1("Transit from SleepReady to MinimumAwake.\n");
        m_ForceUpdateTemperature = true;
    }
}

TemperatureMilliC SettingHandler::GetLastDvfsTemperature() NN_NOEXCEPT
{
    return m_LastDvfsTemperature;
}

}}}} // namespace nn::tc::impl::detail
