﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/ssl_BuiltInManager.h>

#include "detail/ssl_ServiceSession.h"
#include "server/ssl_Util.h"

using namespace nn::ssl::detail;

namespace nn { namespace ssl {


nn::Result BuiltInManager::GetBuiltInCertificates(BuiltInCertificateInfo  **pOutCertInfo,
                                                  uint8_t                 *pOutBuffer,
                                                  uint32_t                bufSize,
                                                  CaCertificateId         *pInIds,
                                                  uint32_t                idCount)
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    outIdCount = idCount;

    do
    {
        if ((pOutBuffer == nullptr) ||
            (pOutCertInfo == nullptr) ||
            (pInIds == nullptr))
        {
            result = ResultInvalidPointer();
            break;
        }

        if (bufSize == 0)
        {
            result = ResultBufferTooShort();
            break;
        }

        if (idCount == 0)
        {
            result = ResultInvalidIdCount();
            break;
        }

        SharedPointer<ISslService>  *controlIf = ServiceSession::GetServiceSession();
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(controlIf, result, ResultLibraryNotInitialized());

        InBuffer  inBuf(reinterpret_cast<const char *>(pInIds),
                        sizeof(CaCertificateId) * idCount);
        OutBuffer outBuf(reinterpret_cast<char *>(pOutBuffer), bufSize);
        result = (*controlIf)->GetCertificates(outBuf, &outIdCount, inBuf);
        if (result.IsFailure())
        {
            break;
        }

        //  The out buffer is now filled with an array of
        //  BuiltInCertificateInfo, one for each cert.  However, SF won't
        //  handle pointer translation within the buffer for us, so we
        //  need to convert each entry's data into a valid pointer.
        //  The reserved1 field contains an offset from the start of the
        //  buffer where the actual cert DER data is located.
        BuiltInCertificateInfo *pCertInfoArray =
            reinterpret_cast<BuiltInCertificateInfo *>(pOutBuffer);
        for (uint32_t i = 0; i < outIdCount; i++)
        {
            pCertInfoArray[i].data.ptr.pCertificateDerData =
                pOutBuffer + pCertInfoArray[i].data.priv.reserved1;
        }

        *pOutCertInfo = pCertInfoArray;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result BuiltInManager::GetBuiltInCertificateBufSize(uint32_t         *pOutBufSize,
                                                        CaCertificateId  *pInIds,
                                                        uint32_t         idCount)
{
    nn::Result                  result = nn::ResultSuccess();
    uint32_t                    bufSize = 0;

    do
    {
        if ((pOutBufSize == nullptr) ||
            (pInIds == nullptr))
        {
            result = ResultInvalidPointer();
            break;
        }

        if (idCount == 0)
        {
            result = ResultInvalidIdCount();
            break;
        }

        SharedPointer<ISslService>  *controlIf = ServiceSession::GetServiceSession();
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(controlIf, result, ResultLibraryNotInitialized());

        InBuffer buf(reinterpret_cast<const char *>(pInIds),
                     sizeof(CaCertificateId) * idCount);
        result = (*controlIf)->GetCertificateBufSize(&bufSize, buf);
        if (result.IsSuccess())
        {
            *pOutBufSize = bufSize;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

} }    //  nn::ssl
