﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstdlib>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/ssl_Api.debug.h>

#include "detail/ssl_ServiceSession.h"
#include "server/ssl_Util.h"

namespace nn { namespace ssl {
// ------------------------------------------------------------------------------------------------
// HeapTrackPoint
// ------------------------------------------------------------------------------------------------
Debug::HeapTrackPoint::HeapTrackPoint()
{
    snprintf(_name, NameSize - 1, "%lx", reinterpret_cast<unsigned long>(this));
    _name[NameSize - 1] = 0x00;
}


// ------------------------------------------------------------------------------------------------
// Public interface
// ------------------------------------------------------------------------------------------------
nn::Result Debug::Ioctl(
    Output* pOutBufferInfo,
    Input*  pInBufferInfo,
    IoctlCommand cmd) NN_NOEXCEPT
{
    nn::Result  result        = nn::ResultSuccess();
    char*       pOutBuffer    = nullptr;
    const char* pInBuffer     = nullptr;
    size_t      outBufferSize = 0;
    size_t      inBufferSize  = 0;
    char        dummyOutBuffer;
    char        dummyInBuffer;

    do
    {
        SharedPointer<ISslService>  *controlIf = detail::ServiceSession::GetServiceSession();
        NN_DETAIL_SSL_VALIDATE_SHARED_POINTER(controlIf, result, ResultLibraryNotInitialized());

        if (pOutBufferInfo != nullptr)
        {
            if (pOutBufferInfo->pBuffer == nullptr)
            {
                result = ResultInvalidPointer();
                break;
            }

            if (pOutBufferInfo->bufferSize == 0)
            {
                result = ResultBufferTooShort();
                break;
            }

            pOutBuffer    = pOutBufferInfo->pBuffer;
            outBufferSize = pOutBufferInfo->bufferSize;
        }
        else
        {
            pOutBuffer    = &dummyOutBuffer;
            outBufferSize = sizeof(dummyOutBuffer);
        }

        if (pInBufferInfo != nullptr)
        {
            if (pInBufferInfo->pBuffer == nullptr)
            {
                result = ResultInvalidPointer();
                break;
            }

            if (pInBufferInfo->bufferSize == 0)
            {
                result = ResultBufferTooShort();
                break;
            }

            pInBuffer    = pInBufferInfo->pBuffer;
            inBufferSize = pInBufferInfo->bufferSize;
        }
        else
        {
            pInBuffer    = &dummyInBuffer;
            inBufferSize = sizeof(dummyInBuffer);
        }

        OutBuffer outBuf(reinterpret_cast<char*>(pOutBuffer), outBufferSize);
        InBuffer  inBuf(reinterpret_cast<const char*>(pInBuffer), inBufferSize);

        result = (*controlIf)->DebugIoctl(outBuf, inBuf, static_cast<uint64_t>(cmd));
        if (result.IsFailure())
        {
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

}}
