﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ssl_SslTrustedCertManager.h"
#include "ssl_Util.h"


namespace nn { namespace ssl { namespace detail {

//////////////////////////////////////////////////////////////////////////////
//  TcfNullLoader - a Trusted Cert loader which does nothing
//////////////////////////////////////////////////////////////////////////////
TcfNullLoader::TcfNullLoader()
{
}

TcfNullLoader::~TcfNullLoader()
{
}

uint32_t TcfNullLoader::GetDataBufSize()
{
    return 0;
}


int TcfNullLoader::Load(uint8_t   *pOutBuf,
                        uint32_t  maxBufSize,
                        uint32_t *pOutDataSize)
{
    *pOutDataSize = 0;
    return 0;
}


//////////////////////////////////////////////////////////////////////////////
//  TcfSourceLoader - a Trusted Cert loader which loads data from the BDF
//////////////////////////////////////////////////////////////////////////////
TcfSourceLoader::TcfSourceLoader(BdfMetaDataEntry *pCertEntry) NN_NOEXCEPT
{
    m_CertEntry = *pCertEntry;
}


uint32_t TcfSourceLoader::GetDataBufSize()
{
    return m_CertEntry.dataSize;
}


int TcfSourceLoader::Load(uint8_t   *pOutBuf,
                          uint32_t  maxBufSize,
                          uint32_t  *pOutDataSize)
{
    int                         ret = -1;
    nn::Result                  result;
    BdfReader                   *pReader = nullptr;

    do
    {
        if (maxBufSize < m_CertEntry.dataSize)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TcfSourceLoader::Load] invalid buf size. Got %u, need %u\n",
                                    maxBufSize,
                                    m_CertEntry.dataSize);
            break;
        }

        //  Get a BDF reader for the trusted cert data
        pReader = BdfReaderFactory::Create(BdfReader::BdfId_TrustedCerts);
        if (pReader == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TcfSourceLoader::Load] failed to get TCF reader\n");
            break;
        }

        result = pReader->GetData(pOutBuf, maxBufSize, &m_CertEntry);

        //  Destroy the reader regardless of what happens
        delete pReader;
        pReader = nullptr;
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[TcfSourceLoader::Load] failed get trusted cert data for id %8.8X: %d-%d\n",
                                    m_CertEntry.dataId,
                                    result.GetModule(),
                                    result.GetDescription());
        }

        *pOutDataSize = m_CertEntry.dataSize;
        ret = 0;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


//////////////////////////////////////////////////////////////////////////////
//  NssSourceLoader - a Trusted Cert loader which loads data from the NSS
//                    CERTCertificate object containing this cert's DER data.
//////////////////////////////////////////////////////////////////////////////
NssSourceLoader::NssSourceLoader(CERTCertificate *pCert) NN_NOEXCEPT
{
    m_pCert = CERT_DupCertificate(pCert);
}


NssSourceLoader::~NssSourceLoader()
{
    CERT_DestroyCertificate(m_pCert);
    m_pCert = nullptr;
}


uint32_t NssSourceLoader::GetDataBufSize()
{
    uint32_t                    ret;

    ret = static_cast<uint32_t>(m_pCert->derCert.len);
    return ret;
}


int NssSourceLoader::Load(uint8_t   *pOutBuf,
                          uint32_t  maxBufSize,
                          uint32_t  *pOutCertSize)
{
    int                         ret = -1;
    SECItem                     *pDerCert = &m_pCert->derCert;

    do
    {
        if (maxBufSize < pDerCert->len)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssSourceLoader::Load] invalid buf size. Got %u, need %u\n",
                                    maxBufSize,
                                    pDerCert->len);
            break;
        }

        memcpy(pOutBuf, pDerCert->data, pDerCert->len);
        ret = 0;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


//////////////////////////////////////////////////////////////////////////////
//  TrustedCertManager - Top level manager for all Trusted Certificates
//////////////////////////////////////////////////////////////////////////////
const char                      *TrustedCertManager::g_TcmDeferredInitId = "tcm";
const char                      *TrustedCertManager::g_RootCaTrust = "CT,CT,CT";
const char                      *TrustedCertManager::g_IntermediateCaTrust = "P,P,P";
TrustedCertManager              *TrustedCertManager::g_pInstance = nullptr;
nn::os::Mutex                   TrustedCertManager::g_Lock(false);


TrustedCertManager::TrustedCertManager(PLHashTable  *pTcHashTbl,
                                       CertStore    *pCertStore) :
    m_TcCount(0),
    m_pTcHashTbl(pTcHashTbl),
    m_pCertStore(pCertStore)
{
}


TrustedCertManager::~TrustedCertManager()
{
    delete m_pCertStore;
    m_pCertStore = nullptr;

    PL_HashTableDestroy(m_pTcHashTbl);
    m_pTcHashTbl = nullptr;
}


BuiltinDataInfo *TrustedCertManager::FindTrustedCertInfo(CaCertificateId id)
{
    BuiltinDataInfo             *pBdi = nullptr;
    void                        *pEntry = nullptr;

    do
    {
        pEntry = PL_HashTableLookup(m_pTcHashTbl,
                                    reinterpret_cast<const void *>(id));
        if (pEntry != nullptr)
        {
            pBdi = reinterpret_cast<BuiltinDataInfo *>(pEntry);
        }
    } while (NN_STATIC_CONDITION(false));

    return pBdi;
}


nn::Result TrustedCertManager::ImportTrustedCert(CERTCertificate   **pOutCert,
                                                 BdfMetaDataEntry  *pInCertEntry,
                                                 BdfReader         *pReader)
{
    nn::Result                  result = nn::ResultSuccess();
    uint8_t                     *pCertData = nullptr;
    uint64_t                    tmpId;
    CERTCertList                *pCertList = nullptr;
    CERTCertListNode            *pCertNode = nullptr;
    CERTCertificate             *pCert = nullptr;
    CERTCertDBHandle            *pCertDb = nullptr;
    CERTCertTrust               trust;
    SECStatus                   secStatus;

    do
    {
        //  Allocate a buffer large enough for the data
        pCertData = new uint8_t[pInCertEntry->dataSize];
        if (pCertData == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR failed to create data buffer for cert idx\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Read the data from the file, import into NSS by first importing
        //  it into our CertStore then add to the default trust used by NSS.
        result = pReader->GetData(pCertData,
                                  pInCertEntry->dataSize,
                                  pInCertEntry);
        if (!result.IsSuccess())
        {
            //  Error already logged and result set to error
            break;
        }

        result = m_pCertStore->ImportServerPki(&tmpId,
                                               reinterpret_cast<const char *>(pCertData),
                                               pInCertEntry->dataSize,
                                               CertificateFormat_Der);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR failed to import trusted: %d-%d\n",
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }

        pCertList = CERT_NewCertList();
        if (pCertList == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR failed to create scratch cert list\n");
            result = ResultInsufficientMemory();
            break;
        }

        result = m_pCertStore->GetTrustedCerts(pCertList, tmpId);
        if ((!result.IsSuccess()) || CERT_LIST_EMPTY(pCertList))
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR failed to find imported cert\n");
            break;
        }

        //  There should only be a single cert in the list as each of our
        //  trusted certs is stand-alone, not multiple in a single DER or
        //  a chain.
        pCertNode = CERT_LIST_HEAD(pCertList);
        pCert = pCertNode->cert;

        //  Setup the trust to use based on whether this cert is a root or not
        secStatus =
            CERT_DecodeTrustString(&trust,
                                   (pCert->isRoot == PR_TRUE) ?
                                       g_RootCaTrust : g_IntermediateCaTrust);
        if (secStatus != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR decoding %s trust string\n",
                                    (pCert->isRoot == PR_TRUE) ? "ROOT" : "INTERMED");
            result = ResultErrorLower();
            break;
        }

        pCertDb = CERT_GetDefaultCertDB();
        if (pCertDb == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR unable to get default cert db!\n");
            result = ResultErrorLower();
            break;
        }

        //  Add it to the store as a perm certificate.  No nickname is needed
        //  as one was requested when the temp cert was created and NSS will
        //  use it internally.
        secStatus = CERT_AddTempCertToPerm(pCert, NULL, &trust);
        if (secStatus != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportTrustedCert] ERROR unable to change trust of cert\n");
            result = ResultErrorLower();
            break;
        }

        *pOutCert = pCert;
    } while (NN_STATIC_CONDITION(false));

    if (pCertList != nullptr)
    {
        CERT_DestroyCertList(pCertList);
        pCertList = nullptr;
    }

    if (pCertData != nullptr)
    {
        delete[] pCertData;
        pCertData = nullptr;
    }

    return result;
}


nn::Result TrustedCertManager::ProcessTrustedCertsFile()
{
    nn::Result                          result = nn::ResultSuccess();
    BdfReader                           *pReader = nullptr;
    BuiltinDataInfo                     *pCurBdi;
    uint32_t                            dataCount;

    do
    {
        //  Get an instance of the BDF reader so we can process the contents
        pReader = BdfReaderFactory::Create(BdfReader::BdfId_TrustedCerts);
        if (pReader == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ProcessTrustedCertsFile] ERROR failed to get TCF reader\n");
            break;
        }

        result = pReader->GetDataCount(&dataCount);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ProcessTrustedCertsFile] ERROR reading hdr: %d-%d\n",
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }

        //  Walk each cert entry and look for the certs and make entries for
        //  each.
        for (uint32_t i = 0; i < dataCount; i++)
        {
            PLHashEntry         *pNewEntry = nullptr;

            pCurBdi = nullptr;
            result = CreateTrustedCertInfo(&pCurBdi, pReader, i);
            if (!result.IsSuccess())
            {
                break;
            }

            intptr_t tmp = static_cast<intptr_t>(pCurBdi->GetId());
            pNewEntry = PL_HashTableAdd(m_pTcHashTbl,
                                        reinterpret_cast<const void *>(tmp),
                                        reinterpret_cast<void *>(pCurBdi));
            if (pNewEntry == nullptr)
            {
                result = ResultInsufficientMemory();
                break;
            }

            m_TcCount++;
        }

    } while (NN_STATIC_CONDITION(false));

    if (pReader != nullptr)
    {
        delete pReader;
        pReader = nullptr;
    }

    return result;
}


PLHashNumber TrustedCertManager::GenHashFromId(const void *key)
{
    intptr_t                    tmp;

    tmp = reinterpret_cast<intptr_t>(key);
    return static_cast<PLHashNumber>(tmp);
}


nn::Result TrustedCertManager::CreateTrustedCertInfo(BuiltinDataInfo  **pOutCertInfo,
                                                     BdfReader        *pReader,
                                                     uint32_t         certIdx)
{
    nn::Result                  result = nn::ResultSuccess();
    BdfMetaDataEntry            certEntry;
    nn::ssl::TrustedCertStatus  status;
    BuiltinDataLoader           *pLoader = nullptr;
    BuiltinDataInfo             *pNewCertInfo = nullptr;
    CERTCertificate             *pNssCert = nullptr;

    do
    {
        //  Get the meta data entry for the specific data index
        result = pReader->GetMetaDataEntry(&certEntry, certIdx);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] faiiled to get meta data for cert index %u: %d-%d\n",
                                    certIdx,
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }

        //  The cert entry will tell us all we need to know about the size and
        //  creating the correct loader.
        status = static_cast<nn::ssl::TrustedCertStatus>(certEntry.dataStatus);
        if (status == nn::ssl::TrustedCertStatus_EnabledTrusted)
        {
            //  Import the trusted cert into NSS' default trust and get
            //  back the CERTCertificate.
            result = ImportTrustedCert(&pNssCert,
                                       &certEntry,
                                       pReader);
            if (!result.IsSuccess())
            {
                NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR failed to import trusted cert %u, %d-%d\n",
                                        certIdx,
                                        result.GetModule(),
                                        result.GetDescription());
                break;
            }

            //  Create a NssSourceLoader linked to the CERTCertificate
            pLoader = new NssSourceLoader(pNssCert);
            if (pLoader == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR failed to create NssSourceLoader for cert idx %d\n",
                                        certIdx);
                result = ResultInsufficientMemory();
                break;
            }
        }
        else if (status == nn::ssl::TrustedCertStatus_EnabledNotTrusted)
        {
            //  Create a TcfSourceLoader with the current info
            pLoader = new TcfSourceLoader(&certEntry);
            if (pLoader == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR failed to create TcfSourceLoader for cert idx %d\n",
                                        certIdx);
                result = ResultInsufficientMemory();
                break;
            }
        }
        else if ((status == nn::ssl::TrustedCertStatus_Removed) ||
                 (status == nn::ssl::TrustedCertStatus_Revoked))
        {
            //  Create a NullLoader as there will never be data loaded
            //  for a cert of this status.
            pLoader = new TcfNullLoader();
            if (pLoader == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR failed to create TcfNullLoader for cert idx %d\n",
                                        certIdx);
                result = ResultInsufficientMemory();
                break;
            }
        }
        else
        {
            //  Skip this entry completely, bail out - it is INVALID!
            NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR found entry with id 0x%8.8X, invalid status %X\n",
                                    certEntry.dataId,
                                    certEntry.dataStatus);
            result = ResultInvalidCertStoreId();
            break;
        }

        nn::ssl::detail::BuiltinDataInfo::BuiltinDataStatus bdstatus =
            static_cast<nn::ssl::detail::BuiltinDataInfo::BuiltinDataStatus>(status);
        pNewCertInfo = new BuiltinDataInfo(certEntry.dataId,
                                           bdstatus,
                                           nullptr,
                                           pLoader);
        if (pNewCertInfo == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[CreateTrustedCertInfo] ERROR failed to create TrustedCertInfo for cert idx %d\n",
                                    certIdx);
            result = ResultInsufficientMemory();
            break;
        }

        //  Give the caller the new trust info, we're done!
        *pOutCertInfo = pNewCertInfo;
    } while (NN_STATIC_CONDITION(false));

    //  If we got a NSS cert, destroy our reference since the loader
    //  will have created its own.
    if (pNssCert != nullptr)
    {
        CERT_DestroyCertificate(pNssCert);
        pNssCert = nullptr;
    }

    if (!result.IsSuccess())
    {
        if (pLoader != nullptr)
        {
            delete pLoader;
            pLoader = nullptr;
        }
    }

    return result;
}    //  NOLINT(impl/function_size)


NssCore::InitStatus TrustedCertManager::TcmDeferredInitCb(void *pInArg)
{
    NssCore::InitStatus         status = NssCore::InitStatus_InitDone;
    nn::Result                  result;

    do
    {
        NN_DETAIL_SSL_DBG_PRINT("[TcmDeferredInitCb] processing trusted cert BDF...\n");
        result = TrustedCertManager::g_pInstance->ProcessTrustedCertsFile();
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[TcmDeferredInitCb] ERROR: unable to process trusted certs BDF\n");
            status = NssCore::InitStatus_InitFail;
            break;
        }
    } while(NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[TcmDeferredInitCb] DONE (%d)\n", status);
    return status;
}


nn::Result TrustedCertManager::Initialize()
{
    nn::Result                          result = nn::ResultSuccess();
    uint8_t                             *pCacheBuf = nullptr;
    CertStore                           *pCertStore = nullptr;
    PLHashTable                         *pHashTbl = nullptr;

    g_Lock.Lock();

    do
    {
        if (TrustedCertManager::g_pInstance != nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initialize] WARNING, already initialized!\n");
            break;
        }

        //  Init the BDF reader factory here so it is ready to go when we need
        //  an instance.
        result = BdfReaderFactory::Initialize();
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initialize] ERROR: failed to init BDF reader: %d-%d\n",
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }

        //  Creaate a hash table (provided by NSPR).  The key is the cert ID
        //  as a 32 bit integer, the value is a pointer to a new TrustedCertInfo
        //  object.  Therefore, the key and value comparators can just be
        //  simple value comparators.
        pHashTbl = PL_NewHashTable(0,
                                   TrustedCertManager::GenHashFromId,
                                   PL_CompareValues,
                                   PL_CompareValues,
                                   nullptr,
                                   nullptr);
        if (pHashTbl == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initialize] ERROR: failed to create trusted cert info map\n");
            result = ResultInsufficientMemory();
            break;
        }

        pCertStore = new CertStore(g_CertStoreFakePid);
        if (pCertStore == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initialize] ERROR: failed to create CertStore\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Create the singleton instance then tell it to process the input
        TrustedCertManager::g_pInstance =
            new TrustedCertManager(pHashTbl, pCertStore);
        if (g_pInstance == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initilize] ERROR: unable to alloc memory\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Add our deferred init callback to NssCore's deferred init
        result = NssCore::AddDeferredInit(g_TcmDeferredInitId,
                                          TrustedCertManager::TcmDeferredInitCb,
                                          nullptr);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::Initialize] ERROR: unable to setup deferred init\n");
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    g_Lock.Unlock();

    if (!result.IsSuccess())
    {
        if (TrustedCertManager::g_pInstance != nullptr)
        {
            delete TrustedCertManager::g_pInstance;
            TrustedCertManager::g_pInstance = nullptr;
        }
        else
        {
            if (pCertStore != nullptr)
            {
                delete pCertStore;
                pCertStore = nullptr;
            }

            if (pHashTbl != nullptr)
            {
                PL_HashTableDestroy(pHashTbl);
                pHashTbl = nullptr;
            }

            //  Have the reader finalize, in case it was successfully
            //  initialized.
            BdfReaderFactory::Finalize();

            if (pCacheBuf != nullptr)
            {
                delete[] pCacheBuf;
                pCacheBuf = nullptr;
            }
        }
    }

    return result;
}


nn::Result TrustedCertManager::Finalize()
{
    nn::Result                  result;

    g_Lock.Lock();

    if (TrustedCertManager::g_pInstance != nullptr)
    {
        delete TrustedCertManager::g_pInstance;
        TrustedCertManager::g_pInstance = nullptr;
    }

    result = BdfReaderFactory::Finalize();

    g_Lock.Unlock();

    return result;
}


nn::Result TrustedCertManager::GetTrustedCertInfo(BuiltinDataInfo  **pOutBdi,
                                                  CaCertificateId  id)
{
    nn::Result                  result = nn::ResultSuccess();
    BuiltinDataInfo             *pBdi = nullptr;

    g_Lock.Lock();

    do
    {
        if (g_pInstance == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::GetTrustedCertInfo] not initialized!\n");
            result = ResultErrorLower();
            break;
        }

        pBdi = g_pInstance->FindTrustedCertInfo(id);
        if (pBdi == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::GetTrustedCertInfo] unable to find BDI for ID %8.8X\n",
                                    id);
            result = ResultInvalidCertStoreId();
            break;
        }

        *pOutBdi = pBdi;
    } while (NN_STATIC_CONDITION(false));

    g_Lock.Unlock();

    return result;
}


PRIntn TrustedCertManager::TcmHashEnumerator(PLHashEntry *pEntry, PRIntn i, void *pArg)
{
    PRIntn                      ret = HT_ENUMERATE_NEXT;
    TcmIdInfo                   *pInfo = reinterpret_cast<TcmIdInfo *>(pArg);
    uint64_t                    index = static_cast<uint64_t>(i);
    CaCertificateId             curId;

    do
    {
        //  Ensure the current item is within bounds
        if (index >= pInfo->maxIds)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::TcmHashEnumerator] index beyond max: %u, %u\n",
                                    index,
                                    pInfo->maxIds);
            ret = HT_ENUMERATE_STOP;
            break;
        }

        //  Copy the current entry's key (ID)
        intptr_t key = reinterpret_cast<intptr_t>(pEntry->key);
        curId = static_cast<CaCertificateId>(key);
        pInfo->pIdBuf[index] = curId;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


void TrustedCertManager::TcmAscSortIdArray(CaCertificateId  *pArray,
                                           int              arraySize)
{
    int                         i;
    int                         j;

    //  Simple sort from least to greatest
    for (i = 1; i < arraySize; i++)
    {
        if (pArray[i] < pArray[i - 1])
        {
            CaCertificateId tmp = pArray[i];
            pArray[i] = pArray[i - 1];
            pArray[i - 1] = tmp;

            for (j = i - 1; j > 0; j--)
            {
                if (pArray[j] < pArray[j - 1])
                {
                    tmp = pArray[j];
                    pArray[j] = pArray[j - 1];
                    pArray[j - 1] = tmp;
                }

            }
        }
    }
}


bool TrustedCertManager::IsAllId(CaCertificateId id)
{
    return (id == nn::ssl::CaCertificateId_All);
}


nn::Result TrustedCertManager::GetAllIds(CaCertificateId **pOutIds, uint32_t *pOutNumIds)
{
    nn::Result                  result = ResultSuccess();
    PRIntn                      copiedIds;
    TcmIdInfo                   info = { 0 };

    g_Lock.Lock();

    do
    {
        if (g_pInstance == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::GetAllIds] not initialized!\n");
            result = ResultErrorLower();
            break;
        }

        info.maxIds = g_pInstance->m_TcCount;
        info.pIdBuf = new CaCertificateId[static_cast<size_t>(g_pInstance->m_TcCount)];
        if (info.pIdBuf == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::GetAllIds] failed to alloc mem for IDs\n");
            result = ResultInsufficientMemory();
            break;
        }

        copiedIds = PL_HashTableEnumerateEntries(g_pInstance->m_pTcHashTbl,
                                                 TrustedCertManager::TcmHashEnumerator,
                                                 &info);
        if (copiedIds != static_cast<PRIntn>(g_pInstance->m_TcCount))
        {
            NN_DETAIL_SSL_DBG_PRINT("[TrustedCertManager::GetAllIds] total copied mismatch!\n");
            result = ResultErrorLower();
            break;
        }

        TrustedCertManager::TcmAscSortIdArray(info.pIdBuf, copiedIds);
        *pOutIds = info.pIdBuf;
        *pOutNumIds = static_cast<uint32_t>(copiedIds);

        //  null out the ID buffer we allocated as it is up to the caller to
        //  delete the memory when we return success
        info.pIdBuf = nullptr;
    } while (NN_STATIC_CONDITION(false));

    g_Lock.unlock();

    if (info.pIdBuf != nullptr)
    {
        delete[] info.pIdBuf;
        info.pIdBuf = nullptr;
    }

    return result;
}


} } }    //  nn::ssl::detail
