﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/ssl_Connection.h>

#include "ssl_SslSession.h"
#include "ssl_NssCommon.h"
#include "ssl_Util.h"

namespace nn { namespace ssl { namespace detail {
// ------------------------------------------------------------------------------------------------
// SslSessionManager
// ------------------------------------------------------------------------------------------------
nn::Result SslSessionManager::SetSessionCacheMode(
    PRFileDesc* pInSocket,
    SessionCacheMode mode)
{
    NN_SDK_REQUIRES_NOT_NULL(pInSocket);
    if(pInSocket == nullptr)
    {
        return ResultInvalidReference();
    }

    nn::Result result = nn::ResultSuccess();
    SECStatus  status = SECSuccess;
    switch(mode)
    {
    case SessionCacheMode::SessionCacheMode_None:
        status = SSL_OptionSet(pInSocket, SSL_NO_CACHE, PR_TRUE);
        if(status != SECSuccess)
        {
            break;
        }
        status = SSL_OptionSet(pInSocket, SSL_ENABLE_SESSION_TICKETS, PR_FALSE);
        break;
    case SessionCacheMode::SessionCacheMode_SessionId:
        status = SSL_OptionSet(pInSocket, SSL_NO_CACHE, PR_FALSE);
        break;
    case SessionCacheMode::SessionCacheMode_SessionTicket:
        status = SSL_OptionSet(pInSocket, SSL_NO_CACHE, PR_FALSE);
        if(status != SECSuccess)
        {
            break;
        }
        status = SSL_OptionSet(pInSocket, SSL_ENABLE_SESSION_TICKETS, PR_TRUE);
        break;
    default:
        NN_DETAIL_SSL_DBG_PRINT("(SslSessionManager::SetSessionCache) Invalid mode was passed (%d).\n",
            mode);
        result = ResultInternalLogicError();
        break;
    }

    if((status != SECSuccess) || result.IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("(SslSessionManager::SetSessionCache) failed - mode:%d PR_Error:%d(%s)\n",
            mode, PR_GetError(), PR_ErrorToName(PR_GetError()));
        result = ResultErrorLower();
    }

    return result;
}

nn::Result SslSessionManager::GetSessionCacheMode(PRFileDesc* pInSocket, SessionCacheMode* pOutMode)
{
    NN_SDK_REQUIRES_NOT_NULL(pInSocket);
    NN_SDK_REQUIRES_NOT_NULL(pOutMode);
    if(pInSocket == nullptr || pOutMode == nullptr)
    {
        return ResultInvalidReference();
    }

    nn::Result result = nn::ResultSuccess();
    PRBool     optionValue;

    do
    {
        SECStatus status = SSL_OptionGet(pInSocket, SSL_NO_CACHE, &optionValue);
        if(status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("(SslSessionManager::GetSessionCacheMode) failed - PR_Error:%d(%s)\n",
                PR_GetError(), PR_ErrorToName(PR_GetError()));
            result = ResultErrorLower();
            break;
        }

        if(optionValue == PR_FALSE)
        {
            // Let's return this if session ID is enabled because session ticket
            // cannot be enabled at the same time
            *pOutMode = SessionCacheMode::SessionCacheMode_SessionId;
            break;
        }

        status = SSL_OptionGet(pInSocket, SSL_ENABLE_SESSION_TICKETS, &optionValue);
        if(status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("(SslSessionManager::GetSessionCacheMode) failed - PR_Error:%d(%s)\n",
                PR_GetError(), PR_ErrorToName(PR_GetError()));
            result = ResultErrorLower();
            break;
        }

        if(optionValue == PR_TRUE)
        {
            *pOutMode = SessionCacheMode::SessionCacheMode_SessionTicket;
            break;
        }

        // Nothing is enabled
        *pOutMode = SessionCacheMode::SessionCacheMode_None;
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result SslSessionManager::DeleteSessionCache(PRFileDesc* pInDescriptor)
{
    nn::Result result = ResultSuccess();
    do
    {
        if(pInDescriptor == nullptr)
        {
            result = ResultInternalLogicError();
            break;
        }

        // Removes the current session on a particular SSL socket from the session
        // DeleteSessionCache After SSL_InvalidateSession is called, the existing connection
        // using the session can continue, but no new connections can resume this SSL session.
        SECStatus status = SSL_InvalidateSession(pInDescriptor);
        if(status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("(SslSessionManager::DeleteSessionCache) failed - PR_Error%d(%s)\n",
                PR_GetError(), PR_ErrorToName(PR_GetError()));
            // This can happen only when there's no SSL connection
            result = ResultNoSslConnection();
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result SslSessionManager::GetSessionId(
    char* pOutSessionIdBuff,
    uint64_t* pOutsessionIdSize,
    PRFileDesc* pInDescriptor,
    uint64_t sessionIdBuffSize)
{
    SECItem*   pItem = nullptr;
    nn::Result result = ResultSuccess();
    do
    {
        if(pInDescriptor == nullptr || pOutSessionIdBuff == nullptr)
        {
            result = ResultInternalLogicError();
            break;
        }

        pItem = SSL_GetSessionID(pInDescriptor);
        if(pItem == nullptr)
        {
            result = ResultInternalNoSessionIdFound();
            break;
        }

        if(pItem->len > sessionIdBuffSize)
        {
            result = ResultBufferTooShort();
            if(pOutsessionIdSize != nullptr)
            {
                *pOutsessionIdSize = pItem->len;
            }
            break;
        }

        memcpy(pOutSessionIdBuff, pItem->data, pItem->len);
    } while (NN_STATIC_CONDITION(false));

    if(pItem != nullptr)
    {
        SECITEM_FreeItem(pItem, PR_TRUE);
    }

    return result;
}

}}} // namespace nn { namespace ssl { namespace detail {
