﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SystemThreadDefinition.h>

#include "server/ssl_SslServiceManager.h"
#include "detail/ssl_ISslServiceDfcFactory.h"

#include "server/ssl_NssCommon.h"

namespace nn { namespace ssl { namespace detail {


SslServiceManager::SslServiceManager()
{
}


SslServiceManager::~SslServiceManager()
{
}


nn::Result SslServiceManager::Initialize()
{
    nn::Result                  ret = ResultSuccess();

    do
    {
        //  Init our ports used for SF/HIPC sessions
        ret = InitializePort(0,
                             SslServiceManagerLimits::g_MaxServerSessions,
                             nn::ssl::detail::SslServiceSessionName);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslServiceManager::Initialize] failed to init ports: %d-%d\n",
                                    ret.GetModule(),
                                    ret.GetDescription());
            break;
        }

        //  Start our HIPC manager base
        this->Start();

        //  Create our threads used to handle sessions
        for (uint32_t i = 0; i < g_SessionThreadCount; i++)
        {
#ifdef USE_NNOS_THREAD_DIRECT
            m_SessionThreadAlloc[i] = new uint8_t[g_SessionThreadStackSize + os::StackRegionAlignment];
            if (m_SessionThreadAlloc[i] == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslServiceManager::Initialize] failed to alloc heap for thread %d\n", i);
                ret = ResultInsufficientMemory();
                break;
            }

            if (sizeof(uint8_t *) == 4)
            {
                m_SessionThreadStack[i] =
                    (uint8_t *)((uint32_t)(m_SessionThreadAlloc[i] + (os::StackRegionAlignment - 1)) &
                                (uint32_t)((uint8_t *)~(os::StackRegionAlignment - 1)));
            }
            else
            {
                m_SessionThreadStack[i] =
                    (uint8_t *)((uint64_t)(m_SessionThreadAlloc[i] + (os::StackRegionAlignment - 1)) &
                                (uint64_t)((uint8_t *)~(os::StackRegionAlignment - 1)));
            }

            ret = nn::os::CreateThread(&m_SessionThreads[i],
                                       SessionThreadEntry,
                                       this,
                                       m_SessionThreadStack[i],
                                       g_SessionThreadStackSize,
                                       NN_SYSTEM_THREAD_PRIORITY(ssl, SessionThread));
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslServiceManager::Initialize] unable to create thread %d\n", i);
                break;
            }

            nn::os::SetThreadNamePointer(&m_SessionThreads[i],
                                         NN_SYSTEM_THREAD_NAME(ssl, SessionThread));

            NN_DETAIL_SSL_DBG_PRINT("[SslServiceManager::Initialize] thread %p, stack %p (%p)\n",
                                    &m_SessionThreads[i],
                                    m_SessionThreadStack[i],
                                    m_SessionThreadAlloc[i]);
#else
            m_SessionThreads[i] = PR_CreateThread(PR_SYSTEM_THREAD,
                                                  SessionThreadEntry,
                                                  this,
                                                  PR_PRIORITY_URGENT,
                                                  PR_GLOBAL_THREAD,
                                                  PR_JOINABLE_THREAD,
                                                  g_SessionThreadStackSize);
            if (m_SessionThreads[i] == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslServiceManager::Initialize] unable to create thread %d\n", i);
            }
#endif
        }

        if (ret.IsFailure())
        {
            break;
        }

#ifdef USE_NNOS_THREAD_DIRECT
        //  Start the threads, get them ready and listening
        for (uint32_t i = 0; i < g_SessionThreadCount; i++)
        {
            nn::os::StartThread(&m_SessionThreads[i]);
        }
#endif
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result SslServiceManager::Finalize()
{
    //  TODO: Stop threads and free their stack space
    return ResultSuccess();
}


nn::Result SslServiceManager::OnNeedsToAccept(int portIndex, PortForAllInOne *pPort) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    ISslServiceDfcFactory       factory;

    NN_DETAIL_SSL_DBG_PRINT("[OnNeedsToAccept] thread %p, portIndex %d\n",
                            nn::os::GetCurrentThread(),
                            portIndex);
    do
    {
        SharedPointer<ISslService>  spService;

        //  Create a new DFC instance of the service.  This is the top level
        //  client connection to our service (the session!)
        ret = factory.CreateISslService(&spService);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[OnNeedsToAccept] failed to create service instance: %d-%d\n",
                                    ret.GetModule(),
                                    ret.GetDescription());
            break;
        }

        //  Accept the HIPC connection and tie it to the new service instance
        NN_DETAIL_SSL_DBG_PRINT("[OnNeedsToAccept] calling AcceptImpl...\n");
        ret = AcceptImpl(pPort, spService);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[OnNeedsToAccept] failed accept: %d-%d\n",
                                    ret.GetModule(),
                                    ret.GetDescription());
            spService = nullptr;
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[OnNeedsToAccept] accepted, return success\n");
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


void SslServiceManager::SessionThreadEntry(void *arg) NN_NOEXCEPT
{
    SslServiceManager           *pThis = reinterpret_cast<SslServiceManager *>(arg);

    NN_DETAIL_SSL_DBG_PRINT("[ssl SessionThreadEntry] entered for thread %p\n",
                            nn::os::GetCurrentThread());

#ifndef USE_NNOS_THREAD_DIRECT
    /*  NSPR only allows the thread name to be set from the thread itself,
        so set it right at the start.  */
    PR_SetCurrentThreadName(NN_SYSTEM_THREAD_NAME(ssl, SessionThread));
#endif

    /*  Drop into the HipcSimpleAllInOneServerManager::LoopAuto() call
        to start listening for incoming SF connections.  */
    pThis->LoopAuto();

    NN_DETAIL_SSL_DBG_PRINT("[ssl SessionThreadEntry] exiting for thread %p\n",
                            nn::os::GetCurrentThread());
}

} } }
