﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/os.h>

#include <nn/ssl/ssl_Connection.h>
#include <nn/ssl/detail/ssl_ISslConnection.h>

#include "server/ssl_ServiceDatabase.h"
#include "server/ssl_NssCommon.h"
#include "server/ssl_SslSfObservable.h"
#include "server/ssl_SslContextImpl.h"
#include "server/ssl_NssUtil.h"

using namespace nn::sf;
using namespace nn::ssl::sf;

namespace nn { namespace ssl { namespace detail {


class SslConnectionImpl : public nn::sf::ISharedObject,
                          public SslSfObservable
{
private:
    nn::sf::SharedPointer<SslContextImpl>   m_Parent;
    PRFileDesc                              *m_SslSocket;
    int                                     m_Socket;
    char                                    m_HostName[MaxHostNameLength + 1];
    Connection::VerifyOption                m_VerifyOptionMask;
    Connection::IoMode                      m_IoMode;

    uint8_t                                 *m_pAuthCertData;
    uint32_t                                m_AuthCertMaxSize;
    uint32_t                                m_AuthCertSizeNeeded;
    uint32_t                                m_authCertSize;
    uint32_t                                m_authCertCount;

    nn::Result                              m_AuthCertError;
    nn::Result                              m_LastAuthCertError;
    NssUtil::ErrorList                      m_errorList;
    nn::os::Mutex                           m_ioModeLock;

    bool                                    m_IsDoNotCloseSocketInShim;
    bool                                    m_GetServerCertChain;
    bool                                    m_IsSkipDefaultValidation;

    //  Internal SSL callback methods
    static SECStatus SslHandshakeCb(PRFileDesc *socket, void *arg);
    static SECStatus SslAuthCertCb(void       *arg,
                                   PRFileDesc *socket,
                                   PRBool     checkSig,
                                   PRBool     isServer);
    static SECStatus SslBadCertCb(void *arg, PRFileDesc *socket);
    static SECStatus SslAuthClientCb(void              *arg,
                                     PRFileDesc        *socket,
                                     CERTDistNames     *caNames,
                                     CERTCertificate   **outCert,
                                     SECKEYPrivateKey  **outKey);

    //  Internal helper methods
    nn::Result SetNssSslSocketOptions(nn::ssl::Context::SslVersion version) NN_NOEXCEPT;
    PRIntervalTime GetIoTimeout() NN_NOEXCEPT;
    nn::Result DoHandshake(uint32_t *outCertSize,
                           uint32_t *outCertCount,
                           void     *outCertBuf,
                           uint32_t bufSize) NN_NOEXCEPT;
    nn::Result Read(uint32_t        *bytesRead,
                    void            *outBuf,
                    uint32_t        outBufSize,
                    PRIntn          flags,
                    PRIntervalTime  timeout);

    nn::Result PrepServerCertChainBuf();
    uint32_t GetCertSizeForChain(CERTCertificate *pCert);
    uint32_t GetCertDataOffsetForChain(uint32_t certCount);
    uint32_t AddCertToChainBuf(CERTCertificate *pCert, uint32_t curOffset);
    nn::Result SetPeerId() NN_NOEXCEPT;

    enum ErrorPriority
    {
        ErrorPriority_NoError = 0,
        ErrorPriority_Lowest  = 1,
        ErrorPriority_Low     = 2,
        ErrorPriority_Medium  = 3,
        ErrorPriority_High    = 4,
    };

    static ErrorPriority ConvertErrorToPriorityValue(PRErrorCode error) NN_NOEXCEPT;
    static const char* GetSslVersionStrFromInt(uint16_t sslVersion) NN_NOEXCEPT;

public:
    explicit SslConnectionImpl(SslContextImpl *parent) NN_NOEXCEPT;
    ~SslConnectionImpl() NN_NOEXCEPT;

    SslContextImpl *GetParentSslContext();
    NssUtil::ErrorList *GetErrorList();
    Connection::VerifyOption GetVerifyOptionMask();

    nn::Result SaveServerCertChain(CERTCertificate *pCert);
    nn::Result SaveServerCertChain(CERTCertList *pCertList);
    bool IsServerChainNeeded();

    //  SF object interface APIs
    nn::Result SetSocketDescriptor(int32_t socket,
                                   Out<int32_t> socketToClose) NN_NOEXCEPT;
    nn::Result SetHostName(const InBuffer& hostName) NN_NOEXCEPT;
    nn::Result SetVerifyOption(VerifyOption optionValue) NN_NOEXCEPT;
    nn::Result SetIoMode(IoMode mode) NN_NOEXCEPT;
    nn::Result SetSessionCacheMode(SessionCacheMode mode) NN_NOEXCEPT;
    nn::Result SetRenegotiationMode(RenegotiationMode mode) NN_NOEXCEPT;
    nn::Result GetSocketDescriptor(Out<int32_t> outValue) NN_NOEXCEPT;
    nn::Result GetHostName(const OutBuffer& hostNameBuffer,
                           Out<uint32_t> outHostNameLength) NN_NOEXCEPT;
    nn::Result GetVerifyOption(Out<VerifyOption> outVerifyOption) NN_NOEXCEPT;
    nn::Result GetIoMode(Out<IoMode> outIoMode) NN_NOEXCEPT;
    nn::Result GetSessionCacheMode(Out<SessionCacheMode> outSessionCachemode) NN_NOEXCEPT;
    nn::Result GetRenegotiationMode(Out<RenegotiationMode> outRenegotiationMode) NN_NOEXCEPT;
    nn::Result FlushSessionCache() NN_NOEXCEPT;
    nn::Result DoHandshake();
    nn::Result DoHandshakeGetServerCert(Out<uint32_t> outServerCertSize,
                                        Out<uint32_t> outServerCertCount,
                                        const OutBuffer& outServerCert) NN_NOEXCEPT;
    nn::Result Read(Out<uint32_t> bytesRead,
                    const OutBuffer& buffer) NN_NOEXCEPT;
    nn::Result Write(Out<uint32_t> bytesWritten,
                     const InBuffer& buffer) NN_NOEXCEPT;
    nn::Result Pending(Out<int32_t> byteCount) NN_NOEXCEPT;
    nn::Result Peek(Out<uint32_t> byteCount,
                    const OutBuffer& buffer) NN_NOEXCEPT;
    nn::Result Poll(Out<PollEvent> outEvent,
                    PollEvent inEvent,
                    uint32_t msecTimeout) NN_NOEXCEPT;
    nn::Result GetVerifyCertError() NN_NOEXCEPT;
    nn::Result GetVerifyCertErrors(const OutBuffer& outResultArray, Out<uint32_t> outResultCountWritten, Out<uint32_t> outTotalResultCount) NN_NOEXCEPT;
    nn::Result GetNeededServerCertBufferSize(Out<uint32_t> outNeededSize) NN_NOEXCEPT;

    nn::Result SetOption(OptionType optionType, bool enable) NN_NOEXCEPT;
    nn::Result GetOption(Out<bool> outIsEnabled, OptionType optionType) NN_NOEXCEPT;
    nn::Result GetCipherInfo(uint32_t structureVersion, OutBuffer outCipherInfo) NN_NOEXCEPT;
};

} } }
