﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/TargetConfigs/build_Platform.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nn/nsd/nsd_ApiForMiddleware.h>

#include <nn/ssl/ssl_Context.h>
#include <nn/ssl/ssl_Context.private.h>

#include "detail/ssl_CertChain.h"
#include "server/ssl_NssCommon.h"
#include "server/ssl_NssCore.h"
#include "server/ssl_NssPkcsUtil.h"
#include "server/ssl_NssUtil.h"
#include "server/ssl_ServiceDatabase.h"
#include "server/ssl_SslConnectionImpl.h"
#include "server/ssl_SslSession.h"
#include "server/ssl_SslTrustedCertManager.h"
#include "server/ssl_Util.h"
#include "detail/ssl_ISslServiceFactory.h"

#include <mutex> // std::lock_guard

using namespace nn::sf;
using namespace nn::ssl::sf;

namespace nn { namespace ssl { namespace detail {

//  Private SSL callbacks
SECStatus SslConnectionImpl::SslHandshakeCb(PRFileDesc *socket, void *arg)
{
    //  UNUSED on Siglo
    NN_UNUSED(socket);
    NN_UNUSED(arg);

    NN_DETAIL_SSL_DBG_PRINT("[SslHandshakeCb] entered, not used, fail\n");
    return SECFailure;
}


SECStatus SslConnectionImpl::SslAuthCertCb(void       *arg,
                                           PRFileDesc *socket,
                                           PRBool     checkSig,
                                           PRBool     isServer)
{
    SECStatus                   ret = SECSuccess;
    SslConnectionImpl           *conn = static_cast<SslConnectionImpl *>(arg);
    char                        *hostname = SSL_RevealURL(socket);
    NssUtil::CertTool           certVerifyHelper;
    PRErrorCode                 verifyError;
    nn::Result                  result;
    CERTCertificate             *serverCert = nullptr;

    conn->m_authCertCount = 0;
    conn->m_authCertSize  = 0;
    conn->m_errorList.ClearErrors();

    NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] hostname:%s checkSize:%d isServer:%d\n",
                            (hostname != nullptr) ? hostname : "<NULL>",
                            checkSig,
                            isServer);
    do
    {
        serverCert = SSL_PeerCertificate(socket);
        if (serverCert == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] ERROR: server cert not found\n");
            ret = SECFailure;
            break;
        }

        //  If the caller has not requested the entire chain, verify that we
        //  have enough room to stash off a copy and if so, do it.
        if (!conn->m_GetServerCertChain && (conn->m_AuthCertMaxSize > 0))
        {

            //  As long as there is room in the buffer provided, let's give it to them.
            if (conn->m_AuthCertMaxSize < serverCert->derCert.len)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] caller buf not big enough, got %u, need %u\n",
                                        conn->m_AuthCertMaxSize,
                                        serverCert->derCert.len);

                //  Not a fatal error; if validation fails below it will
                //  override this error (it is more important!)
                conn->m_AuthCertError = ResultInsufficientServerCertBuffer();
                conn->m_AuthCertSizeNeeded = serverCert->derCert.len;
            }
            else
            {
                memcpy(conn->m_pAuthCertData,
                       serverCert->derCert.data,
                       serverCert->derCert.len);

                //  We always copy only a single cert here.  If the full chain
                //  is needed, it is done after verify.
                conn->m_authCertCount = 1;

                NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] copied back %u bytes into %p\n",
                                        serverCert->derCert.len,
                                        conn->m_pAuthCertData);
            }

            //  Regardless of whether we could copy or not, copy back the size
            //  of the so the caller knows.  If the buffer was not big enough then
            //  they know what size is needed.
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] setting size to %d\n",
                                    serverCert->derCert.len);
            conn->m_authCertSize = serverCert->derCert.len;
        }

#if defined(NN_DETAIL_SSL_DBG_PRINT_PEER_CERT_INFO)
        certVerifyHelper.DumpCertificateInformation(serverCert);
#endif //NN_DETAIL_SSL_DBG_PRINT_PEER_CERT_INFO

#ifdef NN_DETAIL_SSL_DBG_PRINT_PEER_CERT
        NssUtil::DumpPublicKeyInCert(serverCert);
#endif // NN_DETAIL_SSL_DBG_PRINT_PEER_CERT

#ifdef NN_DETAIL_SSL_DBG_PRINT_PEER_PUBLIC_KEY
        NssUtil::DumpDerCertificate(serverCert);
#endif // NN_DETAIL_SSL_DBG_PRINT_PEER_PUBLIC_KEY

        // ********************************************************************************************
        // Validation
        // ********************************************************************************************
        // TODO: Support OCSP
        // TODO: It might be requried to report multiple reasons of validation failure rather than
        //       stop validation when it fails
        // TODO: Not to allow skipping certificate validation in production code

        // Just return SECSuccess if validation is not required (should be available only for debugging)
        if ((conn->m_VerifyOptionMask & Connection::VerifyOption::VerifyOption_PeerCa)
            != Connection::VerifyOption::VerifyOption_PeerCa)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] Certificate validation is not performed.\n");
            break;
        }

        // 1- Verify certificate **********************************************************************
        // Date will be verified only when it is specified to do so in verifyOption.
        PRInt64                 dateToCheck = -1;    //  Skip by default

        if ((conn->m_VerifyOptionMask & Connection::VerifyOption::VerifyOption_DateCheck)
            == Connection::VerifyOption::VerifyOption_DateCheck)
        {
            dateToCheck = 0;    //  Equivalent of PR_Now()
        }

        NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] serverCert subject key ID:\n");
        NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(serverCert->subjectKeyID.data,
                                          serverCert->subjectKeyID.len);
        result = certVerifyHelper.VerifyCertificate(serverCert, dateToCheck, conn);

        //  It is possible for the full server cert chain to not fit in the
        //  provided buffer and that info returned here.  If so, treat this
        //  as successful (so far) and the upper layers will get the details.
        if (ResultInsufficientServerCertBuffer::Includes(result))
        {
            conn->m_AuthCertError = result;
            result = ResultSuccess();
        }

        if (result.IsFailure())
        {
            conn->m_AuthCertError = result;

            NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] ERROR: Certificate validation failed.\n");
            ret = SECFailure;

            if( ResultInsufficientMemory::Includes(result) || ResultInvalidReference::Includes(result) )
            {
                // The error list cannot be reliable if there was a memory error,
                //    also an ResultInvalidReference error should just fail the operation.
                break;
            }
        }

        // 2- Verify host name ************************************************************************
        //     Need to be performed manucally because default authCertcallback which is
        //     SSL_AuthCertificate is not called when this callback is used
        if ((conn->m_VerifyOptionMask & Connection::VerifyOption::VerifyOption_HostName)
            == Connection::VerifyOption::VerifyOption_HostName)
        {
            result = certVerifyHelper.VerifyName(serverCert,
                                                 socket,
                                                 nullptr,
                                                 &verifyError);
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] ERROR: Host name validation failed (%d - %s).\n",
                                        verifyError,
                                        PR_ErrorToName(verifyError));
                PORT_SetError(verifyError);

                if( !conn->m_errorList.AddError(verifyError) )
                {
                    conn->m_AuthCertError = ResultInsufficientMemory();
                }
                else
                {
                    conn->m_AuthCertError = ResultInternalCertVerifyFailed();
                }

                ret = SECFailure;
            }
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] REPORT: Host name was not verified.\n");
        }

        //  If we get here, we're good
    } while (NN_STATIC_CONDITION(false));

    // ********************************************************************************************
    // Cleanup
    // ********************************************************************************************
    // TODO: Free server certificate in db here
    // SSL makes and keeps internal copies (or increments the reference counts, as appropriate)
    // of certificate and key structures. The application should destroy its copies when it has
    // no further use for them by calling CERT_DestroyCertificate and SECKEY_DestroyPrivateKey.
    if (serverCert != nullptr)
    {
        CERT_DestroyCertificate(serverCert);
    }

    if (hostname != nullptr)
    {
        PL_strfree(hostname);
        hostname = nullptr;
    }

    if(ret != SECSuccess)
    {
        // If the error list is empty, the PR error will be set to SECSuccess.
        //  NSS sends a 'Bad cert' alert, when this function returns an error and SECSuccess is set.
        PRErrorCode prError = SECSuccess;

        const auto* pErrorNode = conn->m_errorList.GetHead();
        while( pErrorNode )
        {
            if( ConvertErrorToPriorityValue(pErrorNode->error) > ConvertErrorToPriorityValue(prError) )
            {
                prError = pErrorNode->error;
            }

            pErrorNode = pErrorNode->pNext;
        }

        PORT_SetError(prError);
    }

    NN_DETAIL_SSL_DBG_PRINT("[SslAuthCertCb] returning %d\n", ret);
    return ret;
}    //  NOLINT(impl/function_size);



SECStatus SslConnectionImpl::SslBadCertCb(void *arg, PRFileDesc *socket)
{
    SECStatus                   ret = SECFailure;
    SslConnectionImpl           *conn = static_cast<SslConnectionImpl *>(arg);
    PRErrorCode                 prError = SECSuccess;

    NN_DETAIL_SSL_DBG_PRINT("[SslBadCertCb] entered\n");

    do
    {
        if (conn == nullptr)
        {
            //  Should not be possible, just fail out
            NN_SDK_ASSERT(NN_STATIC_CONDITION(false));
            NN_DETAIL_SSL_DBG_PRINT("[SslBadCertCb] ERROR: no connection obj!\n");
            break;
        }

        if( ResultInsufficientMemory::Includes(conn->m_AuthCertError) || ResultInvalidReference::Includes(conn->m_AuthCertError) )
        {
            // conn->m_errorList is not reliable if there was a memory error.
            // Also, ResultInvalidReference should just fail the operation.
            break;
        }

        const auto* pErrorNode = conn->m_errorList.GetHead();
        while( pErrorNode )
        {
            if( ConvertErrorToPriorityValue(pErrorNode->error) > ConvertErrorToPriorityValue(prError) )
            {
                prError = pErrorNode->error;
            }

            pErrorNode = pErrorNode->pNext;
        }

        //  Just to be on the safe side, check the NSPR error as well as any pending
        //  auth cert error.  If NSPR status is success or auth cert is anything but
        //  an internal verify cert failure, allow a success.
        if ((prError == SECSuccess) ||
            !ResultInternalCertVerifyFailed::Includes(conn->m_AuthCertError))
        {
            //  This should not be possible, but allow success.
            NN_DETAIL_SSL_DBG_PRINT("[SslBadCertCb] WARN: pr err %d, cert error %d:%d\n",
                                    prError,
                                    conn->m_AuthCertError.GetModule(),
                                    conn->m_AuthCertError.GetDescription());

            NN_SDK_ASSERT(NN_STATIC_CONDITION(false));
            ret = SECSuccess;
            break;
        }

        // *******************************************************************
        // The code reaches here only when certificate validation fails
        // *******************************************************************
        nn::Result              result = NssUtil::ConvertNssErrorToResult(prError);

        NN_DETAIL_SSL_DBG_PRINT("[SslBadCertCb] NSS error - %d (%s) -> Result (desc:%d)\n",
                                prError,
                                PR_ErrorToName(prError),
                                result.GetDescription());

        // TODO: should we adjust error reason based on SSL version?
        conn->m_AuthCertError = result;
    } while (NN_STATIC_CONDITION(false));

    // In SslConnectionImpl::SslAuthCertCb we manually set the PR error code.
    //  So here we reset it to success.
    PORT_SetError(SECSuccess);

    NN_DETAIL_SSL_DBG_PRINT("[SslBadCertCb] returning %d\n", ret);
    return ret;
}


SECStatus SslConnectionImpl::SslAuthClientCb(void              *pArg,
                                             PRFileDesc        *pSocket,
                                             CERTDistNames     *pCaNames,
                                             CERTCertificate   **pOutCert,
                                             SECKEYPrivateKey  **pOutKey)
{
    SECStatus                   ret = SECFailure;
    SslConnectionImpl           *pConn = static_cast<SslConnectionImpl *>(pArg);
    CertStore                   *pCertStore = pConn->GetParentSslContext()->GetCertStore();
    nn::Result                  errResult = ResultSslErrorClientCertificateNotFound();
    CERTCertificate             *pCert = nullptr;
    SECKEYPrivateKey            *pKey = nullptr;

    NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] entered\n");

    do
    {
        ClientCertEntry         *pClientCert = pCertStore->GetClientCert();

        if (pClientCert == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] no client cert entry found\n");
            break;
        }

        //  We have a winner!  Get the cert and key from the entry and get out.
        pCert = CERT_DupCertificate(pClientCert->GetCert());
        if (pCert == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] unable to dup client cert\n");
            errResult = ResultInsufficientMemory();
            break;
        }

        pKey  = SECKEY_CopyPrivateKey(pClientCert->GetKey());
        if (pKey == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] unable to dup client key\n");
            errResult = ResultInsufficientMemory();
            break;
        }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        char     *pNicknameStr = CertStore::ExtractNickname(pCert);
        if (pNicknameStr == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] WARNING: no nickname found for cert\n");
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] client cert imported, use it (%p, \'%s\')\n",
                                    pClientCert,
                                    pNicknameStr);

            PR_Free(pNicknameStr);
            pNicknameStr = nullptr;
        }
#else
        NN_DETAIL_SSL_DBG_PRINT("[SslAuthClientCb] client cert imported, use it (%p)\n",
                                pClientCert);
#endif    //  NN_SDK_BUILD_DEBUG || NN_SDK_BUILD_DEVELOP

        *pOutCert = pCert;
        *pOutKey = pKey;
        ret = SECSuccess;
    } while (NN_STATIC_CONDITION(false));

    if (ret == SECFailure)
    {
        pConn->m_AuthCertError = errResult;

        if (pCert != nullptr)
        {
            CERT_DestroyCertificate(pCert);
            pCert = nullptr;
        }

        if (pKey != nullptr)
        {
            SECKEY_DestroyPrivateKey(pKey);
            pKey = nullptr;
        }
    }

    return ret;
}


nn::Result SslConnectionImpl::SetNssSslSocketOptions(nn::ssl::Context::SslVersion version) NN_NOEXCEPT
{
    nn::Result                  ret = ResultInternalNssFailedToSetSslVersion();
    SECStatus                   status;

    do
    {
        //  Setup some basic options for enabling security and client SSL
        if (SSL_OptionSet(m_SslSocket, SSL_SECURITY, PR_TRUE) != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed enable security");
            break;
        }

        if (SSL_OptionSet(m_SslSocket, SSL_HANDSHAKE_AS_SERVER, PR_FALSE) != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to mark server as FALSE");
            break;
        }

        if (SSL_OptionSet(m_SslSocket, SSL_HANDSHAKE_AS_CLIENT, PR_TRUE) != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to mark client as TRUE");
            break;
        }

        //  Set options basic on the provided SSL version
        if (version & Context::SslVersion_Auto)
        {
            status = SSL_OptionSet(m_SslSocket, SSL_ENABLE_SSL2, PR_FALSE);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to disable SSL2 (SslVersion_Auto)\n");
                break;
            }

            status = SSL_OptionSet(m_SslSocket, SSL_ENABLE_SSL3, PR_FALSE);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to enable SSL3 (SslVersion_Auto)\n");
                break;
            }

            status = SSL_OptionSet(m_SslSocket, SSL_ENABLE_TLS, PR_TRUE);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed enable TLS (SslVersion_Auto)\n");
                break;
            }
        }
        else
        {
            status = SSL_OptionSet(m_SslSocket,
                                   SSL_ENABLE_SSL3,
                                   (version & ContextPrivate::SslVersion_SslV3) ? PR_TRUE : PR_FALSE);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to set SSL3 to %s\n",
                                        (version & ContextPrivate::SslVersion_SslV3) ? "TRUE" : "FALSE");
                break;
            }
            else if((status == SECSuccess) && (version & ContextPrivate::SslVersion_SslV3))
            {
                NN_DETAIL_SSL_INFO_PRINT("Enabled SSLv3.\n");
            }

            if (version &
                (Context::SslVersion_TlsV10 |
                 Context::SslVersion_TlsV11 |
                 Context::SslVersion_TlsV12))
            {
                status = SSL_OptionSet(m_SslSocket,
                                       SSL_ENABLE_TLS,
                                       PR_TRUE);
                if (status != SECSuccess)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to set TLS to TRUE\n");
                    break;
                }
            }
            else
            {
                status = SSL_OptionSet(m_SslSocket,
                                       SSL_ENABLE_TLS,
                                       PR_FALSE);
                if (status != SECSuccess)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[SetNssSslSocketOptions] failed to set TLS to FALSE\n");
                    break;
                }
            }
        }

        //  If we get here, change return value to success!
        ret = ResultSuccess();
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


PRIntervalTime SslConnectionImpl::GetIoTimeout() NN_NOEXCEPT
{
    PRIntervalTime              timeout;

    //  Determine the timeout based on the IO mode which was picked
    std::lock_guard<decltype(m_ioModeLock)> lock(m_ioModeLock);
    switch (m_IoMode)
    {
        case Connection::IoMode::IoMode_NonBlocking:
            timeout = 0;
            break;

        case Connection::IoMode::IoMode_Blocking:
        default:
            timeout = g_DefaultIoMsecTimeout;
            break;
    }

    return PR_MillisecondsToInterval(timeout);
}


SslConnectionImpl::SslConnectionImpl(SslContextImpl *parent) NN_NOEXCEPT
    : m_Parent(parent, true),
      m_SslSocket(nullptr),
      m_Socket(-1),
      m_VerifyOptionMask(Connection::VerifyOption::VerifyOption_Default),
      m_IoMode(Connection::IoMode::IoMode_Blocking),
      m_pAuthCertData(nullptr),
      m_AuthCertMaxSize(0),
      m_AuthCertSizeNeeded(0),
      m_authCertSize(0),
      m_authCertCount(0),
      m_ioModeLock(false),
      m_IsDoNotCloseSocketInShim(false),
      m_GetServerCertChain(false),
      m_IsSkipDefaultValidation(false)
{
    memset(m_HostName, 0, sizeof(m_HostName));
    m_AuthCertError = nn::ResultSuccess();
    m_LastAuthCertError = nn::ResultSuccess();
}


SslConnectionImpl::~SslConnectionImpl() NN_NOEXCEPT
{
    if (m_SslSocket != nullptr)
    {
#if defined(NN_DETAIL_SSL_BUILD_FORCE_SESSION_CACHE_FLUSH_UPON_CONN_CLOSE)
        SslSessionManager::DeleteSessionCache(m_SslSocket);
#endif // NN_DETAIL_SSL_BUILD_FORCE_SESSION_CACHE_FLUSH_UPON_CONN_CLOSE
        PR_Close(m_SslSocket);
        m_SslSocket = nullptr;
        m_Socket = -1;

        nnsdkNssPortCheckSessionCacheMemory();
    }
}


SslContextImpl *SslConnectionImpl::GetParentSslContext()
{
    return m_Parent.Get();
}


NssUtil::ErrorList *SslConnectionImpl::GetErrorList()
{
    return &m_errorList;
}


Connection::VerifyOption SslConnectionImpl::GetVerifyOptionMask()
{
    return m_VerifyOptionMask;
}


nn::Result SslConnectionImpl::PrepServerCertChainBuf()
{
    nn::Result                  result = ResultSuccess();
    uint32_t                    baseSize;
    CertChainInfo               *pChainInfo;

    do
    {
        //  Make sure we have a buffer and it is large enough to hold at least
        //  the cert details header.  Not an exact check (will be done later
        //  when we have cert data.)
        baseSize = GetCertDataOffsetForChain(1);
        if ((m_pAuthCertData == nullptr) || (m_AuthCertMaxSize < baseSize))
        {
            NN_DETAIL_SSL_DBG_PRINT("[PrepServerCertChainBuf] buffer not large enough: %p (%u, need %u)\n",
                                    m_pAuthCertData,
                                    m_AuthCertMaxSize,
                                    baseSize);

            //  Do not set size needed, just bail out as the handshake will
            //  continue until we get real size info.
            result = ResultInsufficientServerCertBuffer();
            break;
        }

        pChainInfo = reinterpret_cast<CertChainInfo *>(m_pAuthCertData);
        pChainInfo->magicNum = nn::ssl::detail::CertChainMagicNum;
        pChainInfo->certCount = 0;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


uint32_t SslConnectionImpl::GetCertSizeForChain(CERTCertificate *pCert)
{
    uint32_t                    certSize;
    uint32_t                    alignedSize;

    certSize = static_cast<uint32_t>(pCert->derCert.len);
    alignedSize = NN_DETAIL_SSL_CALC_ALIGN(certSize, sizeof(uint32_t));
    return alignedSize;
}


uint32_t SslConnectionImpl::GetCertDataOffsetForChain(uint32_t certCount)
{
    uint32_t                    offset;

    offset = sizeof(CertChainInfo) + (certCount * sizeof(CertChainEntry));
    return offset;
}


uint32_t SslConnectionImpl::AddCertToChainBuf(CERTCertificate *pCert, uint32_t curOffset)
{
    CertChainInfo               *pChainInfo;
    CertChainEntry              *pChainEntries;

    pChainInfo = reinterpret_cast<CertChainInfo *>(m_pAuthCertData);
    pChainEntries = reinterpret_cast<CertChainEntry *>(pChainInfo + 1);

    pChainEntries[pChainInfo->certCount].len =
        static_cast<uint32_t>(pCert->derCert.len);
    pChainEntries[pChainInfo->certCount].offset = curOffset;

    memcpy(m_pAuthCertData + curOffset,
           pCert->derCert.data,
           pCert->derCert.len);

    pChainInfo->certCount++;
    curOffset += GetCertSizeForChain(pCert);
    return curOffset;
}


nn::Result SslConnectionImpl::SaveServerCertChain(CERTCertificate *pCert)
{
    nn::Result                  result = ResultSuccess();
    uint32_t                    curOffset;
    uint32_t                    totalSize = 0;

    do
    {
        //  If the full chain has not been requested, just bail out
        if (!m_GetServerCertChain)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] (server only) chain not requested, skip\n");
            break;
        }

        //  Prep the chain buffer.  If it fails, keep going through as we
        //  will get more size info later when trying to fill the buffer.
        result = PrepServerCertChainBuf();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] failed to prep cert chain buf, auth will fail later\n");
        }

        //  Determine the total size need.  This is going to be the base
        //  chain info along with the specified certificate data.
        curOffset = GetCertDataOffsetForChain(1);
        totalSize = curOffset + GetCertSizeForChain(pCert);

        //  Save off the size and count of certs in the chain, regardless
        //  of whether we have enough space or not.  This way the caller
        //  knows what is needed.
        m_authCertSize  = totalSize;
        m_authCertCount = 1;

        //  Make sure the provided buffer is large enough
        if ((m_pAuthCertData == nullptr) || (m_AuthCertMaxSize < totalSize))
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] (server only) buffer not large enough: %p (%u, need %u)\n",
                                    m_pAuthCertData,
                                    m_AuthCertMaxSize,
                                    totalSize);
            m_AuthCertSizeNeeded = totalSize;
            result = ResultInsufficientServerCertBuffer();
            break;
        }

        AddCertToChainBuf(pCert, curOffset);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result SslConnectionImpl::SaveServerCertChain(CERTCertList *pCertList)
{
    nn::Result                  result = ResultSuccess();
    uint32_t                    certCount = 0;
    CERTCertListNode            *pCurCertNode;
    uint32_t                    curOffset;
    uint32_t                    totalSize = 0;

    do
    {
        //  If the full chain has not been requested, just bail out
        if (!m_GetServerCertChain)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] chain not requested, skip\n");
            break;
        }

        //  Prep the chain buffer.  If it fails, keep going through as we
        //  will get more size info later when trying to fill the buffer.
        result = PrepServerCertChainBuf();
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] failed to prep cert chain buf, auth will fail later\n");
        }

        //  Unfortunately, we have to do this in two passes because we need
        //  to know how big of a header table to have in the buffer then
        //  follow that table up with the raw certs (aligned).
        for (pCurCertNode = CERT_LIST_HEAD(pCertList);
             !CERT_LIST_END(pCurCertNode, pCertList);
             pCurCertNode = CERT_LIST_NEXT(pCurCertNode))
        {
            certCount++;
            totalSize += GetCertSizeForChain(pCurCertNode->cert);
        }

        //  The current offset is PAST the header info and the chain entries
        //  array.
        curOffset = GetCertDataOffsetForChain(certCount);
        totalSize += curOffset;

        //  Save off the size and count of certs in the chain, regardless
        //  of whether we have enough space or not.  This way the caller
        //  knows what is needed.
        m_authCertSize  = totalSize;
        m_authCertCount = certCount;

        //  Make sure the provided buffer is large enough
        if ((m_pAuthCertData == nullptr) || (m_AuthCertMaxSize < totalSize))
        {
            NN_DETAIL_SSL_DBG_PRINT("[SaveServerCertChain] buffer not large enough: %p (%u, need %u)\n",
                                    m_pAuthCertData,
                                    m_AuthCertMaxSize,
                                    totalSize);
            m_AuthCertSizeNeeded = totalSize;
            result = ResultInsufficientServerCertBuffer();
            break;
        }

        //  Re-walk the list and populate each entry
        for (pCurCertNode = CERT_LIST_HEAD(pCertList);
             !CERT_LIST_END(pCurCertNode, pCertList);
             pCurCertNode = CERT_LIST_NEXT(pCurCertNode))
        {
            curOffset = AddCertToChainBuf(pCurCertNode->cert, curOffset);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


bool SslConnectionImpl::IsServerChainNeeded()
{
    return m_GetServerCertChain;
}


nn::Result SslConnectionImpl::SetSocketDescriptor(int32_t      socket,
                                                  Out<int32_t> outSocketToClose) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    PRFileDesc                  *nsprSocket = nullptr;
    SECStatus                   status;
    int32_t                     sharedSocket = -1;
    int32_t                     socketToClose = -1;
    uint64_t                    pid = this->GetParentSslContext()->GetParentSslService()->GetClientProcessId();

    NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] enter\n");

    do
    {
        if (m_SslSocket != nullptr)
        {
            ret = ResultSocketAlreadyRegistered();
            break;
        }

        if (pid != 0)
        {
            //  We need to "share" the caller's socket since it is coming
            //  from a different process.
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] pid %ull calling, dup socket %d\n",
                                    pid,
                                    socket);
            sharedSocket = nn::socket::DuplicateSocket(socket, pid);
            if (sharedSocket < 0)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to share socket %d from %llu\n",
                                        socket,
                                        pid);
                ret = ResultInvalidSocketDescriptor();
                break;
            }

            if (m_IsDoNotCloseSocketInShim == false)
            {
                socketToClose = socket;
            }
        }
        else
        {
            //  Use a passed socket as a shared socket because the client is in the same process
            sharedSocket = socket;
        }

        // Check if TCP connection is established on the passed socket
        nn::socket::SockAddr    socketAddress;
        nn::socket::SockLenT    socketAddressLength = sizeof(socketAddress);
        int rval = nn::socket::GetPeerName(sharedSocket, &socketAddress, &socketAddressLength);
        if(rval < 0)
        {
            nn::socket::Errno lastError = nn::socket::GetLastError();
            if (lastError == nn::socket::Errno::ENotConn)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] Socket is not connected (socket:%d)\n", socket);
                ret = ResultNoTcpConnection();
            }
            else if (lastError == nn::socket::Errno::ENotSock)
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] Passed descriptor (%d) is not socket.\n", socket);
                ret = ResultInvalidSocketDescriptor();
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] GetPeerName failed (errno:%d)\n", lastError);
                ret = ResultErrorLower();
            }
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] importing socket %d\n", sharedSocket);

        nsprSocket = PR_ImportTCPSocket(sharedSocket);
        if (nsprSocket == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to convert socket to NSPR socket\n");
            ret = NssUtil::ConvertNssErrorToResult(PR_GetError());
            break;
        }

        m_SslSocket = SSL_ImportFD(nullptr, nsprSocket);
        if (m_SslSocket == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to convert NSPR socket to SSL socket.\n");
            ret = NssUtil::ConvertNssErrorToResult(PR_GetError());
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] ssl socket %X (%d)\n",
                                m_SslSocket,
                                sharedSocket);
        m_Socket = socket;

        PRSocketOptionData socketOption;
        socketOption.option = PR_SockOpt_Nonblocking;

        // Scope is for mutex AutoLock
        {
            std::lock_guard<decltype(m_ioModeLock)> lock(m_ioModeLock);
            PRStatus prRet = PR_GetSocketOption(m_SslSocket, &socketOption);
            if(prRet == PR_SUCCESS)
            {
                if(socketOption.value.non_blocking != 0)
                {
                    m_IoMode = Connection::IoMode::IoMode_NonBlocking;
                }
                else
                {
                    m_IoMode = Connection::IoMode::IoMode_Blocking;
                }
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] Error: PR_GetSocketOption failed. PR Error: %d\n", PR_GetError());
                ret = ResultErrorLower();
                break;
            }
        }

        SslContextImpl *parent = GetParentSslContext();
        ret = SetNssSslSocketOptions(parent->GetSslVersion());
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set NSS SSL options.\n");
            ret = NssUtil::ConvertNssErrorToResult(PR_GetError());
            break;
        }

        //  Register the various SSL callbacks to handle handshake, cert verification,
        //  etc.  Within NSS these are mapped to the specific SSL socket, so go ahead
        //  and perform this as we are setting the SSL socket in our connection.
        status = SSL_AuthCertificateHook(m_SslSocket,
                                         SslAuthCertCb,
                                         this);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set auth cert cb: %d\n", status);
            ret = ResultInternalNssFailedToRegisterCallback();
            break;
        }

        status = SSL_BadCertHook(m_SslSocket, &SslBadCertCb, this);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set bad cert cb: %d\n", status);
            ret = ResultInternalNssFailedToRegisterCallback();
            break;
        }

        status = SSL_GetClientAuthDataHook(m_SslSocket, &SslAuthClientCb, this);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set auth client cb: %d\n", status);
            ret = ResultInternalNssFailedToRegisterCallback();
            break;
        }

        //  Reset the SSL handshake logic for CLIENT mode.
        if (SSL_ResetHandshake(m_SslSocket, PR_FALSE) != SECSuccess)
        {
            // This should not happen
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to reset handshake.\n");
            ret = ResultErrorLower();
            break;
        }

        ret = SetPeerId();
        if(ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set peer id.\n");
            break;
        }

        //  If we are on Horizon, Shared Sockets are in use and the client side
        //  will need to close the originating socket when this connection is
        //  destroyed.  Provide it back to the client side caller.  If this is
        //  not Horizon, then the default -1 tells the caller to do nothing:
        //  the SslConnection implementation is the only close needed.
        *outSocketToClose = socketToClose;
    } while (NN_STATIC_CONDITION(false));

    if (ret.IsFailure() && !nn::ssl::ResultSocketAlreadyRegistered::Includes(ret))
    {
        if (m_SslSocket != nullptr)
        {
            PR_Close(m_SslSocket);
            m_SslSocket = nullptr;
        }
        else if (nsprSocket != nullptr)
        {
            /*  We ONLY close the "raw" NSPR socket if the SSL socket was not
                successfully created (per NSS docs.)  This is because the SSL
                socket is a new "layer" file descriptor on top of the base
                NSPR descriptor, so when we close the SSL socket above, it
                will also close the underlying NSPR socket for us.  */
            PR_Close(nsprSocket);
            nsprSocket = nullptr;
        }
        else if (sharedSocket >= 0 && pid != 0)
        {
            //  Socket was duplicated but not imported into NSPR.
            //  This is needed only on cross-processing environment.
            nn::socket::Close(sharedSocket);
        }

        m_Socket = -1;
    }

    NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
} // NOLINT(impl/function_size)


nn::Result SslConnectionImpl::SetHostName(const InBuffer& hostName) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[SetHostName] enter\n");

    do
    {
        //  Bounds check the incoming buffer
        size_t nameSize = hostName.GetSize();
        const char *name = hostName.GetPointerUnsafe();

        if (nameSize >= g_MaxHostNameLength)
        {
            ret = ResultHostNameTooLong();
            break;
        }

#ifdef NN_DETAIL_SSL_BUILD_ENABLE_NINTENDO_SERVICE_DISCOVERY
        nn::nsd::Fqdn fqdnTry;
        nn::nsd::Fqdn fqdnResolved;

        memset(&fqdnTry, 0x00, sizeof(fqdnTry));
        memset(&fqdnResolved, 0x00, sizeof(fqdnResolved));

        // nameSize <= 255, fqdnTry.Size == 256 so the name pointer could be smaller than the
        // max size of Fqdn data
        PL_strncpy(fqdnTry.value, name, static_cast<PRUint32>(nameSize));
        nn::nsd::ResolveEx(&fqdnResolved, fqdnTry);

        // Copy the resolved fqdn data into our local copy
        // m_HostName is defined as "char m_HostName[MaxHostNameLength + 1]" which length is 256
        // so it's ok just to copy Fqdn.Size which max size is 256.
        PL_strncpy(m_HostName, fqdnResolved.value, static_cast<PRUint32>(fqdnResolved.Size));

        // Update nameSize properly (allow MaxHostNameLength for max to have a space for
        // null terminated string)
        nameSize = PL_strnlen(m_HostName, MaxHostNameLength);
#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT
        char tmpStr[MaxHostNameLength + 1] = {0};
        PL_strncpy(tmpStr, name, (PRUint32)nameSize);
        NN_DETAIL_SSL_DBG_PRINT("[SetHostName] Set resolved name by NSD (%s -> %s) (len:%d)\n",
            tmpStr, m_HostName, nameSize);
#endif
#else
        //  Copy the hostname data into our local copy
        PL_strncpy(m_HostName, name, static_cast<PRUint32>(nameSize));
#endif // NN_DETAIL_SSL_BUILD_ENABLE_NINTENDO_SERVICE_DISCOVERY

        //  Be sure to null terminate the buffer
        m_HostName[nameSize] = 0;

        NN_DETAIL_SSL_DBG_PRINT("[SetHostName] %p host name: %s\n", this, m_HostName);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[SetHostName] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}

nn::Result SslConnectionImpl::SetPeerId() NN_NOEXCEPT
{
    nn::Result ret = ResultSuccess();

    if (m_SslSocket != nullptr)
    {
        char pTmpStr[NN_SSL_NSS_PORT_SESSION_CACHE_PEER_ID_BUF_LEN] = {0};

        nnsdkNssPortSetupSessionCachePeerId(
            pTmpStr,
            static_cast<PRUint64>(GetParentSslContext()->GetParentSslService()->GetClientProcessId()),
            static_cast<nnsslConnectionVerifyOption>(m_VerifyOptionMask));

        NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor]Set peer ID (%s)\n", pTmpStr);
        SECStatus status = SSL_SetSockPeerID(m_SslSocket, pTmpStr);

        if (status != SECSuccess)
        {
            // Because we chek to make sure m_SslSocket is valid,
            // this should only happen when there is a failed memory allocation.
            NN_DETAIL_SSL_DBG_PRINT("[SetSocketDescriptor] failed to set peer ID.\n");
            ret = ResultInsufficientMemory();
        }
    }

    return ret;
}

nn::Result SslConnectionImpl::SetVerifyOption(VerifyOption optionValue) NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[SetVerifyOption] enter\n");
    nn::Result result = ResultSuccess();

    do
    {
        Connection::VerifyOption verifyOptionMask =
            static_cast<Connection::VerifyOption>(optionValue.options);

        // Make sure only valid options taken.
        verifyOptionMask &= (Connection::VerifyOption::VerifyOption_PeerCa    |
                             Connection::VerifyOption::VerifyOption_HostName  |
                             Connection::VerifyOption::VerifyOption_DateCheck |
                             Connection::VerifyOption::VerifyOption_EvCertPartial );

        if (this->GetParentSslContext()->GetParentSslService()->GetInterfaceVersion()
            >= ISslServiceFactory::ISslServiceVersionType_Nup500)
        {
            if (m_IsSkipDefaultValidation == false &&
                (verifyOptionMask & Connection::VerifyOption::VerifyOption_Default) != Connection::VerifyOption::VerifyOption_Default)
            {
                result = nn::ssl::ResultOperationNotAllowed();
                break;
            }
        }

        m_VerifyOptionMask = verifyOptionMask;
        result = SetPeerId();
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[SetVerifyOption] exit\n");
    return Util::ConvertResultFromInternalToExternal(result);
}


nn::Result SslConnectionImpl::SetIoMode(IoMode mode) NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[SetIoMode] enter\n");

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        Connection::IoMode tmpMode = static_cast<Connection::IoMode>(mode.mode);
        if (tmpMode != Connection::IoMode_Blocking && tmpMode != Connection::IoMode_NonBlocking)
        {
            result = ResultInvalidIoMode();
            break;
        }

        PRSocketOptionData prSockOpt;
        prSockOpt.option             = PR_SockOpt_Nonblocking;
        prSockOpt.value.non_blocking = (tmpMode == Connection::IoMode_Blocking)?PR_FALSE:PR_TRUE;

        std::lock_guard<decltype(m_ioModeLock)> lock(m_ioModeLock);
        if (PR_SetSocketOption(m_SslSocket, &prSockOpt) != PR_SUCCESS)
        {
            result = ResultErrorLower();
            break;
        }

        m_IoMode = tmpMode;
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[SetIoMode] exit\n");
    return result;
}

nn::Result SslConnectionImpl::SetSessionCacheMode(SessionCacheMode mode) NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[SetSessionCacheMode] enter\n");

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        SslSessionManager::SessionCacheMode setCacheMode;
        Connection::SessionCacheMode tmpMode = static_cast<Connection::SessionCacheMode>(mode.mode);
        if (tmpMode == Connection::SessionCacheMode_None)
        {
            setCacheMode = SslSessionManager::SessionCacheMode_None;
        }
        else if (tmpMode == Connection::SessionCacheMode_SessionId)
        {
            setCacheMode = SslSessionManager::SessionCacheMode_SessionId;
        }
        else if (tmpMode == Connection::SessionCacheMode_SessionTicket)
        {
            setCacheMode = SslSessionManager::SessionCacheMode_SessionTicket;
        }
        else
        {
            result = ResultInvalidSessionCacheMode();
            break;
        }

        result = SslSessionManager::SetSessionCacheMode(m_SslSocket, setCacheMode);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[SetSessionCacheMode] exit\n");
    return result;
}

nn::Result SslConnectionImpl::SetRenegotiationMode(RenegotiationMode mode) NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[SetRenegotiationMode] enter\n");

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        PRBool                        isUseSecureRenegotiation = false;
        Connection::RenegotiationMode tmpMode = static_cast<Connection::RenegotiationMode>(mode.mode);
        if(tmpMode == Connection::RenegotiationMode_None)
        {
            isUseSecureRenegotiation = false;
        }
        else if(tmpMode == Connection::RenegotiationMode_Secure)
        {
            isUseSecureRenegotiation = true;
        }
        else
        {
            result = ResultInvalidRenegotiationMode();
            break;
        }

        if (SSL_OptionSet(m_SslSocket, SSL_ENABLE_RENEGOTIATION, isUseSecureRenegotiation)
            == SECFailure)
        {
            result = ResultErrorLower();
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[SetRenegotiationMode] exit\n");
    return result;
}

nn::Result SslConnectionImpl::GetSocketDescriptor(Out<int32_t> outValue) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetSocketDescriptor] enter\n");

    do
    {
        if (m_SslSocket == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetSocketDescriptor] no descriptor set\n");
            ret = ResultSocketNotRegistered();
            break;
        }

        *outValue = m_Socket;
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[GetSocketDescriptor] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::GetHostName(const OutBuffer& hostNameBuffer, Out<uint32_t> outHostNameLength) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    char                        *buf;
    size_t                      bufSize;
    size_t                      nameLen = PL_strlen(m_HostName);

    NN_DETAIL_SSL_DBG_PRINT("[GetHostName] enter\n");

    do
    {
        buf = hostNameBuffer.GetPointerUnsafe();
        bufSize = hostNameBuffer.GetSize();

        if (bufSize < nameLen)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetHostName] provided buffer %d bytes, need %d\n",
                                    bufSize,
                                    nameLen + 1);
            ret = ResultBufferTooShort();
            break;
        }
        PL_strncpy(buf, m_HostName, static_cast<PRUint32>(nameLen));
        buf[nameLen] = 0;
        *outHostNameLength = static_cast<uint32_t>(nameLen);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[GetHostName] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::GetVerifyOption(Out<VerifyOption> outVerifyOption) NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyOption] enter\n");

    outVerifyOption->options = static_cast<uint32_t>(m_VerifyOptionMask);

    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyOption] exit\n");

    return ResultSuccess();
}


nn::Result SslConnectionImpl::GetIoMode(Out<IoMode> outIoMode) NN_NOEXCEPT
{
    IoMode                      ioMode;

    NN_DETAIL_SSL_DBG_PRINT("[GetIoMode] enter\n");

    m_ioModeLock.Lock();
    ioMode.mode = static_cast<uint32_t>(m_IoMode);
    m_ioModeLock.Unlock();

    *outIoMode = ioMode;

    NN_DETAIL_SSL_DBG_PRINT("[GetIoMode] exit\n");

    return ResultSuccess();
}

nn::Result SslConnectionImpl::GetSessionCacheMode(Out<SessionCacheMode> outSessionCachemode) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetSessionCacheMode] enter\n");
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        SessionCacheMode                    sessionCacheMode;
        SslSessionManager::SessionCacheMode mode = SslSessionManager::SessionCacheMode_None;
        result = SslSessionManager::GetSessionCacheMode(m_SslSocket, &mode);
        if (result.IsFailure())
        {
            break;
        }

        if (mode == SslSessionManager::SessionCacheMode_SessionId)
        {
            sessionCacheMode.mode = static_cast<uint32_t>(Connection::SessionCacheMode_SessionId);
        }
        else
        {
            sessionCacheMode.mode = static_cast<uint32_t>(Connection::SessionCacheMode_None);
        }
        // TODO: Connection::SessionCacheMode_SessionTicket is not supported yet

        *outSessionCachemode = sessionCacheMode;
    } while (NN_STATIC_CONDITION(false));
    NN_DETAIL_SSL_DBG_PRINT("[GetSessionCacheMode] exit\n");

    return result;
}

nn::Result SslConnectionImpl::GetRenegotiationMode(Out<RenegotiationMode> outRenegotiationMode) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetRenegotiationMode] enter\n");
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        PRBool            isSecureRenegotiationOn = false;
        RenegotiationMode renegotiationMode;
        if (SSL_OptionGet(m_SslSocket, SSL_ENABLE_RENEGOTIATION, &isSecureRenegotiationOn)
            == SECFailure)
        {
            result = ResultErrorLower();
            break;
        }

        if (isSecureRenegotiationOn == PR_FALSE)
        {
            renegotiationMode.mode = static_cast<uint32_t>(Connection::RenegotiationMode_None);
        }
        else
        {
            renegotiationMode.mode = static_cast<uint32_t>(Connection::RenegotiationMode_Secure);
        }

        *outRenegotiationMode = renegotiationMode;
    } while (NN_STATIC_CONDITION(false));
    NN_DETAIL_SSL_DBG_PRINT("[GetRenegotiationMode] exit\n");

    return result;
}

nn::Result SslConnectionImpl::FlushSessionCache() NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_PRINT("[FlushSessionCache] enter\n");

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        result = SslSessionManager::DeleteSessionCache(m_SslSocket);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[FlushSessionCache] exit\n");
    return result;
}

//  Common implementation since SFDL doesn't support method overloading
nn::Result SslConnectionImpl::DoHandshake(uint32_t *outCertSize,
                                          uint32_t *outCertCount,
                                          void     *outCertBuf,
                                          uint32_t bufSize) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    PRInt32                     timeout;
    int                         rval;
    SECStatus                   status;
    PRErrorCode                 prError;

    do
    {
        //  Verify the user has setup a socket and host name
        if (m_SslSocket == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] no socket set\n");
            ret = ResultSocketNotRegistered();
            break;
        }

        if ((m_VerifyOptionMask & Connection::VerifyOption::VerifyOption_HostName)
            == Connection::VerifyOption::VerifyOption_HostName)
        {
            if (PL_strnlen(m_HostName, g_MaxHostNameLength) == 0)
            {
                NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] hostname not set\n");
                ret = ResultHostNameNotRegistered();
                break;
            }
        }

        //  Set the server's host name so we can verify the server cert
        //  Always set the host name (if available) because it is used to lookup session ID
        if (m_HostName[0] != '\0')
        {
            rval = SSL_SetURL(m_SslSocket, m_HostName);
            if (rval != 0)
            {
                NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] failed to set server hostname: %d\n", rval);
                ret = ResultHostNameNotRegistered();
                break;
            }
        }

        //  Before handshake can proceed, the TrustedCertManager must be done
        //  initializing, otherwise we will not have trusted CAs.
        NssCore::InitStatus nssStatus =
            NssCore::GetInitStatus(TrustedCertManager::g_TcmDeferredInitId, true);
        if (nssStatus == NssCore::InitStatus_InitFail)
        {
            NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] TCM failed to init\n");
            ret = ResultErrorLower();
            break;
        }

        //  Save off the buf pointer so auth callback can fill it in
        //  during the handshake operation.
        m_pAuthCertData      = static_cast<uint8_t *>(outCertBuf);
        m_AuthCertSizeNeeded = 0;
        m_AuthCertMaxSize    = bufSize;
        m_AuthCertError      = ResultSuccess();

        timeout = GetIoTimeout();

#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT_IO
        NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] calling force w/to, sock %p, to %u\n",
                                m_SslSocket,
                                timeout);
#endif

        status = SSL_ForceHandshakeWithTimeout(m_SslSocket, timeout);
        if (m_AuthCertError.IsFailure())
        {
            //  Save the error occurred during verification for non-blocking IoMode because
            //  DoHandshake could return ResultIoWouldBlock when IoMode is set to non-blocking
            //  mode even after something is set to m_AuthCertError (e.g. in auth cert callback).
            //  In this case, a user would call DoHandshake again and m_AuthCertError would
            //  get reset above.
            m_LastAuthCertError = m_AuthCertError;
        }

        if (status != SECSuccess)
        {
            prError = PR_GetError();

            if (prError == PR_IO_TIMEOUT_ERROR)
            {
                //  If our timeout was 0, we are in non-blocking mode
                //  so we want to bail out with an error of "would block".
                if (timeout == 0)
                {
                    ret = ResultIoWouldBlock();
                    break;
                }

                ret = NssUtil::ConvertNssErrorToResult(prError);
                break;
            }
            else if (prError == PR_END_OF_FILE_ERROR)
            {
                //  TCP connection was closed by the peer
                ret = ResultConnectionClosed();
                break;
            }

            //  If the failure is due to the auth failing due to cert
            //  validation, report that now. This is done only when SSL_ForceHandshakeWithTimeout
            //  returns an error which is not PR_WOULD_BLOCK_ERROR because SSL handshake is still
            //  in progress when PR_WOULD_BLOCK_ERROR is returned
            if (m_AuthCertError.IsFailure() && prError != PR_WOULD_BLOCK_ERROR)
            {
                NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] server auth failed: %d:%d\n",
                                        m_AuthCertError.GetModule(),
                                        m_AuthCertError.GetDescription());
                ret = ResultVerifyCertFailed();
                break;
            }

            //  Any other error just gets converted and passed up
            ret = NssUtil::ConvertNssErrorToResult(
                prError,
                this->GetParentSslContext()->GetParentSslService()->GetInterfaceVersion());
            break;
        }
        else
        {
            if (m_LastAuthCertError.IsFailure())
            {
                //  Auth may have succeeded, but if the caller provided a buffer
                //  which was not big enough we need to report that up.
                if (ResultInsufficientServerCertBuffer::Includes(m_LastAuthCertError))
                {
                    ret = m_LastAuthCertError;
                }
                else
                {
                    ret = ResultVerifyCertFailed();
                    m_AuthCertError = m_LastAuthCertError;
                }
            }

            NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] success, last auth error: %d:%d (ret:%d)\n",
                                    m_LastAuthCertError.GetModule(),
                                    m_LastAuthCertError.GetDescription(),
                                    ret.GetDescription());
        }
    } while (NN_STATIC_CONDITION(false));

    if(ret.IsSuccess())
    {
        if(outCertCount != nullptr)
        {
            *outCertCount = m_authCertCount;
        }

        if(outCertSize != nullptr)
        {
            *outCertSize = m_authCertSize;
        }
    }

    // Clears cert count and size when handshake is complete.
    //  - If result is success
    //  - Or if there is an error and the error is NOT IoWouldBlock
    if(!ResultIoWouldBlock::Includes(ret))
    {
        m_authCertCount = 0;
        m_authCertSize  = 0;

        // Clears verify error list if
        //  - Result is success or
        //  - Result is not ResultIoWouldBlock and not ResultVerifyCertFailed
        if(!ResultVerifyCertFailed::Includes(ret))
        {
            m_errorList.ClearErrors();
        }
    }

    //  Reset any server cert buffer parms stashed off
    m_pAuthCertData    = nullptr;
    m_AuthCertMaxSize  = 0;

    return Util::ConvertResultFromInternalToExternal(ret);
} // NOLINT(impl/function_size)

nn::Result SslConnectionImpl::DoHandshake()
{
    nn::Result                  ret;

#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT_IO
    NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] enter\n");
#endif

    ret = DoHandshake(nullptr, nullptr, nullptr, 0);

#ifdef NN_DETAIL_SSL_ENABLE_DEBUG_PRINT_IO
    NN_DETAIL_SSL_DBG_PRINT("[DoHandshake] exit\n");
#endif

    return ret;
}


nn::Result SslConnectionImpl::DoHandshakeGetServerCert(Out<uint32_t> outServerCertSize,
                                                       Out<uint32_t> outServerCertCount,
                                                       const OutBuffer& outServerCert) NN_NOEXCEPT
{
    uint32_t                    certSize = 0;
    uint32_t                    certCount = 0;
    char                        *certBuf = outServerCert.GetPointerUnsafe();
    nn::Result                  ret;

    NN_DETAIL_SSL_DBG_PRINT("[DoHandshakeGetServerCert] enter\n");

    ret = DoHandshake(&certSize,
                      &certCount,
                      certBuf,
                      static_cast<uint32_t>(outServerCert.GetSize()));
    if (outServerCertSize.GetPointer() != nullptr)
    {
        *outServerCertSize = certSize;
    }

    if (outServerCertCount.GetPointer() != nullptr)
    {
        *outServerCertCount = certCount;
    }

    NN_DETAIL_SSL_DBG_PRINT("[DoHandshakeGetServerCert] exit\n");

    return ret;
}


nn::Result SslConnectionImpl::Read(uint32_t        *bytesRead,
                                   void            *outBuf,
                                   uint32_t        outBufSize,
                                   PRIntn          flags,
                                   PRIntervalTime  timeout)
{
    nn::Result                  ret = ResultSuccess();
    PRInt32                     rxBytes;

    NN_DETAIL_SSL_DBG_PRINT_IO("[Read] enter\n");

    do
    {
        if (m_SslSocket == nullptr)
        {
            ret = ResultSocketNotRegistered();
            break;
        }

        rxBytes = PR_Recv(m_SslSocket, outBuf, outBufSize, flags, timeout);
        if (rxBytes < 0)
        {
            PRErrorCode prErr = PR_GetError();
            if ((prErr == PR_WOULD_BLOCK_ERROR) ||
                ((prErr == PR_IO_TIMEOUT_ERROR) && (timeout == 0)))
            {
                NN_DETAIL_SSL_DBG_PRINT_IO("[Read] (%s) would block\n",
                                        (flags == 0) ? "normal" : "peek");
                ret = ResultIoWouldBlock();
            }
            else if (prErr == PR_IO_TIMEOUT_ERROR)
            {
                NN_DETAIL_SSL_DBG_PRINT("[Read] (%s) timeout\n",
                                        (flags == 0) ? "normal" : "peek");
                ret = ResultIoTimeout();
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[Read] (%s) err %d\n",
                                        (flags == 0) ? "normal" : "peek",
                                        prErr);
                ret = NssUtil::ConvertNssErrorToResult(prErr);
            }

            break;
        }

        *bytesRead = static_cast<uint32_t>(rxBytes);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT_IO("[Read] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::Read(Out<uint32_t> bytesRead, const OutBuffer& buffer) NN_NOEXCEPT
{
    uint32_t                    rxBytes;
    char                        *buf = buffer.GetPointerUnsafe();
    uint32_t                    bufSize = static_cast<uint32_t>(buffer.GetSize());
    nn::Result                  ret;

    ret = Read(&rxBytes, buf, bufSize, 0, GetIoTimeout());
    if (ret.IsSuccess())
    {
        *bytesRead = rxBytes;
    }

    return ret;
}


nn::Result SslConnectionImpl::Write(Out<uint32_t> bytesWritten, const InBuffer& buffer) NN_NOEXCEPT
{
    int32_t                     txBytes;
    const char                  *buf = buffer.GetPointerUnsafe();
    uint32_t                    bufSize = static_cast<uint32_t>(buffer.GetSize());
    nn::Result                  ret = ResultSuccess();
    PRIntervalTime              timeout = GetIoTimeout();
    PRErrorCode                 prErr;

    NN_DETAIL_SSL_DBG_PRINT_IO("[Write] enter (bufSize:%d)(buf:%p)\n", bufSize, buf);

    do
    {
        if (m_SslSocket == nullptr)
        {
            ret = ResultSocketNotRegistered();
            break;
        }

        txBytes = PR_Send(m_SslSocket, buf, bufSize, 0, timeout);
        if (txBytes < 0)
        {
            prErr = PR_GetError();
            if ((prErr == PR_WOULD_BLOCK_ERROR) ||
                ((prErr == PR_IO_TIMEOUT_ERROR) && timeout == 0))
            {
                NN_DETAIL_SSL_DBG_PRINT_IO("[Write] would block\n");
                ret = ResultIoWouldBlock();
            }
            else if (prErr == PR_IO_TIMEOUT_ERROR)
            {
                NN_DETAIL_SSL_DBG_PRINT("[Write] timeout\n");
                ret = ResultIoTimeout();
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[Write] err %d\n", prErr);
                ret = NssUtil::ConvertNssErrorToResult(prErr);
            }

            break;
        }

        *bytesWritten = static_cast<uint32_t>(txBytes);
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT_IO("[Write] exit (bytesWritten:%d)(buf:%p)\n", *bytesWritten, buf);

    return  Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::Pending(Out<int32_t> byteCount) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    PRInt32                     pendingBytes;

    NN_DETAIL_SSL_DBG_PRINT_IO("[Pending] enter\n");

    do
    {
        if (m_SslSocket == nullptr)
        {
            ret = ResultSocketNotRegistered();
            break;
        }

        pendingBytes = SSL_DataPending (m_SslSocket);
        if (pendingBytes < 0)
        {
            ret = NssUtil::ConvertNssErrorToResult(PR_GetError());
            break;
        }

        *byteCount = pendingBytes;
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT_IO("[Pending] exit\n");

    return  Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::Peek(Out<uint32_t> byteCount, const OutBuffer& buffer) NN_NOEXCEPT
{
    uint32_t                    rxBytes;
    char                        *buf = buffer.GetPointerUnsafe();
    uint32_t                    bufSize = static_cast<uint32_t>(buffer.GetSize());
    nn::Result                  ret;

    NN_DETAIL_SSL_DBG_PRINT_IO("[Peek] enter\n");

    ret = Read(&rxBytes, buf, bufSize, PR_MSG_PEEK, GetIoTimeout());
    if (ret.IsSuccess())
    {
        *byteCount = rxBytes;
    }

    NN_DETAIL_SSL_DBG_PRINT_IO("[Peek] exit\n");

    return ret;
}


nn::Result SslConnectionImpl::Poll(Out<PollEvent> outEvent, PollEvent inEvent, uint32_t msecTimeout) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();
    Connection::PollEvent       evt = static_cast<Connection::PollEvent>(inEvent.events);
    Connection::PollEvent       rxEvt = Connection::PollEvent::PollEvent_None;
    PRPollDesc                  pollDesc;
    PRInt32                     prRet;
    PRErrorCode                 prErr;

    NN_DETAIL_SSL_DBG_PRINT_IO("[Poll] enter\n");

    //  Setup the NSPR polling descriptor so it knows what to watch
    pollDesc.in_flags = 0;
    pollDesc.out_flags = 0;
    pollDesc.fd = m_SslSocket;
    if ((evt & Connection::PollEvent::PollEvent_Read) == Connection::PollEvent::PollEvent_Read)
    {
        pollDesc.in_flags |= PR_POLL_READ;
    }

    if ((evt & Connection::PollEvent::PollEvent_Write) == Connection::PollEvent::PollEvent_Write)
    {
        pollDesc.in_flags |= PR_POLL_WRITE;
    }

    if ((evt & Connection::PollEvent::PollEvent_Except) == Connection::PollEvent::PollEvent_Except)
    {
        pollDesc.in_flags |= PR_POLL_EXCEPT;
    }

    do
    {
        if (m_SslSocket == nullptr)
        {
            ret = ResultSocketNotRegistered();
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT_IO("[Poll] calling on sock 0x%X, flags 0x%X\n",
                                   m_SslSocket,
                                   pollDesc.in_flags);
        prRet = PR_Poll(&pollDesc, 1, PR_MillisecondsToInterval(msecTimeout));
        if (prRet == -1)
        {
            prErr = PR_GetError();
            NN_DETAIL_SSL_DBG_PRINT("[Poll] failed (%s)\n", PR_ErrorToName(prErr));
            ret = NssUtil::ConvertNssErrorToResult(prErr);
            break;
        }
        else if (prRet == 0)
        {
            NN_DETAIL_SSL_DBG_PRINT("[Poll] timed out\n");
            ret = ResultIoTimeout();
            break;
        }

        //  We have one or more events received.  Handle them!
        if (pollDesc.out_flags & PR_POLL_READ)
        {
            rxEvt |= Connection::PollEvent::PollEvent_Read;
        }

        if (pollDesc.out_flags & PR_POLL_WRITE)
        {
            rxEvt |= Connection::PollEvent::PollEvent_Write;
        }

        if (pollDesc.out_flags & PR_POLL_EXCEPT)
        {
            rxEvt |= Connection::PollEvent::PollEvent_Except;
        }

        if (pollDesc.out_flags & PR_POLL_HUP)
        {
            rxEvt |= Connection::PollEvent::PollEvent_Read;
        }
    } while (NN_STATIC_CONDITION(false));

    (*outEvent).events =  static_cast<uint32_t>(rxEvt);

    NN_DETAIL_SSL_DBG_PRINT_IO("[Poll] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::GetVerifyCertError() NN_NOEXCEPT
{
    nn::Result                  ret = m_AuthCertError;

    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyCertError] enter\n");

    m_AuthCertError = ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyCertError] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}

nn::Result SslConnectionImpl::GetVerifyCertErrors(const OutBuffer& outResultArray, Out<uint32_t> outResultCountWritten, Out<uint32_t> outTotalResultCount) NN_NOEXCEPT
{
    nn::Result ret = ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyCertErrors] enter\n");

    *outResultCountWritten = 0;
    *outTotalResultCount = 0;

    const auto* pErrorNode = m_errorList.GetHead();
    while(pErrorNode != nullptr)
    {
        ++*outTotalResultCount;

        if(outResultArray.GetSize() >= *outTotalResultCount * sizeof(nn::Result))
        {
            nn::Result* pResultArray = reinterpret_cast<nn::Result*>(outResultArray.GetPointerUnsafe());
            pResultArray[*outTotalResultCount - 1] = Util::ConvertResultFromInternalToExternal(NssUtil::ConvertNssErrorToResult(pErrorNode->error));
            ++*outResultCountWritten;
        }

        pErrorNode = pErrorNode->pNext;
    }

    NN_DETAIL_SSL_DBG_PRINT("[GetVerifyCertErrors] exit\n");

    return Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::GetNeededServerCertBufferSize(Out<uint32_t> outNeededSize) NN_NOEXCEPT
{
    nn::Result                  ret = ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetNeededServerCertBufferSize] enter\n");

    do
    {
        *outNeededSize = m_AuthCertSizeNeeded;
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[GetNeededServerCertBufferSize] exit\n");

    return ret;
}


SslConnectionImpl::ErrorPriority SslConnectionImpl::ConvertErrorToPriorityValue(PRErrorCode error) NN_NOEXCEPT
{
    if(error == PR_SUCCESS)
    {
        return ErrorPriority_NoError;
    }
    else if(error == SEC_ERROR_REVOKED_CERTIFICATE)
    {
        return ErrorPriority_High;
    }
    else if(error == SEC_ERROR_EXPIRED_CERTIFICATE ||
            error == SEC_ERROR_EXPIRED_ISSUER_CERTIFICATE)
    {
        return ErrorPriority_Lowest;
    }
    else
    {
        // Anything not handled above will be treated as medium priority
        // e.g.: SSL_ERROR_BAD_CERT_DOMAIN, SEC_ERROR_UNKNOWN_ISSUER
        return ErrorPriority_Medium;
    }
}


nn::Result SslConnectionImpl::SetOption(OptionType optionType, bool enable) NN_NOEXCEPT
{
    nn::Result ret = ResultSuccess();

    Connection::OptionType optType = static_cast<Connection::OptionType>(optionType.type);

    NN_DETAIL_SSL_DBG_PRINT_IO("[SetOption] enter (opt:0x%x)\n", optType);

    do
    {
        switch (optType)
        {
        case Connection::OptionType_DoNotCloseSocket:
            {
                if (m_SslSocket != nullptr)
                {
                    ret = ResultSocketAlreadyRegistered();
                }
                else
                {
                    m_IsDoNotCloseSocketInShim = enable;
                }

                break;
            }
        case Connection::OptionType_GetServerCertChain:
            {
                m_GetServerCertChain = enable;
                break;
            }
        case Connection::OptionType_SkipDefaultVerify:
            {
                m_IsSkipDefaultValidation = enable;
                break;
            }
        default:
            {
                ret = ResultInvalidOptionType();
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT_IO("[SetOption] exit\n");

    return  Util::ConvertResultFromInternalToExternal(ret);
}


nn::Result SslConnectionImpl::GetOption(Out<bool> outIsEnabled, OptionType optionType) NN_NOEXCEPT
{
    nn::Result ret = ResultSuccess();

    Connection::OptionType optType = static_cast<Connection::OptionType>(optionType.type);

    NN_DETAIL_SSL_DBG_PRINT_IO("[GetOption] enter (opt:0x%x)\n", optType);

    do
    {
        switch (optType)
        {
        case Connection::OptionType_DoNotCloseSocket:
            {
                *outIsEnabled = m_IsDoNotCloseSocketInShim;
                break;
            }
        case Connection::OptionType_GetServerCertChain:
            {
                *outIsEnabled = m_GetServerCertChain;
                break;
            }
        case Connection::OptionType_SkipDefaultVerify:
            {
                *outIsEnabled = m_IsSkipDefaultValidation;
                break;
            }
        default:
            {
                ret = ResultInvalidOptionType();
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT_IO("[GetOption] exit\n");

    return  Util::ConvertResultFromInternalToExternal(ret);
}

const char* SslConnectionImpl::GetSslVersionStrFromInt(uint16_t sslVersion) NN_NOEXCEPT
{
    const char* pSslVersion = nullptr;

    switch(sslVersion)
    {
        case SSL_LIBRARY_VERSION_2:
        {
            pSslVersion = "SSLv2";
        } break;

        case SSL_LIBRARY_VERSION_3_0:
        {
            pSslVersion = "SSLv3";
        } break;

        case SSL_LIBRARY_VERSION_TLS_1_0:
        {
            pSslVersion = "TLSv1.0";
        } break;

        case SSL_LIBRARY_VERSION_TLS_1_1:
        {
            pSslVersion = "TLSv1.1";
        } break;

        case SSL_LIBRARY_VERSION_TLS_1_2:
        {
            pSslVersion = "TLSv1.2";
        } break;

        case SSL_LIBRARY_VERSION_TLS_1_3:
        {
            pSslVersion = "TLSv1.3";
        } break;

        default:
        {
            return nullptr;
        }
    }

    return pSslVersion;
}

nn::Result SslConnectionImpl::GetCipherInfo(uint32_t structureVersion, OutBuffer outCipherInfo) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] enter\n");

    do
    {
        SECStatus rval = SECSuccess;
        SSLChannelInfo channel;
        SSLCipherSuiteInfo cipherSuite;

        char* pCipherInfoBuf = outCipherInfo.GetPointerUnsafe();
        size_t cipherInfoBufLen = outCipherInfo.GetSize();

        if( structureVersion == 1 )
        {
            if(cipherInfoBufLen != sizeof(Connection::CipherInfo))
            {
                NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] Unexpected structure size for version. Size %d, Version: %d\n", static_cast<int>(cipherInfoBufLen), structureVersion);
                result = ResultInternalLogicError();
                break;
            }
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] Unsupported structure version! Version: %d\n", structureVersion);
            result = ResultInternalLogicError();
            break;
        }

        Connection::CipherInfo* pCipherInfo = reinterpret_cast<Connection::CipherInfo*>(pCipherInfoBuf);
        if (m_SslSocket == nullptr)
        {
            result = ResultSocketNotRegistered();
            break;
        }

        rval = SSL_GetChannelInfo(m_SslSocket, &channel, sizeof(channel));
        if (rval != SECSuccess || channel.length != sizeof(channel))
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] ERROR - SSL_GetChannelInfo: %d\n\n", rval);
            result = ResultErrorLower();
            break;
        }

        if(channel.cipherSuite == 0)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] No ssl connection.\n");
            result = ResultNoSslConnection();
            break;
        }

        rval = SSL_GetCipherSuiteInfo(channel.cipherSuite, &cipherSuite, sizeof(cipherSuite));
        if(rval != SECSuccess || cipherSuite.cipherSuiteName == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] ERROR - SSL_GetCipherSuiteInfo: %d\n\n", rval);
            result = ResultErrorLower();
            break;
        }

        int requiredBufLen = NN_DETAIL_SSL_SNPRINTF(pCipherInfo->cipherName, Connection::CipherInfo::NameBufLen, "%s", cipherSuite.cipherSuiteName);
        ++requiredBufLen; // One more for null terminator

        if(Connection::CipherInfo::NameBufLen < requiredBufLen)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] Cipher name buffer too short!\n");
            result = ResultInternalLogicError();
            break;
        }

        requiredBufLen = 0;
        const char* pSslVersion = GetSslVersionStrFromInt(channel.protocolVersion);

        if( nullptr == pSslVersion )
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] Unknown ssl protocol version!\n");
            result = ResultErrorLower();
            break;
        }

        requiredBufLen = NN_DETAIL_SSL_SNPRINTF(pCipherInfo->versionName, Connection::CipherInfo::VersionBufLen, "%s", pSslVersion);
        ++requiredBufLen; // One more for null terminator

        if(Connection::CipherInfo::NameBufLen < requiredBufLen)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] Version name buffer too short!\n");
            result = ResultErrorLower();
            break;
        }

    } while (NN_STATIC_CONDITION(false));

    NN_DETAIL_SSL_DBG_PRINT("[GetCipherInfo] exit\n");

    return Util::ConvertResultFromInternalToExternal(result);
}

} } }
