﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/util/util_IntrusiveList.h>

#include "server/ssl_SslServiceImpl.h"
#include "server/ssl_SslContextImpl.h"
#include "server/ssl_SslConnectionImpl.h"
#include "server/ssl_SslSfObservable.h"

namespace nn { namespace ssl { namespace detail {

// -------------------------------------------------------------------------------------------------
// Constants
// -------------------------------------------------------------------------------------------------
const uint32_t g_MaxSslContextCountTotal    = 32;            // Max SSL contexts in the system
const uint32_t g_MaxSslConnectionCountTotal = 32;            // Max SSL connections in the system
const uint32_t g_MaxSslClientCountTotal     = 32;            // Max SSL clients (sub-domains) in the system
const uint32_t g_DefaultIoMsecTimeout       = 1000 * 60 * 5; // Default I/O timeout: 5 minutes

// TODO: How much do we need? We may need 64 "bytes", see ref below.
// Ref: http://stackoverflow.com/questions/8724954/what-is-the-maximum-number-of-characters-for-a-host-name-in-unix
const uint32_t g_MaxHostNameLength          = MaxHostNameLength + 1;

typedef nn::util::IntrusiveList<
    SslSfObservable,
    nn::util::IntrusiveListMemberNodeTraits<SslSfObservable, &SslSfObservable::listNode> >
        SfObservableListType;

class SslDbObjObserver : public SslSfObserver
{
private:
    nn::os::Mutex               *m_Lock;
    SfObservableListType        *m_pList;

public:
    SslDbObjObserver(nn::os::Mutex *lock, SfObservableListType *observerables);
    ~SslDbObjObserver();

    //  SslSfObserver::OnDestroy overload to cleanup tracking
    void OnDestroy(SslSfObservable *object) NN_NOEXCEPT;
};

// -------------------------------------------------------------------------------------------------
// Root database of SSL service
// -------------------------------------------------------------------------------------------------
class SslServiceDatabase
{
public:
    typedef nn::Result (*SslDbActionCb)(PRCList  *pCtxList,
                                        void     *pCbArg);

    static nn::Result FindContextsAndTakeAction(SslServiceImpl  *pOwnerService,
                                                SslDbActionCb   pActionCb,
                                                void            *pCbArg);

    static nn::Result AddSslService(SslServiceImpl *newService) NN_NOEXCEPT;
    static nn::Result AddSslContext(SslContextImpl *newContext) NN_NOEXCEPT;
    static nn::Result AddSslConnection(SslConnectionImpl *newConnection) NN_NOEXCEPT;

    static nn::Result GetContextCount(SslServiceImpl *service, uint32_t *count) NN_NOEXCEPT;
    static nn::Result GetConnectionCount(SslContextImpl *context, uint32_t *count) NN_NOEXCEPT;

private:
    static nn::os::Mutex                g_Lock;
    static SfObservableListType         g_Clients;
    static SfObservableListType         g_Contexts;
    static SfObservableListType         g_Connections;

    static SslDbObjObserver             g_ServiceObserver;
    static SslDbObjObserver             g_ContextObserver;
    static SslDbObjObserver             g_ConnectionObserver;

    static nn::Result GetContextCountLocked(SslServiceImpl *service, uint32_t *count) NN_NOEXCEPT;
    static nn::Result GetConnectionCountLocked(SslContextImpl *context, uint32_t *count) NN_NOEXCEPT;
};


}}}
