﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "server/ssl_NssCommon.h"
#include "server/ssl_NssConfigurator.h"
#include "server/ssl_NssCore.h"
#include "server/ssl_NssConfigFsManager.h"
#include "server/ssl_Util.h"
#include "server/ssl_InternalPki.h"
#include "server/ssl_SslTrustedCertManager.h"
#include "debug/ssl_DebugUtil.h"

namespace nn { namespace ssl { namespace detail {

namespace {
char* PrivGetModulePassword(PK11SlotInfo *slot, PRBool retry, void *arg)
{
    NN_UNUSED(slot);
    NN_UNUSED(retry);
    NN_UNUSED(arg);

    // TODO: real implementation needed

    return nullptr;
}
} // namespace


void                            *NssCore::g_pNssDeferredInitThread = nullptr;
nn::os::Semaphore               NssCore::g_NssDeferredInitStart(0, 1);
void                            *NssCore::g_NssDeferredInitInfoList = nullptr;
NssCore::InitStatus             NssCore::g_NssDeferredInitStatus = NssCore::InitStatus_Idle;
nn::os::Mutex                   NssCore::g_NssDeferredInitLock(false);
nn::os::ConditionVariable       NssCore::g_NssDeferredInitCv;
bool                            NssCore::g_NssInitDone = false;
void                            *NssCore::g_pNssBgTaskThread = nullptr;
bool                            NssCore::g_BgTaskShutdown = false;
nn::os::Mutex                   NssCore::g_NssBgTaskLock(false);
nn::os::ConditionVariable       NssCore::g_NssBgTaskCv;
uint64_t                        NssCore::g_NssBgTasks = 0;
uint64_t                        NssCore::g_ActiveBgTasks = 0;
NssCore::BgTaskInfo             NssCore::g_AllBgTaskInfo[NssCore::g_MaxBgTasks];    //  Need 1 entry per bit


NssCore::InitStatus NssCore::GetInitStatusLocked(const char *pInId)
{
    NssCore::InitStatus         status = InitStatus_Idle;

    do
    {
        if ((!NssCore::g_NssInitDone) ||
            (NssCore::g_NssDeferredInitStatus == InitStatus_InitFail))
        {
            //  NssCore hasn't even finished normal init or it failed, bail out
            break;
        }

        PRCList *pList = reinterpret_cast<PRCList *>(NssCore::g_NssDeferredInitInfoList);
        for (PRCList *pCurNode = PR_LIST_HEAD(pList);
             pCurNode != pList;
             pCurNode = PR_NEXT_LINK(pCurNode))
        {
            NssUtil::ObjListNode    *pObjNode;
            DeferredInitInfo        *pInfo;
            bool                    match = false;

            pObjNode = reinterpret_cast<NssUtil::ObjListNode *>(pCurNode);
            pInfo    = reinterpret_cast<DeferredInitInfo *>(pObjNode->pObj);

            if (pInId != nullptr)
            {
                match = (PL_strcmp(pInId, pInfo->pId) == 0);
            }

            if ((pInId == nullptr) || match)
            {
                if ((status == InitStatus_Idle) ||
                    (status > pInfo->status))
                {
                    status = pInfo->status;
                }

                if (match)
                {
                    break;
                }
            }
        }
    } while (NN_STATIC_CONDITION(false));

    //  Update/sync the overall status if no specific ID was provided
    if ((pInId == nullptr) && (status != NssCore::g_NssDeferredInitStatus))
    {
        NssCore::g_NssDeferredInitStatus = status;
    }

    return status;
}


void NssCore::DeferredInitThreadCb(void *pArg)
{
    //  Wait on the start sem to be released telling us to "go".
    //  This is used to avoid any parallel operations during init
    //  which could cause issues.
    NN_DETAIL_SSL_DBG_PRINT("[DeferredInitThread] entered, wait for signal\n");
    NssCore::g_NssDeferredInitStart.Acquire();
    NN_DETAIL_SSL_DBG_PRINT("[DeferredInitThread] awakened, do init(s)\n");

    //  Walk the deferred init callback list, execute each one
    NssCore::g_NssDeferredInitLock.Lock();

    do
    {
        NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER();

        PRCList *pList = reinterpret_cast<PRCList *>(NssCore::g_NssDeferredInitInfoList);
        for (PRCList *pCurNode = PR_LIST_HEAD(pList);
             pCurNode != pList;
             pCurNode = PR_NEXT_LINK(pCurNode))
        {
            NssUtil::ObjListNode    *pObjNode;
            DeferredInitInfo        *pInfo;

            pObjNode = reinterpret_cast<NssUtil::ObjListNode *>(pCurNode);
            pInfo    = reinterpret_cast<DeferredInitInfo *>(pObjNode->pObj);

            //  Update the status to be initializing so we know it is in-progress
            pInfo->status = InitStatus_Initializing;

            //  Unlock before calling the callback, allowing waiters to
            //  check status.
            NssCore::g_NssDeferredInitLock.Unlock();

            NssCore::InitStatus status = pInfo->pCb(pInfo->pArg);

            //  Lock things back down before we continue
            NssCore::g_NssDeferredInitLock.Lock();

            //  If an error was encountered, just stop
            pInfo->status = status;
            if (pInfo->status != InitStatus_InitDone)
            {
                NN_DETAIL_SSL_DBG_PRINT("[DeferredInitThread] id \'%s\' failed: %d\n",
                                        pInfo->pId,
                                        pInfo->status);
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    //  All deferred init callbacks have run, unlock and signal the CV
    NssCore::g_NssDeferredInitLock.Unlock();
    NssCore::g_NssDeferredInitCv.Broadcast();
}


void NssCore::BgTaskThreadCb(void *pArg)
{
    NN_DETAIL_SSL_DBG_PRINT("[BgTaskThreadCb] started\n");

    //  Take the lock and wait on the cv to be signalled
    NssCore::g_NssBgTaskLock.Lock();
    while (!NssCore::g_BgTaskShutdown)
    {
        int                     i;

        //  Wait for an event to be signalled or shutdown
        NssCore::g_NssBgTaskCv.TimedWait(NssCore::g_NssBgTaskLock,
                                         nn::TimeSpan::FromMilliSeconds(NssCore::g_NssBgTaskWaitTimeoutMs));

        do
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_FOR_BGTASK();

            //  One or more events have been signalled.  Walk each
            //  one to determine if it was signalled.
            for (i = 0; i < NssCore::g_MaxBgTasks; i++)
            {
                BgTaskCb            pCurCb = nullptr;
                uint64_t            curEvent = 1ULL << i;

                if ((NssCore::g_ActiveBgTasks & curEvent) != 0)
                {
                    //  This event was signalled.  Clear the signal state
                    //  then unlock the bg task lock before calling the cb, just
                    //  in case the cb re-signals itself.
                    NssCore::g_ActiveBgTasks &= ~curEvent;

                    pCurCb = NssCore::g_AllBgTaskInfo[i].pCb;
                    if (pCurCb == nullptr)
                    {
                        NN_DETAIL_SSL_WARN_PRINT("No cb for signalled event %d\n", i);
                        continue;
                    }

                    NssCore::g_NssBgTaskLock.Unlock();
                    pCurCb(NssCore::g_AllBgTaskInfo[i].pArg);
                    NssCore::g_NssBgTaskLock.Lock();
                }
            }
        } while (NN_STATIC_CONDITION(false));

        //  All events due to the signal which awakened us have been handled,
        //  and we are still holding the lock, so the loop will go back to
        //  waiting to be signalled again.
    }

    //  Release the lock before we exit
    NssCore::g_NssBgTaskLock.Unlock();
    NN_DETAIL_SSL_DBG_PRINT("[BgTaskThreadCb] exiting\n");
}


nn::Result NssCore::Initialize()
{
    NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER();
    NN_DETAIL_SSL_DBG_PRINT("[NssCore] init\n");

    nn::Result                  ret = ResultSuccess();

    do
    {
        g_NssDeferredInitLock.Lock();
        if (g_NssInitDone)
        {
            //  Done already, break out
            g_NssDeferredInitLock.Unlock();
            break;
        }
        g_NssDeferredInitLock.Unlock();

        //  Init NSPR so we have basic OS functionality available
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("PR_Init");

            PR_Init(PR_SYSTEM_THREAD, PR_PRIORITY_NORMAL, 1);
        }

        //  Enable debug trace in NSS
        //  Make sure to define TRACE, DEBUG and NSS_HAVE_GETENV in the NSS library too
        //  to enable it.
#if defined(NN_DETAIL_SSL_ENABLE_DEBUG_PRINT) && defined(NN_DETAIL_SSL_DBG_PRINT_NSS)
        PR_SetEnv("SSLTRACE=127");
#endif

        //  Set the environment variable NSS_SDB_USE_CACHE to be "no"
        //  so NSS' softoken does not measure access time for database access
        //  and does not cache (saves memory).  This avoids unnecessary delays
        //  due to filesystem speed when importing PKI.
        PR_SetEnv("NSS_SDB_USE_CACHE=no");
        PR_SetEnv("NSS_DISABLE_ARENA_FREE_LIST=true");

        //  Create the list for deferred info
        PRCList *pList = new PRCList;
        if (pList == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] Failed to create deferred init list\n");
            ret = ResultInsufficientMemory();
            break;
        }

        PR_INIT_CLIST(pList);
        NssCore::g_NssDeferredInitInfoList = pList;

        //  The filesystem should have already been mounted and ready.
        //  Get NSS setup and going.  Create the NSS configuration and session
        //  cache directories/
        const char *pCfgDir = NssConfigurator::Initializer::GetConfigDir();
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("SetupNssCfgTree");

            size_t pathLen = strlen(pCfgDir);
            pathLen++;

            ret = NssConfigFsManagerBase::SetupNssCfgTree(pCfgDir, pathLen);
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] Failed to create NSS config dir\n");
                ret = ResultInternalNssFailedFileMount();
                break;
            }
        }

        // Init NSS with our config directory, no prefix and read-write.
        // For read-only mode, use NSS_INIT_READONLY rather than 0 for flags.
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("NSS_Initialize");

            SECStatus status = NSS_Initialize(pCfgDir,
                                    "",
                                    "",
                                    SECMOD_DB,
                                    0);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] Failed to initialize NSS: %s\n",
                    PR_ErrorToName(PR_GetError()));
                ret = ResultInternalNssFailedToInitializeNssLib();
                break;
            }
        }

        NN_DETAIL_SSL_DBG_PRINT("[ssl] initialized NSS\n");

        // ****************************************************************************************
        // TODO: Temporary code
        // Setup the callback function used to prompt the user (or app) for passwords
        // needed to get access to PKCS#11 modules
        PK11_SetPasswordFunc(PrivGetModulePassword);

        ret = NssConfigurator::Initializer::SetDefaultOptions();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] faled to set default options\n");
            ret = ResultErrorLower();
            break;
        }

        ret = NssConfigurator::Initializer::SetDefaultCipherPreference();
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] faled to set default cipher prefs\n");
            ret = ResultErrorLower();
            break;
        }

        ret = NssConfigurator::Initializer::SetDefaultCipherPolicy(NssConfigurator::Initializer::CipherPolicyRegion_US);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] faled to set default cipher policy\n");
            ret = ResultErrorLower();
            break;
        }

        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("Pkcs12ModuleConfigurator::Initialize");

            ret = NssConfigurator::Initializer::Pkcs12ModuleConfigurator::Initialize();
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] faled to init PKCS#12 module\n");
                ret = ResultErrorLower();
                break;
            }
        }

        // Overwrite SSL session cache timeout
        nnsdkNssPortSetSslSidTimeout(
            0,                        // ssl_sid_timeout  : use NSS default
            g_Ssl3SessionCacheTimeout // ssl3_sid_timeout : an hour
        );

        SSL_ClearSessionCache();

        //  Setup the device specific credentials in the core PKCS#11 slot
        //  and default trust.
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("Pk11ModuleUtil::CheckAndSetupDefaultSlot");

            ret = Pk11ModuleUtil::CheckAndSetupDefaultSlot();
            if (ret.IsFailure())
            {
                //  Complain about this, but do not fail.  Users just won't be
                //  able to use custom PKI.
                NN_DETAIL_SSL_DBG_PRINT("[ssl] faled to init default PKCS#11 slot\n");
            }
        }

        //  Init the trusted certificate manager
        NN_DETAIL_SSL_DBG_PRINT("[ssl] initializing TrustedCertManager...\n");
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("TrustedCertManager::Initialize");

            ret = TrustedCertManager::Initialize();
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to init Trusted Cert Manager\n");
            }
        }

        NN_DETAIL_SSL_DBG_PRINT("[ssl] Setup internal PKI...\n");
        {
            NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME("InternalPkiManager::CheckAndInstallIfNeeded");
            ret = InternalPkiManager::CheckAndInstallIfNeeded();
            if (ret.IsFailure())
            {
                //  Complain but do not fail?
                NN_DETAIL_SSL_WARN_PRINT("[ssl] unable to setup device pki (non-fatal)\n");
            }
        }

        //  Create the background task worker thread.  This is used for handling
        //  background maintenance type operations, such as cleaning up caches
        //  or other things which can/should be done in the background.
        g_pNssBgTaskThread =
            PR_CreateThread(PR_SYSTEM_THREAD,
                            NssCore::BgTaskThreadCb,
                            nullptr,
                            PR_PRIORITY_LOW,
                            PR_GLOBAL_THREAD,
                            PR_JOINABLE_THREAD,
                            NssCore::g_BgTaskThreadStackSize);
        if (g_pNssBgTaskThread == nullptr)
        {
            NN_DETAIL_SSL_WARN_PRINT("Failed to create bg task thread\n");
            ret = ResultErrorLower();
            break;
        }

        //  Create the deferred init worker thread.  NSPR managed threads are
        //  automatically cleaned up (including their stack) when they exit,
        //  so we do not need to maintain a reference to it beyond this scope.
        g_pNssDeferredInitThread =
            PR_CreateThread(PR_SYSTEM_THREAD,
                            NssCore::DeferredInitThreadCb,
                            nullptr,
                            PR_PRIORITY_LOW,
                            PR_GLOBAL_THREAD,
                            PR_JOINABLE_THREAD,
                            NssCore::g_DeferredInitThreadStackSize);
        if (g_pNssDeferredInitThread == nullptr)
        {
            NN_DETAIL_SSL_WARN_PRINT("Failed to create deferred init thread\n");
            ret = ResultErrorLower();
            break;
        }

        //  Mark the core NSS as done and the deferred as initializing.
        //  The deferred init thread will be started later, once other
        //  parts of SSL init are complete.
        g_NssDeferredInitLock.Lock();
        g_NssInitDone = true;
        g_NssDeferredInitStatus = InitStatus_NotInitialized;
        g_NssDeferredInitLock.Unlock();

        ret = ResultSuccess();
    } while (NN_STATIC_CONDITION(false));

    if (!ret.IsSuccess())
    {
        //  Inject a "fake" entry into the list so status shows up as failed?
        NssCore::g_NssDeferredInitStatus = InitStatus_InitFail;
    }

    return ret;
}    //  NOLINT(impl/function_size)

nn::Result NssCore::Finalize()
{
    nn::Result                  ret = ResultSuccess();
    PRThread                    *pThread;

    NN_DETAIL_SSL_DBG_PRINT("[NssCore] finish\n");

    do
    {
        SECStatus                   status;

        //  Before we start to shutdown, we need to finish running any
        //  deferred inits which are still going.
        if (NssCore::g_pNssDeferredInitThread != nullptr)
        {
            pThread = reinterpret_cast<PRThread *>(NssCore::g_pNssDeferredInitThread);
            PRStatus prStatus = PR_JoinThread(pThread);
            if (prStatus != PR_SUCCESS)
            {
                NN_DETAIL_SSL_DBG_PRINT("[NssCore] Failed to join deferred init thread\n");
            }
        }

        //  Notify the bg task thread we are shutting down so it will exit,
        //  then wait for it to finish.
        if (NssCore::g_pNssBgTaskThread != nullptr)
        {
            NssCore::g_NssBgTaskLock.Lock();
            NssCore::g_BgTaskShutdown = true;
            NssCore::g_NssBgTaskLock.Unlock();
            NssCore::g_NssBgTaskCv.Signal();
            pThread = reinterpret_cast<PRThread *>(NssCore::g_pNssBgTaskThread);
            PRStatus prStatus = PR_JoinThread(pThread);
            if (prStatus != PR_SUCCESS)
            {
                NN_DETAIL_SSL_DBG_PRINT("[NssCore] Failed to join bg task thread\n");
            }
        }

        //  Lock down the ready lock, we're shutting down
        g_NssDeferredInitLock.Lock();

        if (NssCore::g_NssDeferredInitInfoList != nullptr)
        {
            PRCList *pList = reinterpret_cast<PRCList *>(NssCore::g_NssDeferredInitInfoList);
            for (PRCList *pCurNode = PR_LIST_HEAD(pList);
                 pCurNode != pList;
                 pCurNode = PR_LIST_HEAD(pList))
            {
                NssUtil::ObjListNode    *pObjNode;
                DeferredInitInfo        *pInfo;

                PR_REMOVE_LINK(pCurNode);
                pObjNode = reinterpret_cast<NssUtil::ObjListNode *>(pCurNode);
                pInfo    = reinterpret_cast<DeferredInitInfo *>(pObjNode->pObj);

                pObjNode->pObj = nullptr;
                delete pObjNode;

                PL_strfree(pInfo->pId);
                pInfo->pId = nullptr;

                delete pInfo;
            }

            delete pList;
            pList = nullptr;
            NssCore::g_NssDeferredInitInfoList = nullptr;
        }

        //  Tear down the TrustedCertManager, we're done
        ret = TrustedCertManager::Finalize();
        if (!ret.IsSuccess())
        {
            //  Complain but continue
            NN_DETAIL_SSL_DBG_PRINT("[NssCore] TrustedCertManager::Finalize failed (%d-%d)\n",
                                    ret.GetModule(),
                                    ret.GetDescription());
        }

        // Calling SSL_ClearSessionCache() here per the description in
        // https://developer.mozilla.org/en-US/docs/Mozilla/Projects/NSS/SSL_functions/sslfnc.html
        //
        // NOTE: If an SSL client application does not call SSL_ClearSessionCache before shutdown,
        // NSS_Shutdown fails with the error code SEC_ERROR_BUSY.
        SSL_ClearSessionCache();

        NN_DETAIL_SSL_DBG_PRINT("[NssCore] NSS shutdown\n");
        status = NSS_Shutdown();
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssCore] NSS_Shutdown failed (%d)\n", status);
            //  Allow this to keep going
        }

        NN_DETAIL_SSL_DBG_PRINT("[NssCore] NSPR cleanup\n");
        PR_Cleanup();

        //  Mark ready as false and release the lock.
        g_NssInitDone = false;
        g_NssDeferredInitLock.Unlock();

    } while (NN_STATIC_CONDITION(false));

    return ret;
}


void NssCore::StartDeferredInit()
{
    if (NssCore::g_NssDeferredInitStart.GetCurrentCount() == 0)
    {
        NssCore::g_NssDeferredInitStart.Release();
    }
}


NssCore::InitStatus NssCore::GetInitStatus(bool wait)
{
    InitStatus                  status;

    //  Take the ready lock used to protect the ready flag and
    //  used by the CV if we wish to wait.
    NssCore::g_NssDeferredInitLock.Lock();
    status = NssCore::g_NssDeferredInitStatus;
    while ((status != InitStatus_InitDone) &&
           (status != InitStatus_InitFail))
    {
        if (wait)
        {
            NssCore::g_NssDeferredInitCv.Wait(NssCore::g_NssDeferredInitLock);

            //  Re-sync the global status to cover spurious wakeup case
            status = GetInitStatusLocked(nullptr);
        }
        else
        {
            break;
        }
    }

    NssCore::g_NssDeferredInitLock.Unlock();

    return status;
}


NssCore::InitStatus NssCore::GetInitStatus(const char *pInId, bool wait)
{
    InitStatus                  status;

    //  Take the ready lock used to protect the ready flag and
    //  used by the CV if we wish to wait.
    NssCore::g_NssDeferredInitLock.Lock();
    status = GetInitStatusLocked(pInId);
    while ((status != InitStatus_InitDone) &&
           (status != InitStatus_InitFail))
    {
        if (wait)
        {
            NssCore::g_NssDeferredInitCv.Wait(NssCore::g_NssDeferredInitLock);

            //  Re-sync the status to cover spurious wakeup case
            status = GetInitStatusLocked(pInId);
        }
        else
        {
            break;
        }

    }

    NssCore::g_NssDeferredInitLock.Unlock();

    return status;
}


nn::Result NssCore::AddDeferredInit(const char      *pInId,
                                    DeferredInitCb  pInCb,
                                    void            *pInArg)
{
    nn::Result                  result = ResultSuccess();
    char                        *pId = nullptr;
    DeferredInitInfo            *pInfo = nullptr;
    NssUtil::ObjListNode        *pNode = nullptr;

    do
    {
        if ((pInId == nullptr) || (pInCb == nullptr))
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssCore::AddDeferredInit] invalid id or no callback\n");
            result = ResultInvalidPointer();
            break;
        }

        pId = PL_strdup(pInId);
        if (pId == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssCore::AddDeferredInit] unable to dup name\n");
            result = ResultInsufficientMemory();
            break;
        }

        pInfo = new DeferredInitInfo;
        if (pInfo == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssCore::AddDeferredInit] failed to create DeferredInitInfo for \'%s\'\n", pInId);
            result = ResultInsufficientMemory();
            break;
        }

        pNode = new NssUtil::ObjListNode;
        if (pNode == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssCore::AddDeferredInit] failed to create ObjListNode for \'%s\'\n", pInId);
            result = ResultInsufficientMemory();
            break;
        }

        PR_INIT_CLIST(&pNode->links);
        pNode->pObj   = pInfo;
        pInfo->pId    = pId;
        pInfo->status = InitStatus_NotInitialized;
        pInfo->pCb    = pInCb;
        pInfo->pArg   = pInArg;

        PRCList *pList = reinterpret_cast<PRCList *>(NssCore::g_NssDeferredInitInfoList);
        PR_APPEND_LINK(&pNode->links, pList);
        NN_DETAIL_SSL_DBG_PRINT("[NssCore::AddDeferredInit] added \'%s\'\n", pInfo->pId);
    } while(NN_STATIC_CONDITION(false));

    if (!result.IsSuccess())
    {
        if (pNode != nullptr)
        {
            delete pNode;
        }

        if (pInfo != nullptr)
        {
            delete pInfo;
        }

        if (pId != nullptr)
        {
            PR_Free(pId);
        }
    }

    return result;
}


nn::Result NssCore::AllocBgTask(BgTask *pOutTask, BgTaskCb pInCb, void *pInArg)
{
    nn::Result                  result = ResultSuccess();

    NssCore::g_NssBgTaskLock.Lock();

    do
    {
        uint64_t                event = 0;
        int                     i;

        for (i = 0; i < NssCore::g_MaxBgTasks; i++)
        {
            event = 1ULL << i;
            if ((event & NssCore::g_NssBgTasks) != 0)
            {
                //  event is taken, continue
                continue;
            }

            //  Found a free event, break out
            break;
        }

        if (i == NssCore::g_MaxBgTasks)
        {
            NN_DETAIL_SSL_WARN_PRINT("No more bg tasks available!\n");
            result = ResultResourceMax();
            break;
        }

        //  Claim this slot for the caller
        NssCore::g_NssBgTasks |= event;
        NssCore::g_AllBgTaskInfo[i].pCb = pInCb;
        NssCore::g_AllBgTaskInfo[i].pArg = pInArg;
        *pOutTask = event;
    } while (NN_STATIC_CONDITION(false));

    NssCore::g_NssBgTaskLock.Unlock();

    return result;
}


nn::Result NssCore::FreeBgTask(BgTask task)
{
    nn::Result                  result = ResultSuccess();

    NssCore::g_NssBgTaskLock.Lock();

    do
    {
        //  Make sure we are being provided a valid event (or events)
        if ((NssCore::g_NssBgTasks & task) != task)
        {
            NN_DETAIL_SSL_WARN_PRINT("[FreeBgTask] Invalid task(s) requested: %X, valid are %X\n",
                                     task,
                                     NssCore::g_NssBgTasks);
            result = ResultErrorLower();
            break;
        }

        //  Walk the events, find each that matches and clear out the
        //  bg task info
        for (int i = 0; i < NssCore::g_MaxBgTasks; i++)
        {
            uint64_t            event = 1ULL << i;

            if ((event & task) == 0)
            {
                continue;
            }

            //  Clear the bg task as being allocated and clear all of
            //  its info
            NN_DETAIL_SSL_DBG_PRINT("[FreeBgTask] free task %d\n", i);
            NssCore::g_AllBgTaskInfo[i].pCb = nullptr;
            NssCore::g_AllBgTaskInfo[i].pArg = nullptr;
            NssCore::g_NssBgTasks &= ~event;
        }
    } while (NN_STATIC_CONDITION(false));

    NssCore::g_NssBgTaskLock.Unlock();

    return result;
}


nn::Result NssCore::QueueBgTask(BgTask task)
{
    nn::Result                  result = ResultSuccess();

    do
    {
        //  Be sure this task is actually valid before we queue anything
        NssCore::g_NssBgTaskLock.Lock();

        if ((NssCore::g_NssBgTasks & task) != task)
        {
            NN_DETAIL_SSL_WARN_PRINT("[QueueBgTask] Invalid task(s) requested: %X, valid are %X\n",
                                     task,
                                     NssCore::g_NssBgTasks);
            NssCore::g_NssBgTaskLock.Unlock();
            result = ResultErrorLower();
            break;
        }

        //  Mark the task as active, then unlock and signal the cv so the
        //  bg processing thread wakes up.
        NssCore::g_ActiveBgTasks |= task;
        NssCore::g_NssBgTaskLock.Unlock();
    } while (NN_STATIC_CONDITION(false));

    return result;
}

void NssCore::RunBgTask(BgTask task)
{
    //  Queue it and as long as it was successful, wake up the bg task thread
    nn::Result result = NssCore::QueueBgTask(task);
    if (result.IsSuccess())
    {
        NssCore::g_NssBgTaskCv.Signal();
    }
}


void NssCore::ScheduleBgTask(BgTask task)
{
    //  Just queue it, nothing else to do
    NssCore::QueueBgTask(task);
}

} } }
