﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <tchar.h>

#include <nn/os.h>
#include <nn/fs_Base.h>

#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_NssCommon.h"
#include "server/ssl_NssConfigFsManager.win32.h"
#include "server/ssl_Util.h"
#include "server/ssl_NssConfigurator.h"

namespace nn { namespace ssl { namespace detail {

namespace
{
    const char*  g_pMountedPath        = NN_DETAIL_SSL_NNSDK_FS_MOUNT ":/";
    const char*  g_pMountedPathSiglo   = NN_DETAIL_SSL_NNSDK_FS_MOUNT ":/siglo/";
    const size_t g_MaxFullPathLen = 48;

    // Recursively delete one directory and all files inside it
    void DeleteDir(char* pDir)
    {
        nn::Result result = nn::fs::DeleteDirectoryRecursively(pDir);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_INFO_PRINT("Failed to delete (%s) error(%d:%d)\n",
                pDir,
                result.GetModule(),
                result.GetDescription());
        }
    }

    // Delete subdirectories remained from previous executions where NssConfigFsManagerWindows::Finalize() was not called.
    // The function called from NssConfigFsManagerWindows::Initialize() method.
    void DeleteAllWorkingDirs()
    {
        char pSubDir[MAX_PATH + 1] = {0};
        fs::DirectoryHandle directory;
        nn::fs::DirectoryEntry* pEntries;
        size_t lengthMountedPathSiglo = strlen(g_pMountedPathSiglo);

        // 14 is a minimal possible length of a generated directory name
        if (lengthMountedPathSiglo >= (MAX_PATH - 14))
        {
            NN_DETAIL_SSL_DBG_PRINT("[DeleteAllWorkingDirs] Path (%s) is too long.\n", g_pMountedPathSiglo);
            return;
        }

        nn::Result result = fs::OpenDirectory(&directory, g_pMountedPathSiglo, fs::OpenDirectoryMode_Directory);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[DeleteAllWorkingDirs] Cannoy open directory (%s).\n", g_pMountedPathSiglo);
            return;
        }

        int64_t count;
        nn::fs::GetDirectoryEntryCount(&count, directory);
        pEntries = (count > 0) ? new nn::fs::DirectoryEntry[static_cast<size_t>(count)] : nullptr;
        if (pEntries != nullptr)
        {
            nn::fs::ReadDirectory(&count, &pEntries[0], directory, count);

            for(int i = 0; i < count; i++)
            {
                auto dir = pEntries[i];
                int processId;
                sscanf(dir.name, "ssl%08x", &processId);

                // Delete a subdirectory in case if it does not belong to any running process
                if ((GetProcessVersion(processId) == 0) &&
                   ((strlen(dir.name) + lengthMountedPathSiglo) < MAX_PATH))
                {
                    strncpy(pSubDir, g_pMountedPathSiglo, MAX_PATH);
                    strncat(pSubDir, dir.name, MAX_PATH - lengthMountedPathSiglo);

                    DeleteDir(pSubDir);
                }
            }

            delete[] pEntries;
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[DeleteAllWorkingDirs] Cannot read entries of the directory (%s).\n", g_pMountedPathSiglo);
        }

        nn::fs::CloseDirectory(directory);
    }

    // Delete working directory created by the current process
    // The function called from NssConfigFsManagerWindows::Finalize() method.
    void DeleteCurrentWorkingDir()
    {
        nn::Result   result = nn::ResultSuccess();

        //  It will be SslSave:/siglo_ssl
        char tmpWorkingRootPath[g_MaxFullPathLen] = {0};

        strncpy(tmpWorkingRootPath, g_pMountedPath, sizeof(tmpWorkingRootPath) - 1);

        size_t curAvailableLen = sizeof(tmpWorkingRootPath) - strnlen(tmpWorkingRootPath, sizeof(tmpWorkingRootPath));
        curAvailableLen -= (curAvailableLen != 0)?1:0;

        strncat(tmpWorkingRootPath, NssConfigurator::Initializer::GetConfigDir(), curAvailableLen);

        NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerWindows] Deleting (%s)\n", tmpWorkingRootPath);

        DeleteDir(tmpWorkingRootPath);
    }

    void CleanWorkingDir(bool isInitializing)
    {
        if (isInitializing)
        {
            // Called from NssConfigFsManagerWindows::Initialize() -> delete all subdirs
            // remained from previous executions.
            DeleteAllWorkingDirs();
        }
        else
        {
            // Called from NssConfigFsManagerWindows::Finalize() -> delete the directory created by this process
            DeleteCurrentWorkingDir();
        }
    }

}

nn::Result NssConfigFsManagerWindows::Initialize()
{
    nn::Result                  ret = nn::ResultSuccess();
    char                        pHostMountPath[MAX_PATH] = {0};

    do
    {
        // Get the path of execution file
        wchar_t tmpPath[MAX_PATH];
        if (GetModuleFileName(nullptr, tmpPath, MAX_PATH) > 0)
        {
            wchar_t* pTmp = _tcsrchr(tmpPath, _T('\\'));
            if (pTmp != nullptr)
            {
                pTmp = _tcsinc(pTmp);
                *pTmp = _T('\0');
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerWindows] Invalid path.\n");
                ret = ResultErrorLower();
                break;
            }
        }
        else
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerWindows] Failed to get path.\n");
            ret = ResultErrorLower();
            break;
        }
        wcstombs(pHostMountPath, tmpPath, sizeof(pHostMountPath));

        NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerWindows]Try mount point (%s)\n",
            pHostMountPath);
        ret = MountHostFs(pHostMountPath);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[NssConfigFsManagerWindows]Failed to mount (%s)\n",
                pHostMountPath);
            break;
        }

        //  Just in case when NssConfigFsManagerWindows::Finalize() is not called in the previous
        //  execution.
        CleanWorkingDir(true);

    } while (NN_STATIC_CONDITION(false));

    return Util::ConvertResultFromInternalToExternal(ret);
}

nn::Result NssConfigFsManagerWindows::Finalize()
{
    CleanWorkingDir(false);
    UnmountHostFs();

    return nn::ResultSuccess();
}

} } }
