﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "ssl_NssCommon.h"
#include "ssl_NssUtil.h"
#include "ssl_NssConfigurator.h"
#include "ssl_NssPkcsUtil.h"
#include "ssl_Crl.h"
#include "ssl_BdfReader.h"
#include "ssl_Util.h"
#include "debug/ssl_DebugUtil.h"

namespace nn { namespace ssl { namespace detail {

// ------------------------------------------------------------------------------------------------
// CrlManager
// ------------------------------------------------------------------------------------------------
namespace
{
    PRCList             g_ImportedBuiltInCrlList; // The list for imported built-in CRL

    void CleanupImportedCrl(PRCList *pInList, bool isFreeItem)
    {
        CERTCertDBHandle *pDataBase = CERT_GetDefaultCertDB();

        do
        {
            if (pDataBase == nullptr)
            {
                NN_DETAIL_SSL_DBG_PRINT("[CrlManager] Failed to get default CertDB.\n");
                break;
            }

            for (PRCList *pElem = PR_LIST_HEAD(pInList);
                 pElem != pInList;
                 pElem = PR_LIST_HEAD(pInList))
            {
                NssUtil::ObjListNode    *pNode;
                SECItem                 *pCrlItem;

                PR_REMOVE_AND_INIT_LINK(pElem);
                pNode = reinterpret_cast<NssUtil::ObjListNode *>(pElem);
                pCrlItem = reinterpret_cast<SECItem *>(pNode->pObj);

                if (CERT_UncacheCRL(pDataBase, pCrlItem) != SECSuccess)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[CrlManager] CERT_UncacheCRL failed (%d|%s)\n",
                        PR_GetError(), PR_ErrorToName(PR_GetError()));
                }

                uint8_t *pBuf = static_cast<uint8_t *>(pCrlItem->data);
                pCrlItem->data = nullptr;
                delete[] pBuf;

                if (isFreeItem)
                {
                    SECITEM_ZfreeItem(pCrlItem, PR_TRUE);
                }
                else
                {
                    delete pCrlItem;
                }

                delete pNode;

                NN_DETAIL_SSL_DBG_PRINT("[CrlManager] CleanupImportedCrl - freed %p.\n", pCrlItem);
            }
        } while (NN_STATIC_CONDITION(false));
    }
}

// ------------------------------------------------------------------------------------------------
// CrlManager: Private methods
// ------------------------------------------------------------------------------------------------
nn::Result CrlManager::SetupBuiltInCrl()
{
    nn::Result                  result    = nn::ResultSuccess();
    CERTCertDBHandle            *pDataBase = CERT_GetDefaultCertDB();
    SECItem                     *pCrlItem = nullptr;
    NssUtil::ObjListNode        *pNode = nullptr;
    BdfReader                   *pReaders[] = { nullptr, nullptr, nullptr };    //  Last entry MUST be nullptr
    uint32_t                    i;
    uint32_t                    j;
    uint8_t                     *pBuf = nullptr;

    do
    {
        if (pDataBase == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[CrlManager] Failed to get default CertDB.\n");
            result = ResultErrorLower();
            break;
        }

        PR_INIT_CLIST(&g_ImportedBuiltInCrlList);

        //  Create a BDF reader for CRL and test CRL.  Test CRL will only be
        //  available in special builds, so do not fail if we cannot create
        //  a reader for it, just complain.
        pReaders[0] = BdfReaderFactory::Create(BdfReader::BdfId_Crl);
        if (pReaders[0] == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[CrlManager] Failed to create BDF reader for CRL\n");
            result = ResultErrorLower();
            break;
        }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
        if(Util::IsInitializeForTestEnabled())
        {
            pReaders[1] = BdfReaderFactory::Create(BdfReader::BdfId_TestCrl);
            if (pReaders[1] == nullptr)
            {
                NN_DETAIL_SSL_WARN_PRINT("[CrlManager] Failed to create BDF reader for TEST CRL\n");
            }
        }
#endif

        for (i = 0; pReaders[i] != nullptr; i++)
        {
            uint32_t            dataCount;
            BdfMetaDataEntry    metaData;

            //  Get the data count of CRLs in this BDF
            result = pReaders[i]->GetDataCount(&dataCount);
            if (result.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[CrlManager] failed to get data count for BDF %u\n", i);
                break;
            }

            //  Walk each CRL within the BDF, get its meta data then import
            //  each as a CRL in DER format.
            for (j = 0; j < dataCount; j++)
            {
                result = pReaders[i]->GetMetaDataEntry(&metaData, j);
                if (result.IsFailure())
                {
                    NN_DETAIL_SSL_DBG_PRINT("[CrlManager] failed to get meta data for BDF %u, entry %u\n",
                                            i,
                                            j);
                    break;
                }

                //  Ensure that the status is valid for CRLs (enabled or removed only)
                BuiltinDataInfo::BuiltinDataStatus status =
                    static_cast<BuiltinDataInfo::BuiltinDataStatus>(metaData.dataStatus);
                if ((status > BuiltinDataInfo::BuiltinDataStatus_Enabled) ||
                    (status == BuiltinDataInfo::BuiltinDataStatus_Invalid))
                {
                    NN_DETAIL_SSL_DBG_PRINT("[CrlManager] invalid status (%8.8X) for BDF %u, entry %u (id %u), skipping\n",
                                            metaData.dataStatus,
                                            i,
                                            j,
                                            metaData.dataId);
                    continue;
                }

                //  Now get the backing data for the CRL entry
                pBuf = new uint8_t[metaData.dataSize];
                if (pBuf == nullptr)
                {
                    result = ResultInsufficientMemory();
                    break;
                }

                result = pReaders[i]->GetData(pBuf, metaData.dataSize, &metaData);
                if (result.IsFailure())
                {
                    break;
                }

                pCrlItem = new SECItem;
                if (pCrlItem == nullptr)
                {
                    result = ResultInsufficientMemory();
                    break;
                }

                pNode = new NssUtil::ObjListNode;
                if (pNode == nullptr)
                {
                    result = ResultInsufficientMemory();
                    break;
                }

                PR_INIT_CLIST(&pNode->links);
                pNode->pObj = pCrlItem;

                pCrlItem->type = siBuffer;
                pCrlItem->data = const_cast<unsigned char*>(pBuf);
                pCrlItem->len  = metaData.dataSize;

                if (CERT_CacheCRL(pDataBase, pCrlItem) != SECSuccess)
                {
                    NN_SDK_LOG("[CrlManager] CERT_CacheCRL failed (%d|%s)\n",
                        PR_GetError(), PR_ErrorToName(PR_GetError()));

                    result = ResultErrorLower();
                    break;
                }

                PR_APPEND_LINK(&pNode->links, &g_ImportedBuiltInCrlList);
                NN_DETAIL_SSL_DBG_PRINT("[CrlManager] SetupBuiltInCrl - Imported %p.\n", pCrlItem);
            }

            if (j != dataCount)
            {
                break;
            }
        }

        if (result.IsFailure())
        {
            if (pNode != nullptr)
            {
                delete pNode;
                pNode = nullptr;
            }

            if (pCrlItem != nullptr)
            {
                delete pCrlItem;
                pCrlItem = nullptr;
            }

            if (pBuf != nullptr)
            {
                delete[] pBuf;
                pBuf = nullptr;
            }

            CleanupImportedCrl(&g_ImportedBuiltInCrlList, false);
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}    //  NOLINT(impl/function_size)

// ------------------------------------------------------------------------------------------------
// CrlManager: Public methods
// ------------------------------------------------------------------------------------------------
nn::Result CrlManager::Initialize()
{
    NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER();

    nn::Result result = nn::ResultSuccess();

    do
    {
        result = SetupBuiltInCrl();
        if (result.IsFailure())
        {
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[CrlManager] Imported bulit-in CRL.\n");
    } while (NN_STATIC_CONDITION(false));

    return result;
}

nn::Result CrlManager::Finalize()
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        CleanupImportedCrl(&g_ImportedBuiltInCrlList, false);

        NN_DETAIL_SSL_DBG_PRINT("[CrlManager] Finalize done.\n");
    } while (NN_STATIC_CONDITION(false));

    return result;
}

}}}

