﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/ssl/ssl_Context.h>
#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>
#include "ssl_SslContextImpl.h"
#include "ssl_ServiceDatabase.h"
#include "ssl_Util.h"
#include "ssl_NssUtil.h"
#include "ssl_NssConfigurator.h"
#include "ssl_CertStore.h"
#include "ssl_NssPkcsUtil.h"

#define REPORT_PR_ERROR_UPON_FAILURE(status, funcName)                                             \
    if(status != SECSuccess)                                                                       \
    {                                                                                              \
        PRErrorCode prError = PR_GetError();                                                       \
        NN_DETAIL_SSL_DBG_PRINT("%s failed: %d (%s)\n",                                            \
            funcName, prError, PR_ErrorToName(prError));                                           \
        NN_UNUSED(prError);                                                                        \
    }

namespace nn { namespace ssl { namespace detail {


// ------------------------------------------------------------------------------------------------
//
// CertStore
//
// ------------------------------------------------------------------------------------------------
CertStore::CertStore() : m_Lock(false),
                         m_OwnerId(0),
                         m_pServerEntries(nullptr),
                         m_pCrlEntries(nullptr),
                         m_ClientEntry(nullptr)
{
}


CertStore::CertStore(uint64_t ownerId) : m_Lock(false),
                                         m_OwnerId(ownerId),
                                         m_pServerEntries(nullptr),
                                         m_pCrlEntries(nullptr),
                                         m_ClientEntry(nullptr)
{
}


CertStore::~CertStore()
{
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[~CertStore] destroy %p\n", this);

    if (m_pServerEntries != nullptr)
    {
        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             pListNode != m_pServerEntries;
             pListNode = PR_LIST_HEAD(m_pServerEntries))
        {
            NssUtil::ObjListNode    *pNode;
            ServerCertEntry         *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<ServerCertEntry *>(pNode->pObj);

            PR_REMOVE_LINK(&pNode->links);
            SslMemoryManager::Free(pNode, m_OwnerId);

            delete pCurEntry;
            pCurEntry = nullptr;
        }

        SslMemoryManager::Free(m_pServerEntries, m_OwnerId);
        m_pServerEntries = nullptr;
    }

    if (m_ClientEntry != nullptr)
    {
        delete m_ClientEntry;
        m_ClientEntry = nullptr;
    }

    if (m_pCrlEntries != nullptr)
    {
        for (PRCList *pListNode = PR_LIST_HEAD(m_pCrlEntries);
             pListNode != m_pCrlEntries;
             pListNode = PR_LIST_HEAD(m_pCrlEntries))
        {
            NssUtil::ObjListNode    *pNode;
            CrlEntry                *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<CrlEntry *>(pNode->pObj);

            PR_REMOVE_LINK(&pNode->links);
            delete pNode;
            delete pCurEntry;
        }

        delete m_pCrlEntries;
        m_pCrlEntries = nullptr;
    }
}


nn::Result CertStore::ServerEntryAdd(ServerCertEntry *pNewEntry)
{
    nn::Result                  result = ResultSuccess();
    NssUtil::ObjListNode        *pNewNode = nullptr;
    void                        *pChunk = nullptr;

    do
    {
        //  Lazy init the list
        if (m_pServerEntries == nullptr)
        {
            pChunk = SslMemoryManager::AllocateChunk(sizeof(PRCList),
                                                     m_OwnerId);
            if (pChunk == nullptr)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ServerEntryAdd) failed to alloc server entry list\n");
                result = ResultInsufficientMemory();
                break;
            }

            m_pServerEntries = reinterpret_cast<PRCList *>(pChunk);
            PR_INIT_CLIST(m_pServerEntries);
            pChunk = nullptr;
        }

        //  Create a new node for the entry.  If this fails,
        //  leave m_pServerEntries as it will get cleaned up when the
        //  object is later destroyed.
        pChunk = SslMemoryManager::AllocateChunk(sizeof(NssUtil::ObjListNode),
                                                 m_OwnerId);
        if (pChunk == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ServerEntryAdd) failed to alloc node\n");
            result = ResultInsufficientMemory();
            break;
        }

        pNewNode = reinterpret_cast<NssUtil::ObjListNode *>(pChunk);
        PR_INIT_CLIST(&pNewNode->links);
        pNewNode->pObj = pNewEntry;
        PR_APPEND_LINK(&pNewNode->links, m_pServerEntries);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result CertStore::ServerEntryRemove(ServerCertEntry *pEntry)
{
    nn::Result                  result = ResultInvalidCertStoreId();

    do
    {
        //  Since we lazy init the list, if it is null then we have no
        //  entries in our store so anything is invalid.
        if (m_pServerEntries == nullptr)
        {
            break;
        }

        //  Our lookup is iterative, which is ok because the list
        //  will always be relatively small.
        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             pListNode != m_pServerEntries;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            ServerCertEntry         *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<ServerCertEntry *>(pNode->pObj);

            if (pCurEntry == pEntry)
            {
                //  Found the entry, remove it from the list
                //  and release the node tracking it
                PR_REMOVE_LINK(&pNode->links);
                SslMemoryManager::Free(pNode, m_OwnerId);
                pListNode = nullptr;
                pNode = nullptr;
                result = ResultSuccess();
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


bool CertStore::IsServerEntryPresent(ServerCertEntry *pEntry)
{
    bool                        isPresent = false;

    do
    {
        //  Since we lazy init the list, if it is null then we have no
        //  entries in our store so anything is invalid.
        if (m_pServerEntries == nullptr)
        {
            break;
        }

        //  Our lookup is iterative, which is ok because the list
        //  will always be relatively small.
        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             pListNode != m_pServerEntries;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            ServerCertEntry         *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<ServerCertEntry *>(pNode->pObj);

            if (pCurEntry == pEntry)
            {
                isPresent = true;
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return isPresent;
}


nn::Result CertStore::SetOwnerId(uint64_t id)
{
    nn::Result                  ret = ResultSuccess();

    do
    {
        if (m_OwnerId != 0)
        {
            ret = ResultInternalLogicError();
            break;
        }

        m_OwnerId = id;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


uint64_t CertStore::GetOwnerId()
{
    return m_OwnerId;
}


ClientCertEntry* CertStore::GetClientCert()
{
    return m_ClientEntry;
}


nn::Result CertStore::ImportServerPki(uint64_t                   *pOutNewId,
                                      const char                 *pInCertData,
                                      uint32_t                   certDataSize,
                                      nn::ssl::CertificateFormat fmt)
{
    nn::Result                  ret = ResultSuccess();
    ServerCertEntry             *newServerCertEntry = nullptr;
    uint64_t                    newServerId;
    SECStatus                   status;
    char                        *newNicknameBase = nullptr;

    do
    {
        if ((pOutNewId == nullptr) || (pInCertData == nullptr))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) invalid args, pOutNewId %p, pInCertData %p\n",
                                              pOutNewId,
                                              pInCertData);
            ret = ResultInvalidPointer();
            break;
        }

        if ((fmt != nn::ssl::CertificateFormat::CertificateFormat_Pem) &&
            (fmt != nn::ssl::CertificateFormat::CertificateFormat_Der))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) invalid cert fmt: %u\n",
                                              static_cast<uint32>(fmt));
            ret = ResultInvalidCertFormat();
            break;
        }

        //  Create a new ServerCertEntry to hold the imported certs.  The
        //  collection of all certs in this PKI chain are identified via
        //  an "id" which is tied to this ServerCertEntry instance.
        newNicknameBase = NssUtil::CreateRandIdName(ServerCertEntry::NicknamePrefix, 0);
        if (newNicknameBase == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) unable to create nickname base\n");
            ret = ResultInsufficientMemory();
            break;
        }

        newServerCertEntry = new ServerCertEntry(this, newNicknameBase);
        if (newServerCertEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) unable to create ServerCertEntry\n");
            ret = ResultInsufficientMemory();
            break;
        }

        //  Null out our local for nickname since it is now part of the
        //  ServerCertEntry.  It'll be destroyed with it.
        newNicknameBase = nullptr;

        //  Call NSS CERT decode to handle the incoming cert data (chain).  The
        //  tracking of the cert data is handled in ImportServerCertificateCallbackHook,
        //  which gets a pointer to the ServerCertEntry which will hold all the
        //  data.
        status = CERT_DecodeCertPackage(const_cast<char *>(pInCertData),
                                        certDataSize,
                                        ServerCertEntry::ImportServerCertificateCallbackHook,
                                        reinterpret_cast<void *>(newServerCertEntry));
        if (status != SECSuccess)
        {
            ret = newServerCertEntry->GetImportStatus();
            if (ret.IsFailure())
            {
                break;
            }
            else
            {
                // It is possible for ServerCertEntry::GetImportStatus() to return ResultSuccess()
                // even when CERT_DecodeCertPackage() failed. This happens when the format of a PEM
                // certificate is bad so ServerCertEntry::ImportServerCertificateCallbackHook()
                // never gets called.
                PRErrorCode tmpStatus = PORT_GetError();
                if (tmpStatus != SECSuccess)
                {
                    if (tmpStatus == SEC_ERROR_BAD_DER)
                    {
                        ret = ResultInvalidCertificate();
                    }
                    else
                    {
                        ret = NssUtil::ConvertNssErrorToResult(tmpStatus);
                    }
                    break;
                }
            }
        }

        //  We have a winner!  Add it to the list then generate the ID for the
        //  caller.
        NN_DETAIL_SSL_GET_ID_FROM_PTR(newServerId, newServerCertEntry);

        m_Lock.Lock();
        ret = ServerEntryAdd(newServerCertEntry);
        m_Lock.Unlock();

        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) failed to add tracking for cert\n");
            break;
        }

        *pOutNewId = newServerId;
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerPki) created %x, id %llu\n",
                                          newServerCertEntry,
                                          *pOutNewId);
    } while (NN_STATIC_CONDITION(false));

    //  Cleanup on error
    if (ret.IsFailure())
    {
        if (newServerCertEntry != nullptr)
        {
            delete newServerCertEntry;
            newServerCertEntry = nullptr;
        }
    }

    if (newNicknameBase != nullptr)
    {
        NssUtil::DestroyRandIdName(newNicknameBase);
        newNicknameBase = nullptr;
    }

    return ret;
}


nn::Result CertStore::RemoveServerPki(uint64_t certId)
{
    nn::Result                  ret = ResultSuccess();
    ServerCertEntry             *cert = nullptr;

    do
    {
        NN_DETAIL_SSL_GET_PTR_FROM_ID(cert, certId, ServerCertEntry);
        if (cert == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveServerPki) cert id %llu is not valid\n", certId);
            ret = ResultInvalidCertStoreId();
            break;
        }

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveServerPki) cert %p\n",
                                          cert);
        m_Lock.Lock();

        ret = ServerEntryRemove(cert);
        if (!ret.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveServerPki) cert id %llu is not valid\n", certId);
        }

        m_Lock.Unlock();
    } while (NN_STATIC_CONDITION(false));

    if (ret.IsSuccess())
    {
        //  The cert id entry was removed from the map successfully.
        //  Cleanup the object which will cause its internals to be destroyed.
        delete cert;
    }

    return ret;
}


bool CertStore::IsServerCertTrusted(CERTCertificate *pCert)
{
    bool                        ret = false;

    do
    {
        //  We're being asked if the provided cert is one that we trust.
        //  This MUST be one that was specifically installed to our CertStore
        //  for us to accept it.
        m_Lock.Lock();

        //  The server list is lazy initialized, so we must check for null
        if (m_pServerEntries == nullptr)
        {
            m_Lock.Unlock();
            break;
        }

        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             (ret == false) && (pListNode != m_pServerEntries);
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            ServerCertEntry         *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<ServerCertEntry *>(pNode->pObj);

            //  Each ServerCertEntry could have multiple certs "installed"
            //  so we need to ask it to check its certs.
            ret = pCurEntry->IsServerCertTrusted(pCert);
        }

        m_Lock.Unlock();

    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result CertStore::GetTrustedCerts(CERTCertList *pOutList, uint64_t certId)
{
    nn::Result                                          result = ResultSuccess();
    ServerCertEntry                                     *pEntry = nullptr;

    do
    {
        if (pOutList == nullptr)
        {
            result = ResultInvalidPointer();
            break;
        }

        NN_DETAIL_SSL_GET_PTR_FROM_ID(pEntry, certId, ServerCertEntry);
        if (pEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(GetTrustedCerts) cert id %llu is not valid\n", certId);
            result = ResultInvalidCertStoreId();
            break;
        }

        m_Lock.Lock();

        if (!IsServerEntryPresent(pEntry))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(GetTrustedCerts) unknown cert id: %lu\n", certId);
            result = ResultInvalidCertStoreId();
            m_Lock.Unlock();
            break;
        }

        result = pEntry->GetTrustedCertList(pOutList);

        m_Lock.Unlock();
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result CertStore::GetTrustedCertList(CERTCertList *pList)
{
    nn::Result                  ret = ResultSuccess();

    do
    {
        if (pList == nullptr)
        {
            ret = ResultInvalidPointer();
            break;
        }

        m_Lock.Lock();

        if (m_pServerEntries == nullptr)
        {
            m_Lock.Unlock();
            break;
        }

        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             ret.IsSuccess() && (pListNode != m_pServerEntries);
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            ServerCertEntry         *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<ServerCertEntry *>(pNode->pObj);
            ret = pCurEntry->GetTrustedCertList(pList);
        }

        m_Lock.Unlock();
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result CertStore::ImportClientPki(uint64_t   *pOutNewId,
                                      const char *pInP12Data,
                                      const char *pInPwData,
                                      uint32_t   p12DataLen,
                                      uint32_t   pwDataLen)
{
    nn::Result                  ret = ResultSuccess();
    ClientCertEntry             *newCertEntry = nullptr;
    char                        *nickname = nullptr;

    m_Lock.Lock();

    do
    {
        //  Verify that we have not already imported a client cert.  We only
        //  allow a single import per CertStore.
        if (m_ClientEntry != nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) client cert already imported\n");
            ret = ResultClientPkiAlreadyRegistered();
            break;
        }

        //  Create a randomly generated ID name for this import.  We will pass
        //  this to the import utility to use as a nickname in the certstore.
        //  If the private credentials have already been imported, this
        //  nickname will not be used.  The utility will return us the final
        //  nickname, but we do not care.
        nickname = NssUtil::CreateRandIdName(ClientCertEntry::g_NicknameBase,
                                             ClientCertEntry::GetMaxNameSize());
        if (nickname == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) failed to alloc mem for nickname\n");
            ret = ResultInsufficientMemory();
            break;
        }

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) importing p12, request nickname \'%s\'\n", nickname);

        Pk12ImportDetails       importDetails;
        CERTCertificate         *pCert = nullptr;
        SECKEYPrivateKey        *pPrivKey = nullptr;

        importDetails.pInP12Data = pInP12Data;
        importDetails.pInPwData  = pInPwData;
        importDetails.p12DataLen = p12DataLen;
        importDetails.pwDataLen  = pwDataLen;
        ret = Pk11ModuleUtil::ImportPk12(&importDetails,
                                         nickname,
                                         ClientCertEntry::GetMaxNameSize(),
                                         &pCert,
                                         &pPrivKey);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) P12 decode failed\n");
            break;
        }

        newCertEntry = new ClientCertEntry(pCert, pPrivKey);
        if (newCertEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) failed to create new ClientCertEntry\n");
            ret = ResultInsufficientMemory();
            break;
        }

        m_ClientEntry = newCertEntry;
        NN_DETAIL_SSL_GET_ID_FROM_PTR(*pOutNewId, m_ClientEntry);

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) imported %p (%s), id %llu\n",
                                          newCertEntry,
                                          nickname,
                                          *pOutNewId);
    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    if (nickname != nullptr)
    {
        NssUtil::DestroyRandIdName(nickname);
        nickname = nullptr;
    }

    return ret;
}


nn::Result CertStore::RemoveClientPki(uint64_t certId)
{
    nn::Result                  ret = ResultSuccess();
    ClientCertEntry             *clientCert;

    m_Lock.Lock();

    do
    {
        NN_DETAIL_SSL_GET_PTR_FROM_ID(clientCert, certId, ClientCertEntry);
        if ((clientCert == nullptr) || (clientCert != m_ClientEntry))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveClientPki) cert id %llu does not match\n", certId);
            ret = ResultInvalidCertStoreId();
            break;
        }

        delete m_ClientEntry;
        m_ClientEntry = nullptr;
    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    return ret;
}

nn::Result CertStore::ImportDeviceUniqueClientPki(uint64_t    *pOutNewId,
                                                  const char  *pInNickname)
{
    nn::Result                  ret = ResultSuccess();
    ClientCertEntry             *newCertEntry = nullptr;
    CERTCertificate             *pCert = nullptr;
    SECKEYPrivateKey            *pKey = nullptr;

    m_Lock.Lock();
    do {
        //  Verify that we have not already imported a client cert.  We only
        //  allow a single import per CertStore.
        if (m_ClientEntry != nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportClientPki) client cert already imported\n");
            ret = ResultClientPkiAlreadyRegistered();
            break;
        }

        //  Verify that the internal PKI was actually imported into the
        //  basestore when the process initialized.  If this
        //  failed during init, fail with ResultErrorLower.
        ret = Pk11ModuleUtil::GetClientPki(pInNickname, &pCert, &pKey);
        if (ret.IsFailure())
        {
            NN_DETAIL_SSL_DBG_PRINT("[ImportClientPki] pki (%s) not imported at init (%d-%d)\n",
                                    pInNickname,
                                    ret.GetModule(),
                                    ret.GetDescription());
            ret = ResultErrorLower();
            break;
        }

        //  The device unique ID is already imported into the default
        //  store and trust domain.  Just create a ClientCertEntry
        //  which retains references to the key and cert.
        newCertEntry = new ClientCertEntry(pCert, pKey);
        if (newCertEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportDeviceUniqueClientPki) failed to create new ClientCertEntry\n");
            ret = ResultInsufficientMemory();
            break;
        }

        m_ClientEntry = newCertEntry;
        NN_DETAIL_SSL_GET_ID_FROM_PTR(*pOutNewId, m_ClientEntry);

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportDeviceUniqueClientPki) created %p (%s), id %llu\n",
                                          newCertEntry,
                                          pInNickname,
                                          *pOutNewId);
    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    if (ret.IsFailure())
    {
        if (pCert != nullptr)
        {
            CERT_DestroyCertificate(pCert);
            pCert = nullptr;
        }

        if (pKey != nullptr)
        {
            SECKEY_DestroyPrivateKey(pKey);
            pKey = nullptr;
        }
    }

    return ret;
}


uint64_t CertStore::GetServerCertEntryCount()
{
    uint64_t                    count = 0;

    m_Lock.Lock();

    do
    {
        if (m_pServerEntries == nullptr)
        {
            break;
        }

        for (PRCList *pListNode = PR_LIST_HEAD(m_pServerEntries);
             pListNode != m_pServerEntries;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            count++;
        }
    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    return count;
}


nn::Result CertStore::DecodeAndValidateCrl(CERTSignedCrl **pOutNewCrl,
                                           const uint8_t  *pInCrlDerData,
                                           uint32_t       crlDerDataSize,
                                           bool           enableDateCheck)
{
    nn::Result                              result = ResultSuccess();
    CERTSignedCrl                           *pCrl = nullptr;
    CERTCertificate                         *pIssuerCert = nullptr;
    CERTCertDBHandle                        *pDb = CERT_GetDefaultCertDB();
    SECItem                                 derSignedCrl;
    char                                    *pIssuerName = nullptr;
    SECStatus                               status = SECSuccess;
    PRTime                                  dateToCheck = PR_Now();

    do
    {
        if ((pOutNewCrl == nullptr) ||
            (pInCrlDerData == nullptr) ||
            (crlDerDataSize == 0))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] invalid args (pOutNewCrl %p, pInCrlDerData %p, crlDerDataSize %u)\n",
                                              pOutNewCrl,
                                              pInCrlDerData,
                                              crlDerDataSize);
            result = ResultInvalidPointer();
            break;
        }

        //  Decode the signed CRL data, the default options tell the decode
        //  to copy the data and do not keep bad entries.
        derSignedCrl.type = siBuffer;
        derSignedCrl.data = const_cast<unsigned char *>(pInCrlDerData);
        derSignedCrl.len  = crlDerDataSize;
        pCrl = CERT_DecodeDERCrlWithFlags(nullptr,
                                          &derSignedCrl,
                                          SEC_CRL_TYPE,
                                          (CRL_DECODE_KEEP_BAD_CRL |
                                           CRL_DECODE_SKIP_ENTRIES));
        if (pCrl == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] no memory to decode CRL\n");
            result = ResultInsufficientMemory();
            break;
        }

        status = CERT_CompleteCRLDecodeEntries(pCrl);
        if (status != SECSuccess)
        {
            //  Report a CRL decode error
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] failed to decode CRL\n");
            result = ResultInvalidCrlFormat();
            break;
        }

        //  Get the signing cert and verify the signature of the CRL.  This is
        //  done by matching the distinguished name (DN) of the issuer with a
        //  cert with the same DN.  To do this, the name has to be in its
        //  encoded form so CERT_FindCertByNameString will work.
        pIssuerName = CERT_NameToAsciiInvertible(&pCrl->crl.name,
                                                 CERT_N2A_INVERTIBLE);
        if (pIssuerName == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] unable to get issuer name for CRL\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  This "find" routine uses the database handle because it searches
        //  both the permanent certs (built-in CAs) and temporary certs.
        //  Unfortunately, this means that the CRL can be "validated" via
        //  signature check if a different context has imported the issuer
        //  cert but the context owning this CertStore has not.  But, in the
        //  end this just delays the failure a bit as the handshake itself
        //  where the CRL would get used will likely fail since this context
        //  will not have imported the CA cert.
        pIssuerCert = CERT_FindCertByNameString(pDb, pIssuerName);
        if (pIssuerCert == nullptr)
        {
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] unable to find issuer cert\n");
                result = ResultUnknownCrlIssuer();
                break;
            }
        }

        //  If the date check is enabled, use PR_Now() for the date check.
        //  Otherwise, get the starting date/time for the signing cert and
        //  use that as the comparison date.
        if (!enableDateCheck)
        {
            PRTime              nBefore;
            PRTime              nAfter;
            SECStatus           success;

            //  If we fail to get the issuer cert validity dates, just complain
            //  about it and use the default of NOW.
            success = CERT_GetCertTimes(pIssuerCert, &nBefore, &nAfter);
            if (success != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] WARNING: Unable to get issuer cert times, use NOW");
            }
            else
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] date check disabled, use cert nBefore %llu for date check\n", nBefore);
                dateToCheck = nBefore;
            }
        }

        //  Check if we can trust the CRL by verifying the signature.  If the
        //  signature check fails, then we are done.  If it passes then this
        //  is eligible for use in CRL checking later.
        //  NOTE: THIS IS NOT DOING FULL CHAIN VALIDATION ON THE ISSUING CERT
        //  IT IS ONLY VALIDATING THAT THE SIGNATURE OF THE CRL IS CORRECT.
        //  THE CRL WILL NOT ACTUALLY BE *USED* UNTIL A TRUST CHAIN IS USED
        //  TO VERIFY A CONNECTION.  AT THAT POINT THE CHAIN HAS BEEN VALIDATED
        //  AND WE APPLY THE CRL VIA THE APP VERIFY CALLBACK HOOK.
        status = CERT_VerifySignedData(&pCrl->signatureWrap,
                                       pIssuerCert,
                                       dateToCheck,
                                       nullptr);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[DecodeAndValidateCrl] unable to verify CRL signature\n");
            result = ResultCrlSignatureVerifyFail();
            break;
        }

        //  We are good to go, give the pointer to the CRL to the caller
        //  and null our reference so it isn't released below.
        *pOutNewCrl = pCrl;
        pCrl = nullptr;
    } while (NN_STATIC_CONDITION(false));

    if (pIssuerCert != nullptr)
    {
        CERT_DestroyCertificate(pIssuerCert);
        pIssuerCert = nullptr;
    }

    if (pIssuerName != nullptr)
    {
        PORT_Free(pIssuerName);
        pIssuerName = nullptr;
    }

    if (pCrl != nullptr)
    {
        CERT_DestroyCrl(pCrl);
        pCrl = nullptr;
    }

    return result;
}    //  NOLINT(impl/function_size)


nn::Result CertStore::GetCrlCountForApp(PRCList *pCtxList, void *pCbArg)
{
    GetCrlCountCbArgs           *pCrlCountArgs = reinterpret_cast<GetCrlCountCbArgs *>(pCbArg);
    nn::Result                  result = ResultSuccess();
    PRCList                     *pCurEntry = nullptr;
    NssUtil::ObjListNode        *pCurNode = nullptr;
    SslContextImpl              *pCtx = nullptr;
    CertStore                   *pCertStore = nullptr;
    int                         count = 0;

    do
    {
        pCrlCountArgs->total = 0;
        pCurEntry = PR_LIST_HEAD(pCtxList);
        if (pCurEntry == pCtxList)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[GetCrlCountForApp] no contexts in list?\n");
            result = ResultErrorLower();
            break;
        }

        pCurNode = reinterpret_cast<NssUtil::ObjListNode *>(pCurEntry);
        pCtx = reinterpret_cast<SslContextImpl *>(pCurNode->pObj);

        //  Walk the entire list of contexts, lock each CertStore and get
        //  its CRL count.
        pCertStore = pCtx->GetCertStore();
        result = pCertStore->WalkLockAndGetCrlCount(pCrlCountArgs->pCertStore,
                                                    pCtxList,
                                                    pCurEntry,
                                                    &count);
        if (result.IsFailure())
        {
            //  If we encountered a failure then all CertStores other than
            //  the one which triggered this callback will be unlocked
            //  so we can safely break out.
            break;
        }

        //  Copy back the count and unlock all of the CertStores.
        pCrlCountArgs->total = count;
        pCertStore->WalkAndUnlock(pCrlCountArgs->pCertStore,
                                  pCtxList,
                                  pCurEntry);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result CertStore::WalkLockAndGetCrlCount(CertStore *pReqStore,
                                             PRCList   *pList,
                                             PRCList   *pThisEntry,
                                             int       *pCount)
{
    PRCList                     *pNextEntry = nullptr;
    nn::Result                  result = ResultSuccess();
    bool                        locked = false;
    int                         count = 0;

    do
    {
        //  Do not re-take the lock if this is the CertStore which requested
        //  the CRL count for the app.
        if (pReqStore != this)
        {
            locked = m_Lock.TryLock();
            if (!locked)
            {
                result = ResultResourceBusy();
                break;
            }
        }

        //  See if we need to traverse further down the list
        pNextEntry = PR_NEXT_LINK(pThisEntry);
        if (pNextEntry != pList)
        {
            NssUtil::ObjListNode    *pNextNode;
            SslContextImpl          *pNextCtx;
            CertStore               *pNextCertStore;

            pNextNode = reinterpret_cast<NssUtil::ObjListNode *>(pNextEntry);
            pNextCtx = reinterpret_cast<SslContextImpl *>(pNextNode->pObj);
            pNextCertStore = pNextCtx->GetCertStore();

            //  Get the count from the node(s) after us.  Hooray recursion!
            result = pNextCertStore->WalkLockAndGetCrlCount(pReqStore,
                                                            pList,
                                                            pNextEntry,
                                                            &count);
            if (result.IsFailure())
            {
                break;
            }
        }

        //  Get the count of CRLs we have imported.  If we traversed down
        //  the list further than 'count' now contains the count from
        //  the other contexts.
        if (m_pCrlEntries != nullptr)
        {
            for (PRCList *pCurEntry = PR_LIST_HEAD(m_pCrlEntries);
                 pCurEntry != m_pCrlEntries;
                 pCurEntry = PR_NEXT_LINK(pCurEntry))
            {
                count++;
            }
        }

        //  Done, return our count to the caller
        *pCount = count;
    } while (NN_STATIC_CONDITION(false));

    //  Only unlock if we encountered an error.  Otherwise we need to keep
    //  all of the CertStores locked until we are done counting.
    if (result.IsFailure())
    {
        if (locked)
        {
            m_Lock.Unlock();
        }
    }

    return result;
}


nn::Result CertStore::WalkAndUnlock(CertStore *pReqStore,
                                    PRCList   *pList,
                                    PRCList   *pThisEntry)
{
    PRCList                     *pNextEntry = nullptr;
    nn::Result                  result = ResultSuccess();

    do
    {
        //  Recursively traverse the list, unlock from back to front
        pNextEntry = PR_NEXT_LINK(pThisEntry);
        if (pNextEntry != pList)
        {
            NssUtil::ObjListNode    *pNextNode;
            SslContextImpl          *pNextCtx;
            CertStore               *pNextCertStore;

            pNextNode = reinterpret_cast<NssUtil::ObjListNode *>(pNextEntry);
            pNextCtx = reinterpret_cast<SslContextImpl *>(pNextNode->pObj);
            pNextCertStore = pNextCtx->GetCertStore();
            result = pNextCertStore->WalkAndUnlock(pReqStore,
                                                   pList,
                                                   pNextEntry);
        }

        //  If the store requesting the CRL count and walk is this one,
        //  skip the unlock as it is done in the CRL import method.
        if (pReqStore != this)
        {
            m_Lock.Unlock();
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result CertStore::ImportCrl(uint64_t       *pOutNewId,
                                const uint8_t  *pInCrlDerData,
                                uint32_t       crlDerDataSize)
{
    nn::Result                  result = ResultSuccess();
    CERTSignedCrl               *pCrl = nullptr;
    CrlEntry                    *pCrlEntry = nullptr;
    NssUtil::ObjListNode        *pNode = nullptr;
    GetCrlCountCbArgs           cbArgs;
    SslContextImpl              *pCtx = nullptr;
    bool                        enableDateCheck = true;

    do
    {
        if (m_OwnerId == 0)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] no owning Context, cannot import");
            result = ResultErrorLower();
            break;
        }

        //  Get the owning context of this cert store.  This is needed so
        //  we can ultimately get back to the SslServiceDatabase and tally up
        //  all imported CRLs for the calling app (process).
        NN_DETAIL_SSL_GET_PTR_FROM_ID(pCtx, m_OwnerId, SslContextImpl);
        enableDateCheck = pCtx->IsCrlDateCheckEnabled();

        //  Decode and validate the CRL data.
        result = CertStore::DecodeAndValidateCrl(&pCrl,
                                                 pInCrlDerData,
                                                 crlDerDataSize,
                                                 enableDateCheck);
        if (result.IsFailure())
        {
            //  Error already logged and status set, bail out
            break;
        }

        //  Lock down the list, we need an accurate number and our action
        //  callback will ensure we don't take it 2x.
        m_Lock.Lock();

        cbArgs.pCertStore = this;
        cbArgs.total      = 0;

        while (NN_STATIC_CONDITION(true))
        {
            result =
                SslServiceDatabase::FindContextsAndTakeAction(pCtx->GetParentSslService(),
                                                              CertStore::GetCrlCountForApp,
                                                              &cbArgs);
            if (ResultResourceBusy::Includes(result))
            {
                //  We have a potential deadlock situation, release our lock
                //  and yield before continuing.
                m_Lock.Unlock();
                nn::os::YieldThread();
                m_Lock.Lock();
                continue;
            }

            //  We either succeeded or encountered a different error, break out
            break;
        }

        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] Unable to get CRL count\n");
            break;
        }

        if (cbArgs.total >= nn::ssl::MaxAppCrlImportCount)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] app at max CRL count\n");
            result = ResultResourceMax();
            break;
        }

        //  Create a new CRL entry and track it.
        pCrlEntry = new CrlEntry(pCrl);
        if (pCrlEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] unable to create CrlEntry\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Null out pCrl, the CrlEntry owns the reference now
        pCrl = nullptr;

        //  Create a list node object and link it to the list of imported
        //  CRL.
        pNode = new NssUtil::ObjListNode;
        if (pNode == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] unable to create ObjListNode for new CRL\n");
            result = ResultInsufficientMemory();
            break;
        }

        PR_INIT_CLIST(&pNode->links);
        pNode->pObj = pCrlEntry;

        //  Lazy init the CRL entries list
        if (m_pCrlEntries == nullptr)
        {
            m_pCrlEntries = new PRCList;
            if (m_pCrlEntries == nullptr)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ImportCrl] failed to alloc CRL list\n");
                result = ResultInsufficientMemory();
                break;
            }

            PR_INIT_CLIST(m_pCrlEntries);
        }

        PR_APPEND_LINK(&pNode->links, m_pCrlEntries);

        //  Done, return a new ID to the caller
        //  Null out pNode and pCrlEntry, they are now being tracked
        NN_DETAIL_SSL_GET_ID_FROM_PTR(*pOutNewId, pCrlEntry);
        pNode = nullptr;
        pCrlEntry = nullptr;
    } while (NN_STATIC_CONDITION(false));

    if (m_Lock.IsLockedByCurrentThread())
    {
        m_Lock.Unlock();
    }

    if (pNode != nullptr)
    {
        delete pNode;
        pNode = nullptr;
    }

    if (pCrlEntry != nullptr)
    {
        delete pCrlEntry;
        pCrlEntry = nullptr;
    }

    if (pCrl != nullptr)
    {
        CERT_DestroyCrl(pCrl);
        pCrl = nullptr;
    }

    return result;
}    //  NOLINT(impl/function_size)


nn::Result CertStore::RemoveCrl(uint64_t crlId)
{
    nn::Result                  result = ResultSuccess();
    CrlEntry                    *pCrlEntry = nullptr;
    PRCList                     *pListNode = nullptr;

    do
    {
        //  First, ensure we have a valid ID
        NN_DETAIL_SSL_GET_PTR_FROM_ID(pCrlEntry, crlId, CrlEntry);
        if ((pCrlEntry == nullptr) || (m_pCrlEntries == nullptr))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveCrl) CRL id %llu is not valid\n", crlId);
            result = ResultInvalidCertStoreId();
            break;
        }

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveCrl) crl %p\n", pCrlEntry);

        m_Lock.Lock();

        for (pListNode = PR_LIST_HEAD(m_pCrlEntries);
             pListNode != m_pCrlEntries;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            CrlEntry                *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<CrlEntry *>(pNode->pObj);

            if (pCurEntry == pCrlEntry)
            {
                //  Remove the link and delete the node.  The actual
                //  CrlEntry object will be delete outside of this.
                PR_REMOVE_LINK(&pNode->links);
                pNode->pObj = nullptr;
                delete pNode;
                pNode = nullptr;
                break;
            }
        }

        m_Lock.Unlock();

        if (pListNode == m_pCrlEntries)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(RemoveCrl) CRL id %llu not found, invalid\n", crlId);
            result = ResultInvalidCertStoreId();
            break;
        }

        delete pCrlEntry;
        pCrlEntry = nullptr;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


bool CertStore::HasImportedCrl()
{
    bool                        hasImportedCrl = false;

    m_Lock.Lock();

    if (m_pCrlEntries != nullptr)
    {
        hasImportedCrl = (PR_LIST_HEAD(m_pCrlEntries) != m_pCrlEntries);
    }

    m_Lock.Unlock();

    return hasImportedCrl;
}


bool CertStore::IsAnyCertInChainRevokedFromCrl(CERTCRLEntryReasonCode  *pOutReason,
                                               CERTCertificate         **pOutRevokedCert,
                                               const CERTCertList      *pTrustChain)
{
    bool                        revoked = false;
    CERTCRLEntryReasonCode      reason;

    m_Lock.Lock();

    do
    {
        //  If we have no imported CRLs, then it's not revoked by any import
        if (m_pCrlEntries == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CertStore::IsAnyCertInChainRevokedFromCrl] no imported CRL, pass\n");
            break;
        }

        //  Walk the list of imported CRLs, see if this cert is revoked
        for (PRCList *pListNode = PR_LIST_HEAD(m_pCrlEntries);
             (revoked == false) && (pListNode != m_pCrlEntries);
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            CrlEntry                *pCurEntry;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurEntry = reinterpret_cast<CrlEntry *>(pNode->pObj);

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CertStore::IsAnyCertInChainRevokedFromCrl] check against CRL entry %p\n",
                                              pCurEntry);

            //  Check with this CRL if the cert or any cert in its chain
            //  has been revoked.
            revoked = pCurEntry->IsAnyCertInChainRevoked(&reason,
                                                         pOutRevokedCert,
                                                         pTrustChain);
        }
    } while (NN_STATIC_CONDITION(false));

    m_Lock.Unlock();

    return revoked;
}


char* CertStore::ExtractNickname(CERTCertificate *pCert)
{
    char                        *pRet = nullptr;
    char                        *pTmp = nullptr;

    do
    {
        //  If there is no nickname set then this is likely a temp cert
        //  or one which has been recently added to the store and not
        //  refreshed/forced to update.
        if (pCert->nickname == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ExtractNickname] cert %p has no nickname\n");
            break;
        }

        //  Once added to the store, a prefix can be injected by NSS
        //  to the front, separated by a ":".  Search for the ":" and
        //  move past it when we dup the string, if we find one.
        pTmp = PL_strchr(pCert->nickname, ':');
        if (pTmp == nullptr)
        {
            pTmp = pCert->nickname;
        }
        else
        {
            pTmp++;
        }

        pRet = PL_strdup(pTmp);
    } while (NN_STATIC_CONDITION(false));

    return pRet;
}


// ------------------------------------------------------------------------------------------------
//
// ServerCertEntry
//
// ------------------------------------------------------------------------------------------------
const char* ServerCertEntry::NicknamePrefix = "_nxa_sc";

SECStatus ServerCertEntry::ImportServerCertificateCallbackHook(void    *pArg,
                                                               SECItem **pCertItems,
                                                               int     certsCount)
{
    SECStatus                   ret = SECFailure;
    nn::Result                  result = ResultSuccess();
    ServerCertEntry             *pServerCertEntry = reinterpret_cast<ServerCertEntry *>(pArg);
    CERTCertificate*            pCert = nullptr;
    CERTCertDBHandle            *pCertDb;
    PRUint32                    nicknameLen;
    char                        *nickname = nullptr;

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) started\n");
    do
    {
        if (pServerCertEntry == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) invalid arg\n");
            result = ResultErrorLower();
            break;
        }

        if ((certsCount <= 0) || (pCertItems == nullptr))
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) invalid cert, empty\n");
            result = ResultInvalidCertificate();
            break;
        }

        pCertDb = CERT_GetDefaultCertDB();
        if (pCertDb == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) CERT_GetDefaultCertDB failure\n");
            result = ResultErrorLower();
            break;
        }

        //  Create a temp buffer for nicknames.  It needs to be large enough to
        //  hold the nickname base plus a dash and a hex cert number.
        nicknameLen = PL_strlen(pServerCertEntry->m_NicknameBase) +
                      (sizeof(certsCount) * 2) +
                      2;
        void *pChunk = SslMemoryManager::AllocateChunk(nicknameLen,
                                                       pServerCertEntry->m_CertStore->GetOwnerId());
        nickname = reinterpret_cast<char *>(pChunk);
        if (nickname == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) unable to create nickname scratch mem\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Walk through each of the certificates which were provided by the
        //  app.  Create new temp certificates for them with a new nickname.
        //  This is just for our internal tracking/identification.
        for (int i = 0; i < certsCount; i++)
        {
            //  Create a new temp certificate object, copying the DER data
            //  from the secure item.  This is needed so we do not unnecessarily
            //  leak memory or create a dependency to an item we don't necessarily
            //  own.
            //
            //  NOTE: the cert will be "known" to NSS at this point, but not
            //  considered "trusted" by the trust domain.  We do not want to
            //  change the trust or make the cert permanent as that will make
            //  isolation nearly impossible.
            PR_snprintf(nickname,
                        nicknameLen,
                        "%s-%X",
                        pServerCertEntry->m_NicknameBase,
                        i);
            nickname[nicknameLen - 1] = '\0';
            pCert = CERT_NewTempCertificate(pCertDb,
                                            pCertItems[i],
                                            nickname,
                                            PR_FALSE,
                                            PR_TRUE);
            if (pCert == nullptr)
            {
                REPORT_PR_ERROR_UPON_FAILURE(NN_STATIC_CONDITION(SECFailure),
                                             "CERT_NewTempCertificate");
                result = ResultInvalidCertificate();
                break;
            }

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) create tmp cert %p, requested nickname \'%s\'\n",
                                              pCert,
                                              nickname);
            if (pCert->subjectKeyID.len != 0)
            {
#ifdef NN_DETAIL_SSL_DBG_PRINT_CERTSTORE
                NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(pCert->subjectKeyID.data,
                                                  pCert->subjectKeyID.len);
#endif    //  NN_DETAIL_SSL_DBG_PRINT_CERTSTORE
            }
            else
            {
                //  This should *NOT* happen as NSS should automatically
                //  create the subjectKeyID based on the cert's public key
                //  if it is not already embedded in the cert.
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) cert %p is missing subjectKeyID\n",
                                                  pCert);
                result = ResultInvalidCertificate();
                break;
            }

            //  Insert the new server cert entry into our list.  This will be
            //  used for context specific verification during handshake later.
            result = pServerCertEntry->CertListAdd(pCert);
            if (!result.IsSuccess())
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) failed to add to list (mem)\n");
                break;
            }
        }

        //  If we are successful, update the return value to be SECSuccess
        //  so the import is good.  If we failed in the loop above, we do
        //  not need to revisit the certs which were added to the
        //  ServerCertEntry, they will automatically be destroyed when
        //  that object is deleted.
        if (result.IsSuccess())
        {
            ret = SECSuccess;
        }
    } while (NN_STATIC_CONDITION(false));

    //  Save the import status, this will be used by the caller of the decoder
    pServerCertEntry->m_ImportStatus = result;

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportServerCert) status %d-%d, ret %d\n",
                                      result.GetModule(),
                                      result.GetDescription(),
                                      ret);

    if (nickname != nullptr)
    {
        SslMemoryManager::Free(nickname,
                               pServerCertEntry->m_CertStore->GetOwnerId());
        nickname = nullptr;
    }

    return ret;
}    //  NOLINT(impl/function_size)


nn::Result ServerCertEntry::CertListAdd(CERTCertificate *pCert)
{
    nn::Result                  result = ResultSuccess();
    NssUtil::ObjListNode        *pNode = nullptr;
    void                        *pChunk = nullptr;

    do
    {
        pChunk = SslMemoryManager::AllocateChunk(sizeof(NssUtil::ObjListNode),
                                                 m_CertStore->GetOwnerId());
        if (pChunk == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ServerCertEntry::CertListAddTail] failed to alloc mem for node\n");
            result = ResultInsufficientMemory();
            break;
        }

        pNode = reinterpret_cast<NssUtil::ObjListNode *>(pChunk);
        PR_INIT_CLIST(&pNode->links);
        pNode->pObj = pCert;
        PR_APPEND_LINK(&pNode->links, &m_CertList);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


ServerCertEntry::ServerCertEntry(CertStore *certStore, char *nicknameBase) : m_CertStore(certStore),
                                                                             m_NicknameBase(nicknameBase),
                                                                             m_ImportStatus(ResultSuccess())
{
    //  Init the list used to track certificates. We use NSPR lists
    //  as they are simple, proven C based containers with no exception
    //  support required. Rather than use CERTCertList from NSS, which
    //  requires heap at init, we have methods which do only what we need.
    PR_INIT_CLIST(&m_CertList);
}


ServerCertEntry::~ServerCertEntry()
{
    PRCList                     *pListNode;
    NssUtil::ObjListNode        *pNode;

    //  Walk the list of certificates and destroy them all.  If there are
    //  no other references to the CERTCertificate object, it will be
    //  released back to the heap by NSS.
    for (pListNode = PR_LIST_HEAD(&m_CertList);
         pListNode != &m_CertList;
         pListNode = PR_LIST_HEAD(&m_CertList))
    {
        CERTCertificate         *pCert;

        pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
        pCert = reinterpret_cast<CERTCertificate *>(pNode->pObj);

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[~ServerCertEntry] cert %p purged\n",
                                          pCert);

        CERT_DestroyCertificate(pCert);
        pCert = nullptr;
        PR_REMOVE_LINK(&pNode->links);
        SslMemoryManager::Free(pNode, m_CertStore->GetOwnerId());
        pNode = nullptr;
    }

    //  Release the nickname base, we're done with it
    if (m_NicknameBase != nullptr)
    {
        NssUtil::DestroyRandIdName(m_NicknameBase);
        m_NicknameBase = nullptr;
    }
}


CertStore* ServerCertEntry::GetCertStore()
{
    return m_CertStore;
}


nn::Result ServerCertEntry::GetImportStatus()
{
    return m_ImportStatus;
}


bool ServerCertEntry::IsServerCertTrusted(CERTCertificate *pCert)
{
    bool                        ret = false;

    NN_DETAIL_SSL_DBG_PRINT("[ServerCertEntry::IsServerCertTrusted] check cert w/subject key ID:\n");
    NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(pCert->subjectKeyID.data,
                                      pCert->subjectKeyID.len);

    do
    {
        for (PRCList *pListNode = PR_LIST_HEAD(&m_CertList);
             pListNode != &m_CertList;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            CERTCertificate         *pCurCert;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurCert = reinterpret_cast<CERTCertificate *>(pNode->pObj);

            NN_DETAIL_SSL_DBG_PRINT("[ServerCertEntry::IsServerCertTrusted] imported curCert subject key ID:\n");
            NN_DETAIL_SSL_DBG_PRINT_HEX_TABLE(pCurCert->subjectKeyID.data,
                                              pCurCert->subjectKeyID.len);
            if (SECITEM_ItemsAreEqual(&pCurCert->subjectKeyID, &pCert->subjectKeyID))
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ServerCertEntry::IsServerCertTrusted] matched cert %p, nickname \'%s\'\n",
                                                  pCurCert,
                                                  (pCurCert->nickname != nullptr) ? pCurCert->nickname : "{null}");
                ret = true;
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


nn::Result ServerCertEntry::GetTrustedCertList(CERTCertList *list)
{
    nn::Result                  ret = ResultSuccess();
    CERTCertificate             *pDupCert = nullptr;

    do
    {
        if (list == nullptr)
        {
            ret = ResultInvalidPointer();
            break;
        }

        if (PR_CLIST_IS_EMPTY(&m_CertList))
        {
            //  No installed certs
            break;
        }

        for (PRCList *pListNode = PR_LIST_HEAD(&m_CertList);
             pListNode != &m_CertList;
             pListNode = PR_NEXT_LINK(pListNode))
        {
            NssUtil::ObjListNode    *pNode;
            CERTCertificate         *pCurCert;

            pNode = reinterpret_cast<NssUtil::ObjListNode *>(pListNode);
            pCurCert = reinterpret_cast<CERTCertificate *>(pNode->pObj);

            if (pCurCert->isRoot != PR_TRUE)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[GetTrustedCertList] skipping cert %p, not a root\n");
                continue;
            }

            pDupCert = CERT_DupCertificate(pCurCert);
            if (pDupCert == nullptr)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[GetTrustedCertList] error duping cert %p\n", pCurCert);
                ret = ResultInsufficientMemory();
                break;
            }

            if (CERT_AddCertToListTail(list, pDupCert) != SECSuccess)
            {
                //  The only reason this should fail is due to memory alloc
                //  failures when adding the cert to the list.
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[GetTrustedCertList] failed to add cert %p, \'%s\'\n",
                                                  pDupCert,
                                                  (pDupCert->nickname != nullptr) ? pDupCert->nickname : "{null}");
                ret = ResultInsufficientMemory();
                break;
            }

            //  The list "owns" the cert now, null it so it is not destroyed
            //  in the cleanup below if the loop is done or breaks out in
            //  the next iteration.
            pDupCert = nullptr;
        }
    } while (NN_STATIC_CONDITION(false));

    if (pDupCert != nullptr)
    {
        CERT_DestroyCertificate(pDupCert);
        pDupCert = nullptr;
    }

    return ret;
}


// ------------------------------------------------------------------------------------------------
//
// ClientCertEntry
//
// ------------------------------------------------------------------------------------------------
const char                      *ClientCertEntry::g_NicknameBase = "_nxa_cc";

ClientCertEntry::ClientCertEntry(CERTCertificate   *pCert,
                                 SECKEYPrivateKey  *pPrivKey) :
                                     m_pCert(pCert),
                                     m_pPrivKey(pPrivKey)
{
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[ClientCertEntry] created %p, cert %p, key %p\n",
                                      this,
                                      m_pCert,
                                      m_pPrivKey);
}


ClientCertEntry::~ClientCertEntry()
{
    nn::Result                  result;

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[~ClientCertEntry] destroy %p\n",
                                      this);

    //  "Destroy" our records of the cert and private key.  This may result in
    //  the object being completely cleaned up out of RAM.  This is managed
    //  within NSS.
    CERT_DestroyCertificate(m_pCert);
    SECKEY_DestroyPrivateKey(m_pPrivKey);
    m_pCert    = nullptr;
    m_pPrivKey = nullptr;
}


CERTCertificate* ClientCertEntry::GetCert()
{
    return m_pCert;
}


SECKEYPrivateKey* ClientCertEntry::GetKey()
{
    return m_pPrivKey;
}


uint32_t ClientCertEntry::GetMaxNameSize()
{
    return (sizeof(char) * (g_NicknameBaseLen + 1 + Util::g_NumCharsUint64Hex + 1));
}


// ------------------------------------------------------------------------------------------------
//
// CrlEntry
//
// ------------------------------------------------------------------------------------------------
CrlEntry::CrlEntry(CERTSignedCrl *pCrl) : m_pCrl(pCrl)
{
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry] created %p with CRL %p\n", this, pCrl);
}


CrlEntry::~CrlEntry()
{
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[~CrlEntry] destroy %p\n", this);
    CERT_DestroyCrl(m_pCrl);
    m_pCrl = nullptr;
}


bool CrlEntry::IsCertRevoked(CERTCRLEntryReasonCode  *pOutReason,
                             CERTCertificate         *pInCert)
{
    bool                        revoked = false;
    CERTCRLEntryReasonCode      reason = crlEntryReasonUnspecified;
    CERTName                    *pCertIssuer;
    CERTName                    *pCrlIssuer;
    SECComparison               issuerComp;
    SECItem                     *pCertSn;

    do
    {
        //  Get the cert issuer
        pCertIssuer = &pInCert->issuer;

        //  Get our issuer, compare to the cert's
        //  TODO: Check CRL extensions for "issuerAltNames"?
        pCrlIssuer = &m_pCrl->crl.name;
        issuerComp = CERT_CompareName(pCrlIssuer, pCertIssuer);
        if (issuerComp != SECEqual)
        {
            //  No match, so as far as this CRL is concerned the cert
            //  is unrevoked.
            break;
        }

        //  Get the certificate serial number, we need to check it since
        //  the issuer matches.
        pCertSn = &pInCert->serialNumber;

        //  See if this CRL contains the serial number for this cert.  If it
        //  does then get its revocation status and update the return.
        for (int i = 0; m_pCrl->crl.entries[i] != nullptr; i++)
        {
            SECComparison           snMatch;
            CERTCrlEntry            *pCurEntry = m_pCrl->crl.entries[i];
            SECStatus               secStatus = SECSuccess;

            snMatch = SECITEM_CompareItem(pCertSn, &pCurEntry->serialNumber);
            if (snMatch != SECEqual)
            {
                continue;
            }

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsCertRevoked] crl %p issuer and sn match for cert %p\n",
                                              m_pCrl,
                                              pInCert);

            secStatus = CERT_FindCRLEntryReasonExten(pCurEntry, &reason);
            if (secStatus != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsCertRevoked] crl %p no reason found, treat as UNSPECIFIED\n");
                reason = crlEntryReasonUnspecified;
                revoked = true;
                break;
            }

            //  Verify the encoded reason is valid.  Unfortunately, the value of 7
            //  is a "magic number" in this case as it is undefined so treat it
            //  as UNREVOKED.
            if ((reason < crlEntryReasonUnspecified) ||
                (reason > crlEntryReasonAaCompromise))
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsCertRevoked] crl %p contained bad reason code, assume UNSPECIFIED\n");
                reason = crlEntryReasonUnspecified;
                revoked = true;
                break;
            }

            if (reason == 7)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::GetCertRevokeStatus] crl %p contained bad reason code (7), assume UNREVOKED\n");
                continue;
            }

            //  We do have a special case to handle here: a revocation status
            //  of "removed from CRL", meaning that a cert previously in a HOLD
            //  state or one which has expired has been removed.  If we encounter
            //  this, just treat it like unrevoked.
            if (reason == crlEntryReasonRemoveFromCRL)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsCertRevoked] crl %p has cert %p as REMOVED FROM CRL, treat as UNREVOKED\n");
                break;
            }

            //  If we get here, then the reason extension was provided and contains
            //  a valid reason which means the cert was revoked.  Set the revoked
            //  boolean to true and break out, we're done looking for a match.
            revoked = true;
            break;
        }

        if (revoked)
        {
            *pOutReason = reason;
        }
    } while (NN_STATIC_CONDITION(false));

    return revoked;
}


bool CrlEntry::IsAnyCertInChainRevoked(CERTCRLEntryReasonCode  *pOutReason,
                                       CERTCertificate         **pOutRevokedCert,
                                       const CERTCertList      *pInTrustChain)
{
    bool                        revoked = false;
    CERTCRLEntryReasonCode      reason = crlEntryReasonUnspecified;
    CERTCertificate             *pCurCert = nullptr;

    do
    {
        //  The trust chain contains the cert to be checked (HEAD) down to the
        //  trust anchor (TAIL).
        for (CERTCertListNode *pCurNode = CERT_LIST_HEAD(pInTrustChain);
             (revoked == false) && !CERT_LIST_END(pCurNode, pInTrustChain);
             pCurNode = CERT_LIST_NEXT(pCurNode))
        {
            pCurCert = pCurNode->cert;

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsAnyCertInChainRevoked] %p, check cert %p\n",
                                              this,
                                              pCurCert);
            revoked  = IsCertRevoked(&reason, pCurCert);
        }
    } while (NN_STATIC_CONDITION(false));

    if (revoked)
    {
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("[CrlEntry::IsAnyCertInChainRevoked] cert %p on CRL, reason %d\n",
                                          pCurCert,
                                          reason);
        *pOutReason      = reason;
        *pOutRevokedCert = pCurCert;
    }

    return revoked;
}

}}}
