﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_MemoryManager.h"
#include "server/ssl_Util.h"

#include "ssl_DebugUtil.h"
#include "ssl_DebugSessionCache.h"
#include "ssl_DebugShell.h"

namespace nn { namespace ssl { namespace detail {

// ------------------------------------------------------------------------------------------------
// Global objects
// ------------------------------------------------------------------------------------------------
DebugUtil g_DebugUtil;

// ------------------------------------------------------------------------------------------------
// DebugUtil::Worker
// ------------------------------------------------------------------------------------------------
void DebugUtil::Worker::Executer(void* arg)
{
    DebugUtil*            pDebugUtil = reinterpret_cast<DebugUtil*>(arg);
    Worker*               pWorker    = pDebugUtil->GetWorker();
    nn::os::TimerEvent*   pTimer     = pWorker->GetTimerEvent();
    nn::os::MessageQueue* pTaskQueue = pWorker->GetMessageQueue();

    do
    {
        pTimer->StartOneShot(nn::TimeSpan::FromMilliSeconds(pWorker->GetMsecInterval()));
        pTimer->Wait();
        pTimer->Clear();

        // Run schedulers
        pDebugUtil->GetMemoryTracker()->RunScheduler();
        pDebugUtil->GetSessionCacheDebugger()->RunScheduler();

        // Process tasks
        uintptr_t curTask;
        while (pTaskQueue->TryReceive(&curTask) == true)
        {
            switch (static_cast<TaskType>(curTask))
            {
            case TaskType_UpdateHeapStat:
                {
                    pDebugUtil->GetMemoryTracker()->UpdateHeapStats();
                }
                break;
            case TaskType_DumpHeapStat:
                {
                    pDebugUtil->GetMemoryTracker()->DumpHeapStats();
                }
                break;
            case TaskType_DumpSessionCache:
                {
                    pDebugUtil->GetSessionCacheDebugger()->Dump();
                }
                break;
            default:
                NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Worker::Executer] Unkown task type\n");
                break;
            }
        }
    } while (pWorker->IsReadyToCleanup() == false);

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Worker::Executer] Done\n");
}

DebugUtil::Worker::Worker() :
    m_MsecTimerInterval(0),
    m_TimerEvent(nn::os::EventClearMode_ManualClear),
    m_TaskMessageQueue(m_MessageQueueBuf, MessageQueueSize),
    m_IsInitialized(false),
    m_IsReadyToCleanup(false)
{
}

DebugUtil::Worker::~Worker()
{
}

bool DebugUtil::Worker::IsReadyToCleanup()
{
    return m_IsReadyToCleanup;
}

nn::os::TimerEvent* DebugUtil::Worker::GetTimerEvent()
{
    return &m_TimerEvent;
}

uint32_t DebugUtil::Worker::GetMsecInterval()
{
    return m_MsecTimerInterval;
}

nn::os::MessageQueue* DebugUtil::Worker::GetMessageQueue()
{
    return &m_TaskMessageQueue;
}

void DebugUtil::Worker::AddTask(TaskType taskType)
{
    if (m_TaskMessageQueue.TrySend(static_cast<uintptr_t>(taskType)) == false)
    {
        NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Worker::AddTask] Task queue is full\n");
    }
}

nn::Result DebugUtil::Worker::Initialize(DebugUtil* pInUtilObj)
{
    nn::Result result;

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Worker::Initialize] Start\n");
    do
    {
        result = nn::os::CreateThread(
            &m_Thread,
            Executer,
            pInUtilObj,
            m_ThreadStack,
            DebugUtil::Worker::ThreadStackSize,
            nn::os::LowestThreadPriority);
        if (result.IsFailure())
        {
            break;
        }

        m_MsecTimerInterval = WorkerMsecInterval;

        nn::os::StartThread(&m_Thread);

        m_IsInitialized = true;
    } while (NN_STATIC_CONDITION(false));
    NN_DETAIL_SSL_DBG_UTIL_PRINT("[ DebugUtil::Worker::Initialize] End\n");

    return result;
}

nn::Result DebugUtil::Worker::Finalize()
{
    NN_DETAIL_SSL_DBG_UTIL_PRINT("[ DebugUtil::Worker::Finalize] Start\n");

    if (m_IsInitialized == true)
    {
        m_IsReadyToCleanup = true;
        m_TimerEvent.Signal();
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
    }

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[ DebugUtil::Worker::Finalize] End\n");

    return nn::ResultSuccess();
}

// ------------------------------------------------------------------------------------------------
// DebugUtil::MemoryTracker
// ------------------------------------------------------------------------------------------------
DebugUtil::MemoryTracker::MemoryTracker() :
    m_IsDumpStats(true),
    m_pDebugUtilObj(nullptr),
    m_pOneHeapTrackStatHead(nullptr),
    m_OneHeapTrackStatLock(false),
    m_pStdAllocator(nullptr)
{
}

DebugUtil::MemoryTracker::~MemoryTracker()
{
}

nn::Result DebugUtil::MemoryTracker::Initialize(
    DebugUtil* pInUtilObj,
    nn::mem::StandardAllocator* pInStdAllocator)
{
    memset(&m_HeapStats, 0x00, sizeof(m_HeapStats));
    m_HeapStats.min = SIZE_MAX;
    m_pDebugUtilObj = pInUtilObj;
    m_pStdAllocator = pInStdAllocator;

    return nn::ResultSuccess();
}

nn::Result DebugUtil::MemoryTracker::Finalize()
{
    return nn::ResultSuccess();
}

void DebugUtil::MemoryTracker::ConfigurePeriodicDump(bool enable)
{
    m_IsDumpStats = enable;
}

void DebugUtil::MemoryTracker::DumpHeapStats()
{
    size_t curHeapSpace = m_pStdAllocator->GetTotalFreeSize();
    NN_DETAIL_SSL_DBG_UTIL_PRINT("[SSL] Heap Stats - curr: %d (%dKB) max:%d (%dKB) min:%d (%dKB)\n",
        curHeapSpace, curHeapSpace / 1024,
        m_HeapStats.max, m_HeapStats.max / 1024,
        m_HeapStats.min, m_HeapStats.min / 1024);
}

void DebugUtil::MemoryTracker::GetHeapStats(
    size_t *pOutCurSize,
    size_t *pOutMaxSize,
    size_t *pOutMinSize)
{
    *pOutCurSize = m_pStdAllocator->GetTotalFreeSize();
    *pOutMaxSize = m_HeapStats.max;
    *pOutMinSize = m_HeapStats.min;
}

void DebugUtil::MemoryTracker::UpdateHeapStats()
{
    size_t curHeapSpace = m_pStdAllocator->GetTotalFreeSize();

    if (curHeapSpace > m_HeapStats.max)
    {
        m_HeapStats.max = curHeapSpace;
    }
    else if (curHeapSpace < m_HeapStats.min)
    {
        m_HeapStats.min = curHeapSpace;
    }
}

void DebugUtil::MemoryTracker::RunScheduler()
{
    nn::os::Tick curTick = nn::os::GetSystemTick();
    uint64_t elapsedMsec = 0;

    elapsedMsec = nn::os::ConvertToTimeSpan(curTick - m_HeapStats.lastUpdateTick).GetMilliSeconds();
    if (elapsedMsec > StatsUpdateMsecInterval)
    {
        m_pDebugUtilObj->GetWorker()->AddTask(DebugUtil::TaskType_UpdateHeapStat);
        m_HeapStats.lastUpdateTick = curTick;
    }

    if (m_IsDumpStats)
    {
        elapsedMsec = nn::os::ConvertToTimeSpan(curTick - m_HeapStats.lastDumpTick).GetMilliSeconds();
        if (elapsedMsec > StatsDumpMsecInterval)
        {
            m_pDebugUtilObj->GetWorker()->AddTask(DebugUtil::TaskType_DumpHeapStat);
            m_HeapStats.lastDumpTick = curTick;
        }
    }
}

size_t DebugUtil::MemoryTracker::GetHash(const char* pInTrackName)
{
    const size_t coeff1 = 179;
    const size_t coeff2 = 9236923;
    size_t       hash  = 3;
    const char*  pCur  = pInTrackName;

    while (*pCur != '\0')
    {
        hash += (hash * coeff1 * static_cast<size_t>(*pCur)) % coeff2;
        pCur++;
    }

    return hash;
}

DebugUtil::MemoryTracker::OneHeapTrackStat* DebugUtil::MemoryTracker::FindOneTracker(
    const char* pInTrackName)
{
    OneHeapTrackStat* pTracker = nullptr;
    size_t            hash = this->GetHash(pInTrackName);

    m_OneHeapTrackStatLock.Lock();

    OneHeapTrackStat* pCur = m_pOneHeapTrackStatHead;
    do
    {
        if (pCur == nullptr)
        {
            break;
        }

        while (pCur != nullptr)
        {
            if (pCur->hash == hash)
            {
                pTracker = pCur;
                break;
            }
            pCur = pCur->pNext;
        }
    } while (NN_STATIC_CONDITION(false));
    m_OneHeapTrackStatLock.Unlock();

    return pTracker;
}

void DebugUtil::MemoryTracker::AddOneTracker(OneHeapTrackStat* pInTracker)
{
    m_OneHeapTrackStatLock.Lock();

    if (m_pOneHeapTrackStatHead == nullptr)
    {
        m_pOneHeapTrackStatHead = pInTracker;
    }
    else
    {
        OneHeapTrackStat* pCur = m_pOneHeapTrackStatHead;
        while (pCur->pNext != nullptr)
        {
            pCur = pCur->pNext;
        }
        pCur->pNext = pInTracker;
    }

    m_OneHeapTrackStatLock.Unlock();
}

void DebugUtil::MemoryTracker::RemoveOneTracker(OneHeapTrackStat* pInTracker)
{
    m_OneHeapTrackStatLock.Lock();

    OneHeapTrackStat* pCur = m_pOneHeapTrackStatHead;
    if (pCur == nullptr)
    {
        return;
    }

    OneHeapTrackStat* pPrev = nullptr;
    do
    {
        if (pCur == pInTracker)
        {
            if (pPrev != nullptr)
            {
                pPrev->pNext = pCur->pNext;
            }
            else
            {
                m_pOneHeapTrackStatHead = nullptr;
            }

            free(pCur);
            break;
        }

        pPrev = pCur;
        pCur  = pCur->pNext;
    } while (pCur != nullptr);

    m_OneHeapTrackStatLock.Unlock();
}

void DebugUtil::MemoryTracker::StartOneTrack(const char* pInTrackName)
{
    OneHeapTrackStat* pOneTracker = reinterpret_cast<OneHeapTrackStat*>(malloc(sizeof(OneHeapTrackStat)));

    pOneTracker->startTick  = nn::os::GetSystemTick();
    pOneTracker->startSpace = m_pStdAllocator->GetTotalFreeSize();
    pOneTracker->hash       = this->GetHash(pInTrackName);

    this->AddOneTracker(pOneTracker);
}

void DebugUtil::MemoryTracker::EndOneTrack(const char* pInTrackName)
{
    do
    {
        OneHeapTrackStat* pTracker = FindOneTracker(pInTrackName);
        if (pTracker == nullptr)
        {
            NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::MemoryTracker::EndOneTrack] Tracker not found - name:%s\n",
                pInTrackName);
            break;
        }

        uint64_t elapsedMsec   = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - pTracker->startTick).GetMilliSeconds();
        size_t   curHeapSpace  = m_pStdAllocator->GetTotalFreeSize();
        size_t   delta         = (curHeapSpace > pTracker->startSpace)?(curHeapSpace - pTracker->startSpace):(pTracker->startSpace - curHeapSpace);
        NN_DETAIL_SSL_DBG_UTIL_PRINT("--------------------------------------------------------------------\n");
        NN_DETAIL_SSL_DBG_UTIL_PRINT(" [SSL] cur:%d(%dKB) before:%d(%dKB) delta:(%s)%d(%dKB) elapsed:%dmsec name:%s\n",
            curHeapSpace, curHeapSpace / 1024,
            pTracker->startSpace, pTracker->startSpace / 1024,
            (curHeapSpace > pTracker->startSpace)?"+":"-", delta, delta / 1024,
            elapsedMsec,
            pInTrackName);
        NN_DETAIL_SSL_DBG_UTIL_PRINT("--------------------------------------------------------------------\n");

        this->RemoveOneTracker(pTracker);
    } while (NN_STATIC_CONDITION(false));
}

bool DebugUtil::MemoryTracker::EndOneTrack(TrackStats* pOutStats, const char* pInTrackName)
{
    bool isFound = true;

    do
    {
        OneHeapTrackStat* pTracker = FindOneTracker(pInTrackName);
        if (pTracker == nullptr)
        {
            isFound = false;
            break;
        }

        pOutStats->elapsedMsec = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - pTracker->startTick).GetMilliSeconds();
        pOutStats->curHeapSpace = m_pStdAllocator->GetTotalFreeSize();
        pOutStats->delta = pOutStats->curHeapSpace - pTracker->startSpace;

        this->RemoveOneTracker(pTracker);
    } while (NN_STATIC_CONDITION(false));

    return isFound;
}

// ------------------------------------------------------------------------------------------------
// DebugUtil
// ------------------------------------------------------------------------------------------------
void DebugUtil::Initialize(nn::mem::StandardAllocator* pInStdAllocator)
{
    if (m_MemoryTracker.Initialize(this, pInStdAllocator).IsFailure())
    {
        NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Initialize] Failed to initialize the memory tracker\n");
    }
    m_MemoryTracker.UpdateHeapStats(); // Update heap stats to get the initial size

    if (m_Worker.Initialize(this).IsFailure())
    {
        NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugUtil::Initialize] Failed to initialize the worker\n");
    }

    m_SessionCacheDebugger.Initialize(this);

    g_DebugShellManager.Initialize();
}

void DebugUtil::Finalize()
{
    g_DebugShellManager.Finalize();

    m_MemoryTracker.DumpHeapStats();
    m_SessionCacheDebugger.Dump();

    m_SessionCacheDebugger.Finalize();
    m_MemoryTracker.Finalize();
    m_Worker.Finalize();
}

DebugUtil::Worker* DebugUtil::GetWorker()
{
    return &m_Worker;
}

DebugUtil::MemoryTracker* DebugUtil::GetMemoryTracker()
{
    return &m_MemoryTracker;
}

DebugSessionCache* DebugUtil::GetSessionCacheDebugger()
{
    return &m_SessionCacheDebugger;
}

void DebugUtil::GetHeapStats(size_t *pOutCurSize, size_t* pOutMaxSize, size_t* pOutMinSize)
{
    m_MemoryTracker.GetHeapStats(pOutCurSize, pOutMaxSize, pOutMinSize);
}

void DebugUtil::ConfigurePeriodicDump(bool enable)
{
    m_MemoryTracker.ConfigurePeriodicDump(enable);
}

void DebugUtil::StartHeapTrack(const char* pInTrackName)
{
    m_MemoryTracker.StartOneTrack(pInTrackName);
}

bool DebugUtil::EndHeapTrack(
    uint64_t*   pOutElapsedMsec,
    int64_t*    pOutDelta,
    size_t*     pOutCurHeapSpace,
    const char* pInTrackName)
{
    bool                      ret = false;
    MemoryTracker::TrackStats stats;

    ret = m_MemoryTracker.EndOneTrack(&stats, pInTrackName);
    if (ret == true)
    {
        *pOutElapsedMsec  = stats.elapsedMsec;
        *pOutDelta        = stats.delta;
        *pOutCurHeapSpace = stats.curHeapSpace;
    }

    return ret;
}

// ------------------------------------------------------------------------------------------------
// DebugUtil::ScopedHeapTracker
// ------------------------------------------------------------------------------------------------
DebugUtil::ScopedHeapTracker::ScopedHeapTracker(DebugUtil* pInDebugUtil, const char* pInName)
{
    m_pDebugUtil = pInDebugUtil;
    m_pName      = pInName;

    m_pDebugUtil->GetMemoryTracker()->StartOneTrack(m_pName);
}

DebugUtil::ScopedHeapTracker::~ScopedHeapTracker()
{
    m_pDebugUtil->GetMemoryTracker()->EndOneTrack(m_pName);
}

}}} // namespace nn { namespace ssl { namespace detail {
