﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/socket.h>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_Util.h"
#include "ssl_DebugUtil.h"
#include "ssl_DebugShell.h"

extern "C" {
#if NNSDK_ENABLE_MTRACK
    #include "pr/src/md/nnsdk/mtrack.h"
#endif

extern void SSL_ClearSessionCache(void);
}

namespace {
    static const char AsciiSpace               = 0x20;    // " "
    static const char AsciiLineFeed            = 0x0a;    // LF
    static const char AsciiCarriageReturn      = 0x0d;    // CR
    static const int  MaxCommandCount          = 10;      // The number of max commands (including options)
    static const int  MaxCommandLength         = 32;      // The length of the each command/option
    static const int  CommandDescriptionLength = 384;     // The string length of the command description
    static const int  CommandBufferSize        = 256 + 1; // The buffer size to receive commands

    // How to add new command
    //   1. Add new command index in CommandDeinitionIndex enum
    //      Note that it needs to be added before CommandDeinitionIndex_MaxCount
    //   2. Add new command string and its description in CommandDefinition[][][]
    //        CommandDefinition[][0]: Command string
    //        CommandDefinition[][1]: Description of the command
    //      Note that the order of the command in CommandDeinitionIndex and CommandDefinition needs to be same.
    //   3. Add actual command handling in HandleCommand()
    enum CommandDefinitionEntity {
        CommandDefinitionEntity_Command = 0,
        CommandDefinitionEntity_Description,
    };

    enum CommandDeinitionIndex {
        CommandDeinitionIndex_Help = 0,
        CommandDeinitionIndex_Mtrack,
        CommandDeinitionIndex_Sid,
        CommandDeinitionIndex_MaxCount
    };

    static const char CommandDefinition[][MaxCommandLength][CommandDescriptionLength] {
        {"help","Help command"},
        {"mtrack","Memory tracking dump\n"   "  full : Dump all, optional threahhold can be passed (e.g. mtrack full 2048 : prints allocation larger than 2KB)"
                                             "  delta : Dump only delta, optional threahhold can be passed (e.g. mtrack delta 2048 : prints allocation larger than 2KB)\n"
                                             "  index #:Dump specific mtrack (e.g. mtrack index 100)"},
        {"sid", "Manipulate session cache\n" "  flush : Flush all session cache\n"
                                             "  dump : Dump existing session cache\n"
                                             "  periodic : Turn on/off periodic cache dump (e.g. sid periodic off)"},
    };

    int ParseCommand(char pOutCommandArray[MaxCommandCount][MaxCommandLength], const char* pInCommands)
    {
        const char* pCurr        = nullptr;
        int         commandCount = 0;

        int currentOffset = 0;
        for (pCurr = pInCommands; (*pCurr != '\0') && (commandCount < MaxCommandCount); pCurr++)
        {
            if ((*pCurr == AsciiLineFeed) || (*pCurr == AsciiCarriageReturn))
            {
                continue;
            }

            if (*pCurr == AsciiSpace)
            {
                commandCount++;
                currentOffset = 0;
                continue;
            }

            pOutCommandArray[commandCount][currentOffset] = *pCurr;
            currentOffset++;
        }

        return commandCount + 1;
    }

    void HandleCommand(char commands[][MaxCommandLength], int commandCount)
    {
        int commandIndex = -1;
        for (int i = 0; i < CommandDeinitionIndex_MaxCount; i++)
        {
            if (strncmp(commands[0],
                        CommandDefinition[i][CommandDefinitionEntity_Command],
                        sizeof(CommandDefinition[i][CommandDefinitionEntity_Command])) == 0)
            {
                commandIndex = i;
                break;
            }
        }

        switch (commandIndex)
        {
        // Command for MTRACK
        // [The format of supported sub commands]
        // - mtrack full #
        // - mtrack delta #
        // - mtrack index #
        // Details of each command is found in CommandDefinition[][][]
        case CommandDeinitionIndex_Mtrack:
            {
#if NNSDK_ENABLE_MTRACK
                if (commandCount <= 1)
                {
                    NN_SDK_LOG("Command is missing (full, delta or index #)\n");
                    break;
                }
                if (strncmp("full", commands[1], sizeof(commands[1])) == 0)
                {
                    int threshold =  (commandCount > 2)?atoi(commands[2]):0;
                    mtrackDumpDeltaOrFullThreshhold(false, threshold);
                }
                else if (strncmp("delta", commands[1], sizeof(commands[1])) == 0)
                {
                    int threshold =  (commandCount > 2)?atoi(commands[2]):0;
                    mtrackDumpDeltaOrFullThreshhold(true, threshold);
                }
                else if (strncmp("index", commands[1], sizeof(commands[1])) == 0)
                {
                    if (commandCount <= 2)
                    {
                        NN_SDK_LOG("Index number is missing (e.g. mtrack index 100)\n");
                        break;
                    }
                    mtrackDumpIndex(atoi(commands[2]));
                }
                else
                {
                    NN_SDK_LOG("Received unknown command for mtrack (%s)\n", commands[1]);
                }
#else
                NN_SDK_LOG("MTRACK is not built in right now, rebuild the SSL process with ENABLE_MTRACK=true.\n");
#endif
            }
            break;
        // Command for session cache
        // [The format of supported sub commands]
        // - sid flush
        // - sid dump
        // - sid periodic [on|off]
        // Details of each command is found in CommandDefinition[][][]
        case CommandDeinitionIndex_Sid:
            {
                if (commandCount <= 1)
                {
                    NN_SDK_LOG("Command is missing (full, delta or index #)\n");
                    break;
                }
                if (strncmp("flush", commands[1], sizeof(commands[1])) == 0)
                {
                    SSL_ClearSessionCache();
                    NN_SDK_LOG("All session cache data got flushed.\n");
                }
                else if (strncmp("dump", commands[1], sizeof(commands[1])) == 0)
                {
                    nn::ssl::detail::g_DebugUtil.GetSessionCacheDebugger()->Dump();
                    NN_SDK_LOG("SID dump done.\n");
                }
                else if (strncmp("periodic", commands[1], sizeof(commands[1])) == 0)
                {
                    if (commandCount <= 2)
                    {
                        NN_SDK_LOG("Index number is missing (e.g. mtrack index 100)\n");
                        break;
                    }
                    if (strncmp("on", commands[2], sizeof(commands[2])) == 0)
                    {
                        nn::ssl::detail::g_DebugUtil.GetSessionCacheDebugger()->ConfigurePeriodicDump(true);
                        NN_SDK_LOG("Turned on periodic sid dump.\n");
                    }
                    else if (strncmp("off", commands[2], sizeof(commands[2])) == 0)
                    {
                        nn::ssl::detail::g_DebugUtil.GetSessionCacheDebugger()->ConfigurePeriodicDump(false);
                        NN_SDK_LOG("Turned off periodic sid dump.\n");
                    }
                    else
                    {
                        NN_SDK_LOG("Received unknown command for mtrack periodic (%s)\n", commands[2]);
                    }
                }
                else
                {
                    NN_SDK_LOG("Received unknown command for mtrack (%s)\n", commands[1]);
                }
            }
            break;
        // Command for help and non-supported commands
        case CommandDeinitionIndex_Help:
        default:
            {
                if (commandIndex != CommandDeinitionIndex_Help)
                {
                    NN_SDK_LOG("Received unsupported command (%s)\n", commands[0]);
                }
                NN_SDK_LOG("[SSL Debug Shell Command] ------------------------------------\n");
                for (int i = 0; i < CommandDeinitionIndex_MaxCount; i++)
                {
                    NN_SDK_LOG(" %s : %s\n",
                        CommandDefinition[i][CommandDefinitionEntity_Command],
                        CommandDefinition[i][CommandDefinitionEntity_Description]);
                }
                NN_SDK_LOG("--------------------------------------------------------------\n");
                }
            break;
        }
    } // NOLINT(impl/function_size)
} // Un-named namespace

namespace nn { namespace ssl { namespace detail {

DebugShell g_DebugShellManager;

// ------------------------------------------------------------------------------------------------
// DebugShell::Executer
// ------------------------------------------------------------------------------------------------
void DebugShell::Executer(void* arg) NN_NOEXCEPT
{
    int                sockFd    = -1;
    int                recvdSize = 0;
    struct sockaddr_in srvAddr;

    // This sleep is needed because the SSL debug feature is initialized even before
    // the SSL process initializes the socket library.
    // It is possible to deal with it with additional signaling to wait for the socket
    // library gets initialized. But since this is the code needed only for debugging
    // purpose, simply adding a wait here is better to leave the main code simpler.
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Executer] Start\n");
    do {
        while (NN_STATIC_CONDITION(true))
        {
            sockFd = nn::socket::Socket(AF_INET, SOCK_DGRAM, 0);
            if (sockFd < 0)
            {
                NN_SDK_LOG("[DebugShell::Initialize] Failed to create a socket.\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                break;
            }

            srvAddr.sin_family      = AF_INET;
            srvAddr.sin_port        = nn::socket::InetHtons(DebugShell::ServerPort);
            srvAddr.sin_addr.s_addr = nn::socket::InetHtonl(INADDR_ANY);
            if (nn::socket::Bind(sockFd, (struct sockaddr*)&srvAddr, sizeof(srvAddr)) < 0)
            {
                NN_SDK_LOG("[DebugShell::Initialize] Failed to bind a socket.\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                break;
            }

            NN_SDK_LOG("----------------------------------------------------------\n");
            NN_SDK_LOG(" SSL Debug Shell waiting on port :%d\n", DebugShell::ServerPort);
            NN_SDK_LOG("----------------------------------------------------------\n");

            while (NN_STATIC_CONDITION(true))
            {
                char recvBuffer[CommandBufferSize]               = {0};
                int  commandCount                                = 0;
                char commands[MaxCommandCount][MaxCommandLength] = {{0},{0}};

                recvdSize = nn::socket::RecvFrom(sockFd, recvBuffer, sizeof(recvBuffer) - 1, 0, nullptr, 0);
                if (recvdSize < 0)
                {
                    NN_SDK_LOG("[DebugShell::Executer] RecvFrom failed\n");
                    break;
                }
                else if (recvdSize == CommandBufferSize)
                {
                    NN_SDK_LOG("[DebugShell::Executer] Received commands are too long (max=%d bytes)\n",
                        recvdSize, CommandBufferSize);
                    continue;
                }
                else if (recvdSize == 0)
                {
                    continue;
                }

                recvBuffer[recvdSize] = 0x0;
                commandCount = ParseCommand(commands, recvBuffer);
                if (commandCount > MaxCommandCount)
                {
                    NN_SDK_LOG("[DebugShell::Executer] Received more than max commands (%d commands, max=%d)\n",
                        commandCount, MaxCommandCount);
                    continue;
                }
                HandleCommand(commands, commandCount);
            }

            if (sockFd >= 0)
            {
                nn::socket::Close(sockFd);
            }
        }
    } while (NN_STATIC_CONDITION(true));

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Executer] Done\n");
}

DebugShell::DebugShell() NN_NOEXCEPT
{
}

DebugShell::~DebugShell() NN_NOEXCEPT
{
}

bool DebugShell::Initialize() NN_NOEXCEPT
{
    nn::Result         result;
    bool               bResult = true;

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Initialize] Start\n");
    do {
        result = nn::os::CreateThread(
            &m_Thread,
            Executer,
            this,
            m_ThreadStack,
            DebugShell::ThreadStackSize,
            nn::os::LowestThreadPriority);
        if (result.IsFailure())
        {
            NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Initialize] Failed to create a thread.\n");
            bResult = false;
            break;
        }

        nn::os::StartThread(&m_Thread);
    } while (NN_STATIC_CONDITION(false));
    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Initialize] End\n");

    return bResult;
}

void DebugShell::Finalize() NN_NOEXCEPT
{
    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Finalize] Start\n");

    NN_DETAIL_SSL_DBG_UTIL_PRINT("[DebugShell::Finalize] End\n");
}

}}} // namespace nn { namespace ssl { namespace detail {
