﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/srepo_SystemReport.h>
#include <nn/srepo/srepo_Result.h>
#include <nn/srepo/detail/srepo_ApiDetail.h>
#include <nn/srepo/detail/srepo_SystemReportGenerator.h>
#include <nn/srepo/detail/srepo_ShimLibraryGlobal.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace srepo {

const size_t SystemReport::BufferSizeMin;
const size_t SystemReport::BufferSizeMax;

SystemReport::SystemReport() NN_NOEXCEPT :
    m_ApplicationId(ApplicationId::GetInvalidId()),
    m_Category(ReportCategory_Defualt),
    m_Buffer(nullptr),
    m_Size(0),
    m_Position(0)
{
    m_EventId[0] = '\0';
}

SystemReport::SystemReport(const char* eventId) NN_NOEXCEPT :
    m_ApplicationId(ApplicationId::GetInvalidId()),
    m_Category(ReportCategory_Defualt),
    m_Buffer(nullptr),
    m_Size(0),
    m_Position(0)
{
    nn::Result result = SetEventId(eventId);

    if (result.IsFailure())
    {
        m_EventId[0] = '\0';
        NN_SDK_ASSERT(false, "EventId is invalid.");
    }
}

void SystemReport::SetBuffer(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, BufferSizeMin);

    m_Buffer = static_cast<Bit8*>(buffer);
    m_Size = size;

    Clear();
}

nn::Result SystemReport::SetEventId(const char* eventId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(eventId);

    size_t length;
    NN_RESULT_THROW_UNLESS(detail::VerifyEventId(&length, eventId), ResultInvalidEventId());

    nn::util::Strlcpy(m_EventId, eventId, EventIdLengthMax + 1);

    NN_RESULT_SUCCESS;
}

nn::Result SystemReport::SetApplicationId(const nn::ApplicationId& applicationId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(applicationId, ApplicationId::GetInvalidId());

    m_ApplicationId = applicationId;

    NN_RESULT_SUCCESS;
}

void SystemReport::SetCategory(ReportCategory category) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX(category, ReportCategory_First, ReportCategory_Last);

    m_Category = category;
}

void SystemReport::Clear() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    detail::SystemReportGenerator::Initialize(&m_Position, m_Buffer, m_Size);
}

nn::Result SystemReport::Add(const char* key, int64_t value) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_Buffer, m_Size, m_Position);
}

nn::Result SystemReport::Add(const char* key, const Any64BitId& value) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_Buffer, m_Size, m_Position);
}

nn::Result SystemReport::Add(const char* key, double value) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_Buffer, m_Size, m_Position);
}

nn::Result SystemReport::Add(const char* key, const char* value) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::AddKeyValue(&m_Position, key, value, m_Buffer, m_Size, m_Position);
}

nn::Result SystemReport::Add(const char* key, const void* value, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::AddKeyValue(&m_Position, key, value, size, m_Buffer, m_Size, m_Position);
}

nn::Result SystemReport::Save() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    size_t length;
    NN_RESULT_THROW_UNLESS(detail::VerifyEventId(&length, m_EventId), ResultInvalidEventId());

    NN_RESULT_THROW_UNLESS(m_ApplicationId != ApplicationId::GetInvalidId(), ResultInvalidApplicationId());

    NN_UTIL_SCOPE_EXIT
    {
        Clear();
    };

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    switch (m_Category)
    {
    case ReportCategory_Normal:
        NN_RESULT_DO(detail::HandleSaveResult(session->SaveReport(nn::sf::InArray<char>(m_EventId, length + 1),
            m_ApplicationId, nn::sf::InBuffer(reinterpret_cast<const char*>(m_Buffer), m_Position))));
        break;
    case ReportCategory_AntiPiracy:
        NN_RESULT_DO(detail::HandleSaveResult(session->SaveReportForAntiPiracy(nn::sf::InArray<char>(m_EventId, length + 1),
            m_ApplicationId, nn::sf::InBuffer(reinterpret_cast<const char*>(m_Buffer), m_Position))));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SystemReport::Save(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    size_t length;
    NN_RESULT_THROW_UNLESS(detail::VerifyEventId(&length, m_EventId), ResultInvalidEventId());

    NN_RESULT_THROW_UNLESS(m_ApplicationId != ApplicationId::GetInvalidId(), ResultInvalidApplicationId());

    NN_UTIL_SCOPE_EXIT
    {
        Clear();
    };

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    switch (m_Category)
    {
    case ReportCategory_Normal:
        NN_RESULT_DO(detail::HandleSaveResult(session->SaveReportWithUser(uid, nn::sf::InArray<char>(m_EventId, length + 1),
            m_ApplicationId, nn::sf::InBuffer(reinterpret_cast<const char*>(m_Buffer), m_Position))));
        break;
    case ReportCategory_AntiPiracy:
        NN_RESULT_DO(detail::HandleSaveResult(session->SaveReportWithUserForAntiPiracy(uid, nn::sf::InArray<char>(m_EventId, length + 1),
            m_ApplicationId, nn::sf::InBuffer(reinterpret_cast<const char*>(m_Buffer), m_Position))));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

int SystemReport::GetCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_Buffer);

    return detail::SystemReportGenerator::GetKeyValueCount(m_Buffer, m_Size);
}

}}
