﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/detail/srepo_ApiDetail.h>
#include <nn/srepo/srepo_Result.h>
#include <nn/srepo/detail/msgpack/srepo_MessagePack.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace srepo { namespace detail {

bool VerifyEventId(size_t* outLength, const char* eventId) NN_NOEXCEPT
{
    if (!eventId)
    {
        return false;
    }

    const char* p = eventId;

    while (*p)
    {
        char c = *p;

        // [0-9a-z_]
        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c == '_'))
        {
            p++;
        }
        else
        {
            return false;
        }
        if (static_cast<size_t>(p - eventId) > EventIdLengthMax)
        {
            return false;
        }
    }

    size_t length = p - eventId;

    *outLength = length;

    return (length > 0);
}

bool VerifyKey(size_t* outLength, const char* key) NN_NOEXCEPT
{
    if (!key)
    {
        return false;
    }

    const char* p = key;

    while (*p)
    {
        char c = *p;

        // [0-9a-zA-Z:_]
        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c == ':') || (c == '_'))
        {
            p++;
        }
        else
        {
            return false;
        }
        if (static_cast<size_t>(p - key) > KeyLengthMax)
        {
            return false;
        }
    }

    size_t length = p - key;

    *outLength = length;

    return (length > 0);
}

bool VerifyKey(const char* key, size_t length) NN_NOEXCEPT
{
    if (!key)
    {
        return false;
    }
    if (length == 0 || length > KeyLengthMax)
    {
        return false;
    }

    for (size_t i = 0; i < length; i++)
    {
        char c = key[i];

        // [0-9a-zA-Z:_]
        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c == ':') || (c == '_'))
        {
        }
        else
        {
            return false;
        }
    }

    return true;
}

bool VerifyStringValue(size_t* outLength, const char* value) NN_NOEXCEPT
{
    if (!value)
    {
        return false;
    }

    size_t length = nn::util::Strnlen(value, StringValueLengthMax + 1);

    if (length > StringValueLengthMax)
    {
        return false;
    }

    *outLength = length;

    if (length == 0)
    {
        return true;
    }
    else
    {
        return nn::util::VerifyUtf8String(value, length);
    }
}

bool VerifyStringValue(const char* value, size_t length) NN_NOEXCEPT
{
    if (!value)
    {
        return false;
    }
    if (length > StringValueLengthMax)
    {
        return false;
    }

    if (length == 0)
    {
        return true;
    }
    else
    {
        return nn::util::VerifyUtf8String(value, length);
    }
}

bool VerifyBinaryValue(const void* pointer, size_t size) NN_NOEXCEPT
{
    if (!pointer && size > 0u)
    {
        return false;
    }
    if (size > BinaryValueSizeMax)
    {
        return false;
    }

    return true;
}

bool VerifyReport(const Bit8* data, size_t dataSize) NN_NOEXCEPT
{
    msgpack::InputStreamParam stream = {data, dataSize};
    msgpack::AnyData any = {};

    if (!msgpack::ReadCurrent(&any, &stream))
    {
        return false;
    }
    if (any.type != msgpack::AnyDataType_Map || any.num > KeyValueCountMax)
    {
        return false;
    }

    struct KeyPosition
    {
        size_t length;
        size_t position;
    }
    keyPosision[KeyValueCountMax] = {};

    const int count = any.num;

    for (int i = 0; i < count; i++)
    {
        // キー
        if (!msgpack::ReadCurrent(&any, &stream))
        {
            return false;
        }
        if (any.type != msgpack::AnyDataType_String)
        {
            return false;
        }

        const char* key = reinterpret_cast<const char*>(&stream.data[any.string.position]);

        if (!VerifyKey(key, any.string.length))
        {
            return false;
        }

        keyPosision[i].length = any.string.length;
        keyPosision[i].position = any.string.position;

        // バリュー
        if (!msgpack::ReadCurrent(&any, &stream))
        {
            return false;
        }
        if (any.type != msgpack::AnyDataType_Float64 &&
            any.type != msgpack::AnyDataType_UInteger &&
            any.type != msgpack::AnyDataType_SInteger &&
            any.type != msgpack::AnyDataType_String &&
            any.type != msgpack::AnyDataType_Extension &&
            any.type != msgpack::AnyDataType_Binary)
        {
            return false;
        }
        if (any.type == msgpack::AnyDataType_String)
        {
            const char* value = reinterpret_cast<const char*>(&stream.data[any.string.position]);

            if (!VerifyStringValue(value, any.string.length))
            {
                return false;
            }
        }
        else if (any.type == msgpack::AnyDataType_Extension)
        {
            switch (any.extension.type)
            {
            case MsgpackExtensionType_Any64BitId:
                {
                    if (any.extension.length != 8)
                    {
                        return false;
                    }
                }
                break;
            default:
                return false;
            }
        }
        else if (any.type == msgpack::AnyDataType_Binary)
        {
            if (!VerifyBinaryValue(&stream.data[any.binary.position], any.binary.length))
            {
                return false;
            }
        }
    }

    if (stream.GetRemainSize() != 0)
    {
        return false;
    }

    // 重複確認
    for (int i = 0; i < count - 1; i++)
    {
        const char* key1 = reinterpret_cast<const char*>(&stream.data[keyPosision[i].position]);
        size_t keyLength1 = keyPosision[i].length;

        for (int j = i + 1; j < count; j++)
        {
            const char* key2 = reinterpret_cast<const char*>(&stream.data[keyPosision[j].position]);
            size_t keyLength2 = keyPosision[j].length;

            if (keyLength1 == keyLength2 && nn::util::Strnicmp(key1, key2, static_cast<int>(keyLength1)) == 0)
            {
                return false;
            }
        }
    }

    return true;
}

nn::Result HandleSaveResult(nn::Result result) NN_NOEXCEPT
{
    NN_RESULT_TRY(result)
        // 保存頻度が高すぎて保存処理が追いつかなかった。
        // または、メンテナンスモードなどで、バッファをフラッシュするプロセスが動作していない。
        NN_RESULT_CATCH(ResultOutOfResource)
        {
            NN_SDK_LOG("\033[31m[srepo] Warning: The report data was not saved because the save frequency is too high.\033[m\n");
        }
        // 不正な形式のレポートが保存された。
        NN_RESULT_CATCH(ResultInvalidReportData)
        {
            NN_SDK_ASSERT(false, "[srepo] The report data has invalid format. There is the possibility of memory corruption.");
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

}}}
