﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/detail/service/srepo_SrepoService.h>
#include <nn/srepo/detail/service/srepo_Common.h>

#include <nn/srepo/srepo_StateNotifier.h>
#include <nn/srepo/detail/srepo_ApiDetail.h>

#include <nn/srepo/detail/service/core/srepo_ReportBuffer.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        NN_RESULT_THROW_UNLESS(                                     \
            0 < size && size <= maxSize && value[size - 1] == '\0', \
            ResultInvalidArgument());                               \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

#define CHECK_UID(uid) \
    do                                          \
    {                                           \
        NN_RESULT_THROW_UNLESS(                 \
            uid != nn::account::InvalidUid,     \
            ResultInvalidArgument());           \
    }                                           \
    while (NN_STATIC_CONDITION(false))

#define CHECK_REPORT_DATA(eventId, data) \
    do                                                                                                                              \
    {                                                                                                                               \
        size_t length;                                                                                                              \
        NN_RESULT_THROW_UNLESS(detail::VerifyEventId(&length, eventId.GetData()),                                                   \
            ResultInvalidEventId());                                                                                                \
        NN_RESULT_THROW_UNLESS(detail::VerifyReport(reinterpret_cast<const nn::Bit8*>(data.GetPointerUnsafe()), data.GetSize()),    \
            ResultInvalidReportData());                                                                                             \
    }                                                                                                                               \
    while (NN_STATIC_CONDITION(false))

#if 0 // APIのレスポンス計測
namespace
{
    class Stopwatch
    {
    public:
        Stopwatch(const nn::TimeSpan& start, const char* mark)
            : m_Start(start)
            , m_Mark(mark)
        {
        }

        ~Stopwatch()
        {
            const auto elapsed = nn::os::GetSystemTick().ToTimeSpan() - m_Start;
            NN_DETAIL_SREPO_INFO("(%s) %lld micro ( == %lld milli )\n", m_Mark, elapsed.GetMicroSeconds(), elapsed.GetMilliSeconds());
        }
    private:
        const nn::TimeSpan m_Start;
        const char* m_Mark;
    };
}

#define NN_DETAIL_SREPO_SCOPED_TIME(current) \
    const Stopwatch NN_MAKE_TEMPORARY_NAME(stopwatch)(current, __func__)
#else
#define NN_DETAIL_SREPO_SCOPED_TIME(current) (void)0
#endif

namespace nn { namespace srepo { namespace detail { namespace service {

SrepoService::SrepoService(const char* serviceName, const Capability& capability) NN_NOEXCEPT :
    m_ServiceName(serviceName),
    m_Capability(capability)
{
    NN_UNUSED(m_ServiceName);
    // NN_DETAIL_SREPO_INFO("[srepo] Get srepo service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_DETAIL_SREPO_SCOPED_TIME(nn::os::GetSystemTick().ToTimeSpan());
    m_DurationReportGenerator.Initialize(nn::os::GetSystemTick().ToTimeSpan());
}

SrepoService::~SrepoService() NN_NOEXCEPT
{
}

nn::Result SrepoService::SaveReport(const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Write);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_REPORT_DATA(eventId, data);

    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Push(ReportCategory_Normal, nn::account::InvalidUid, eventId.GetData(), appId, data.GetPointerUnsafe(), data.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::SaveReportWithUser(const nn::account::Uid& uid,
    const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Write);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_UID(uid);

    CHECK_REPORT_DATA(eventId, data);

    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Push(ReportCategory_Normal, uid, eventId.GetData(), appId, data.GetPointerUnsafe(), data.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::SaveReportForAntiPiracy(const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Write);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_REPORT_DATA(eventId, data);

    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Push(ReportCategory_AntiPiracy, nn::account::InvalidUid, eventId.GetData(), appId, data.GetPointerUnsafe(), data.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::SaveReportWithUserForAntiPiracy(const nn::account::Uid& uid,
    const nn::sf::InArray<char>& eventId, nn::ApplicationId appId, const nn::sf::InBuffer& data) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Write);

    CHECK_IPC_STRING(eventId, EventIdLengthMax + 1);

    CHECK_UID(uid);

    CHECK_REPORT_DATA(eventId, data);

    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Push(ReportCategory_AntiPiracy, uid, eventId.GetData(), appId, data.GetPointerUnsafe(), data.GetSize()));

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::PopReport(nn::sf::Out<std::int32_t> outCategory, nn::sf::Out<nn::account::Uid> outUid,
    const nn::sf::OutArray<char>& outEventId, nn::sf::Out<nn::ApplicationId> outAppId, nn::sf::Out<std::int64_t> outDataSize, const nn::sf::OutBuffer& outData) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Read);

    ReportCategory category;
    nn::account::Uid uid;
    nn::ApplicationId appId;
    size_t dataSize;
    NN_RESULT_DO(detail::service::core::ReportBuffer::GetInstance().Pop(&category, &uid, outEventId.GetData(), &appId, &dataSize, outData.GetPointerUnsafe(), outEventId.GetLength(), outData.GetSize()));

    *outCategory = static_cast<int32_t>(category);
    *outUid = uid;
    *outAppId = appId;
    *outDataSize = dataSize;

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::GetPushEventReadableHandle(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Read);

    // バッファへの書き込みを検知するためのイベントは一つしかないが、
    // この API を呼び出せる権限のセッション数は一つに絞っているため問題ない。

    nn::os::NativeHandle handle = detail::service::core::ReportBuffer::GetInstance().GetPushEventReadableHandle();
    outHandle.Set(nn::sf::NativeHandle(handle, false));

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::NotifyUserList(const nn::sf::InArray<nn::account::Uid>& userList) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    // 登録済ユーザーはすべて登録解除し userList で上書き
    m_DurationReportGenerator.UnregisterAllUsers();

    //NN_DETAIL_SREPO_TRACE("SrepoService::NotifyUserList\n");
    for(auto&& uid : userList.ToSpan())
    {
        m_DurationReportGenerator.RegisterUser(uid, current);
        //NN_DETAIL_SREPO_TRACE(" - Uid %016llx-%016llx\n", uid._data[0], uid._data[1]);
    }

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::NotifyUserDeleted(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_DETAIL_SREPO_SCOPED_TIME(nn::os::GetSystemTick().ToTimeSpan());

    //NN_DETAIL_SREPO_TRACE("SrepoService::NotifyUserDelete %016llx-%016llx\n", uid._data[0], uid._data[1]);
    m_DurationReportGenerator.UnregisterUser(uid);

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::NotifyUserRegistered(const nn::account::Uid& uid) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    //NN_DETAIL_SREPO_TRACE("SrepoService::NotifyUserRegistration %016llx-%016llx\n", uid._data[0], uid._data[1]);
    m_DurationReportGenerator.RegisterUser(uid, current);

    NN_RESULT_SUCCESS;
}

nn::Result SrepoService::NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    //NN_DETAIL_SREPO_TRACE("SrepoService::NotifyUserClosed %016llx-%016llx\n", uid._data[0], uid._data[1]);

    return m_DurationReportGenerator.GenerateUserTriggerReport(
        nullptr,
        uid,
        current,
        core::UserDurationKind::AccountStatus,
        core::AccountStatus::Create(false));
}

nn::Result SrepoService::NotifyUserOpened(const nn::account::Uid& uid) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    //NN_DETAIL_SREPO_TRACE("SrepoService::NotifyUserOpened %016llx-%016llx\n", uid._data[0], uid._data[1]);

    return m_DurationReportGenerator.GenerateUserTriggerReport(
        nullptr,
        uid,
        current,
        core::UserDurationKind::AccountStatus,
        core::AccountStatus::Create(true));
}

nn::Result SrepoService::NotifyCompletedNetworkRequestChanged(int8_t type) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    auto completedNetworkRequestType = static_cast<CompletedNetworkRequestType>(type);

    NN_RESULT_THROW_UNLESS(false
        || completedNetworkRequestType == CompletedNetworkRequestType::None
        || completedNetworkRequestType == CompletedNetworkRequestType::WideAreaNetwork
        || completedNetworkRequestType == CompletedNetworkRequestType::LocalAreaNetwork
        || completedNetworkRequestType == CompletedNetworkRequestType::LocalDirectNetwork
        || completedNetworkRequestType == CompletedNetworkRequestType::NeighborDetection
        , nn::srepo::ResultInvalidArgument());

    //NN_DETAIL_SREPO_TRACE("[srepo] NotifyCompletedNetworkRequestChanged type %d\n", type);

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::CompletedNetworkRequestType,
        completedNetworkRequestType);
}

nn::Result SrepoService::NotifyFriendPresenceChanged(const nn::account::Uid& uid, int8_t state) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    auto presence = static_cast<FriendPresence>(state);

    NN_RESULT_THROW_UNLESS(false
        || presence == FriendPresence::Offline
        || presence == FriendPresence::Online
        || presence == FriendPresence::OnlinePlay
        , nn::srepo::ResultInvalidArgument());

    return m_DurationReportGenerator.GenerateUserTriggerReport(
        nullptr,
        uid,
        current,
        core::UserDurationKind::FriendPresence,
        presence);
}

nn::Result SrepoService::NotifyNotificationConnectivityChanged(int8_t state) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    auto connectivity = static_cast<NotificationConnectivity>(state);

    NN_RESULT_THROW_UNLESS(false
        || connectivity == NotificationConnectivity::Available
        || connectivity == NotificationConnectivity::Unavailable
        , nn::srepo::ResultInvalidArgument());

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::ConnectionOfNotification,
        connectivity == NotificationConnectivity::Available ? true : false);
}

nn::Result SrepoService::NotifyDeviceOperationModeChanged(int8_t mode) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    auto operationMode = static_cast<DeviceOperationMode>(mode);

    NN_RESULT_THROW_UNLESS(false
        || operationMode == DeviceOperationMode::Handheld
        || operationMode == DeviceOperationMode::Console
        // || operationMode == DeviceOperationMode::Unknown // Unknownは拒否
        , nn::srepo::ResultInvalidArgument());

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::DeviceOperationMode,
        operationMode);
}

nn::Result SrepoService::NotifySystemPowerStateChanged(int8_t state) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    auto powerState = static_cast<SystemPowerState>(state);

    NN_RESULT_THROW_UNLESS(false
        || powerState == SystemPowerState::FullAwake
        || powerState == SystemPowerState::MinimumAwake
        || powerState == SystemPowerState::MinimumAwakeForLowBatterySign
        || powerState == SystemPowerState::MinimumAwakeForBackgroundTask
        || powerState == SystemPowerState::SleepReady
        || powerState == SystemPowerState::ShutdownReady
        , nn::srepo::ResultInvalidArgument());

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::SystemPowerState,
        powerState);
}

nn::Result SrepoService::NotifyForegroundProgramChanged(const nn::ncm::ProgramId& id) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::ForegroundProgramId,
        id);
}

nn::Result SrepoService::NotifyControllerCountChanged(int8_t total, int8_t rail) NN_NOEXCEPT
{
    const auto current = nn::os::GetSystemTick().ToTimeSpan();
    NN_DETAIL_SREPO_SCOPED_TIME(current);

    NN_RESULT_THROW_UNLESS(total >= 0 && rail >= 0, nn::srepo::ResultInvalidArgument());

    return m_DurationReportGenerator.GenerateSystemTriggerReport(
        nullptr,
        current,
        core::SystemDurationKind::ControllerStatus,
        core::ControllerStatus::Create(total, rail));
}

}}}}
