﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/srepo/detail/service/core/srepo_ReportBuffer.h>

namespace nn { namespace srepo { namespace detail { namespace service { namespace core {

ReportBuffer::ReportBuffer() NN_NOEXCEPT :
    m_PushEvent(nn::os::EventClearMode_AutoClear, true)
{
    m_Buffer.filledSize = 0u;
}

nn::os::NativeHandle ReportBuffer::GetPushEventReadableHandle() NN_NOEXCEPT
{
    return m_PushEvent.GetReadableHandle();
}

nn::Result ReportBuffer::Push(ReportCategory category, const nn::account::Uid& uid, const char* eventId, const nn::ApplicationId& appId, const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(eventId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    msgpack::OutputStreamParam stream = {m_Buffer.GetWritePointer(), m_Buffer.GetRemainSize()};

    NN_RESULT_THROW_UNLESS(msgpack::WriteSignedIntegerAutoSize(&stream, static_cast<int64_t>(category)), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteBinary(&stream, &uid, static_cast<uint32_t>(sizeof (uid))), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteString(&stream, eventId, nn::util::Strnlen(eventId, EventIdLengthMax)), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteUnsignedInteger64(&stream, appId.value), ResultOutOfResource());
    NN_RESULT_THROW_UNLESS(msgpack::WriteBinary(&stream, data, static_cast<uint32_t>(dataSize)), ResultOutOfResource());

    m_Buffer.filledSize += stream.position;

    m_PushEvent.Signal();

    NN_RESULT_SUCCESS;
}

nn::Result ReportBuffer::Pop(ReportCategory* outCategory, nn::account::Uid* outUid, char* outEventIdBuffer, nn::ApplicationId* outAppId, size_t* outDataSize, void* outDataBuffer, size_t outEventIdBufferSize, size_t outDataBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCategory);
    NN_SDK_REQUIRES_NOT_NULL(outUid);
    NN_SDK_REQUIRES_NOT_NULL(outEventIdBuffer);
    NN_SDK_REQUIRES_NOT_NULL(outAppId);
    NN_SDK_REQUIRES_NOT_NULL(outDataSize);
    NN_SDK_REQUIRES_NOT_NULL(outDataBuffer);
    NN_SDK_REQUIRES_GREATER(outEventIdBufferSize, 0u);
    NN_SDK_REQUIRES_GREATER(outDataBufferSize, 0u);

    NN_UTIL_LOCK_GUARD(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_Buffer.filledSize > 0u, ResultBufferEmpty());

    msgpack::AnyData any = {};
    msgpack::InputStreamParam stream = {m_Buffer.buffer, m_Buffer.filledSize};

    NN_ABORT_UNLESS(msgpack::ReadCurrent(&any, &stream));
    NN_ABORT_UNLESS(any.type == msgpack::AnyDataType_SInteger);
    const ReportCategory category = static_cast<ReportCategory>(any.si);

    NN_ABORT_UNLESS(msgpack::ReadCurrent(&any, &stream));
    NN_ABORT_UNLESS(any.type == msgpack::AnyDataType_Binary);
    NN_ABORT_UNLESS(any.binary.length == sizeof (*outUid));
    const void* pUid = &stream.data[any.binary.position];

    NN_ABORT_UNLESS(msgpack::ReadCurrent(&any, &stream));
    NN_ABORT_UNLESS(any.type == msgpack::AnyDataType_String);
    NN_RESULT_THROW_UNLESS(any.string.length < outEventIdBufferSize, ResultOutOfResource());
    const char* pEventId = reinterpret_cast<const char*>(&stream.data[any.string.position]);
    const size_t eventIdLength = any.string.length;

    NN_ABORT_UNLESS(msgpack::ReadCurrent(&any, &stream));
    NN_ABORT_UNLESS(any.type == msgpack::AnyDataType_UInteger);
    const nn::ApplicationId appId = { any.ui };

    NN_ABORT_UNLESS(msgpack::ReadCurrent(&any, &stream));
    NN_ABORT_UNLESS(any.type == msgpack::AnyDataType_Binary);
    NN_RESULT_THROW_UNLESS(any.binary.length <= outDataBufferSize, ResultOutOfResource());
    const void* pData = &stream.data[any.binary.position];
    const size_t dataSize = any.binary.length;

    *outCategory = category;
    std::memcpy(outUid, pUid, sizeof (*outUid));
    std::memcpy(outEventIdBuffer, pEventId, eventIdLength);
    outEventIdBuffer[eventIdLength] = '\0';
    *outAppId = appId;
    std::memcpy(outDataBuffer, pData, dataSize);
    *outDataSize = dataSize;

    std::memmove(m_Buffer.buffer, &stream.data[stream.position], stream.GetRemainSize());
    m_Buffer.filledSize -= stream.position;

    NN_RESULT_SUCCESS;
}

}}}}}
