﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/spsm/spsm_Debug.h>
#include <nn/spsm/server/spsm_Server.h>
#include "spsm_PowerStateMachine.h"
#include "spsm_EventLog.h"
#include "spsm_PerformanceLog.h"
#include "spsm_WakeReason.h"

namespace nn { namespace spsm { namespace server {

    ServerInterfaceImpl::ServerInterfaceImpl() NN_NOEXCEPT
    {
    }

    Result ServerInterfaceImpl::GetCurrentState(nn::sf::Out<std::int32_t> outCurrentState) NN_NOEXCEPT
    {
        outCurrentState.Set(static_cast<int32_t>(PowerStateMachine::GetInstance().GetCurrentState()));
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::EnterSleep(nn::sf::Out<nn::sf::NativeHandle> outAwakeEventHandle) NN_NOEXCEPT
    {
        os::SystemEvent* pAwakeEvent;
        NN_RESULT_DO(PowerStateMachine::GetInstance().EnterSleep(&pAwakeEvent));
        outAwakeEventHandle.Set(sf::NativeHandle(pAwakeEvent->GetReadableHandle(), false));
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::GetLastWakeReason(nn::sf::Out<nn::spsm::WakeReasonFlagSet> outWakeReason) NN_NOEXCEPT
    {
        outWakeReason.Set(server::GetWakeReasonFlagSet());
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::Shutdown(bool reboot) NN_NOEXCEPT
    {
        NN_RESULT_DO(PowerStateMachine::GetInstance().Shutdown(reboot));
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::GetNotificationMessageEventHandle(nn::sf::Out<nn::sf::NativeHandle> outNotificationMessageEventHandle) NN_NOEXCEPT
    {
        auto* pNotificationMessageEvent = PowerStateMachine::GetInstance().GetNotificationMessageQueue()->GetNonEmptyEvent();
        NN_SDK_ASSERT_NOT_NULL(pNotificationMessageEvent);
        outNotificationMessageEventHandle.Set(sf::NativeHandle(pNotificationMessageEvent->GetReadableHandle(), false));
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::ReceiveNotificationMessage(nn::sf::Out<std::int32_t> outMessage) NN_NOEXCEPT
    {
        NotificationMessage message;
        if ( PowerStateMachine::GetInstance().GetNotificationMessageQueue()->TryDequeue(&message) )
        {
            outMessage.Set(static_cast<int32_t>(message));
        }
        else
        {
            outMessage.Set(static_cast<int32_t>(NotificationMessage_None));
        }
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::AnalyzeLogForLastSleepWakeSequence(nn::sf::Out<SleepWakeSequenceAnalyzedData> outData) NN_NOEXCEPT
    {
        if ( outData.GetPointer() )
        {
            SleepWakeSequenceAnalyzedData data;
            server::AnalyzeLogForLastSleepWakeSequence(&data);
            *outData.GetPointer() = data;
        }
        NN_RESULT_SUCCESS; // null でも無視
    }

    Result ServerInterfaceImpl::AnalyzePerformanceLogForLastSleepWakeSequence(const nn::sf::OutBuffer& outData) NN_NOEXCEPT
    {
        auto pData = reinterpret_cast<nn::spsm::SleepWakeSequencePerformanceData*>(outData.GetPointerUnsafe());
        server::AnalyzePerformanceLogForLastSleepWakeSequence(pData);
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::ResetEventLog() NN_NOEXCEPT
    {
        server::ResetLog();
        server::ResetPerformanceLog();
        NN_RESULT_SUCCESS;
    }

    Result ServerInterfaceImpl::PutErrorState() NN_NOEXCEPT
    {
        NN_RESULT_DO(PowerStateMachine::GetInstance().PutErrorState());
        NN_RESULT_SUCCESS;
    }

    void InitializePowerStateMachine(const InitializeMode mode) NN_NOEXCEPT
    {
        PowerStateMachine::GetInstance().Initialize(mode);
    }

    void LoopPowerStateMachine() NN_NOEXCEPT
    {
        PowerStateMachine::GetInstance().LoopStateMachine();
    }

    PowerStateMessageQueue* GetPowerStateMessageQueue() NN_NOEXCEPT
    {
        return PowerStateMachine::GetInstance().GetPowerStateMessageQueue();
    }

    void AddStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pObserver);
        return PowerStateMachine::GetInstance().AddStateChangeObserver(pObserver);
    }

    void RemoveStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pObserver);
        return PowerStateMachine::GetInstance().RemoveStateChangeObserver(pObserver);
    }

}}}

