﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/srepo/srepo_StateNotifier.h>

#include <nn/spsm/detail/spsm_Log.h>
#include "spsm_Debug.h"
#include "spsm_ErrorReporter.h"
#include "spsm_EventLog.h"
#include "spsm_IPowerStateHandler.h"
#include "spsm_PowerStateHandlerImpl.h"
#include "spsm_PowerStateMachine.h"

namespace nn { namespace spsm { namespace server {

    PowerStateMachine::PowerStateMachine() NN_NOEXCEPT :
        m_StateTransitionMutex(false),
        m_PowerStateMessageQueue(),
        m_CurrentState(),
        m_Context(),
        m_AwakeEvent(nn::os::EventClearMode_ManualClear, true), // TORIAEZU: 今はプロセス内通信 (DFC) でしか使っていないが GetReadableHandle はできるようにしておく
        m_ObserverList(),
        m_IsOnError(false)
    {
    }

    void PowerStateMachine::Initialize(const InitializeMode mode) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateTransitionMutex)> lock(m_StateTransitionMutex);

        // ハンドラの初期化
        InitializePowerStateHandlerImpl(&m_PowerStateMessageQueue, &m_NotificationMessageQueue, &m_AwakeEvent, &m_Context, mode);

        // 内部ステートマシンの稼働を開始
        m_IsOnError = false;
        m_CurrentState = PowerState_Initial;
        m_Context.previousState = PowerState_Initial;
        m_Context.destinationState = PowerState_None;
        m_PowerStateMessageQueue.ResetQueue();
        m_PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_StartStateMachine));
    }

    PowerStateMessageQueue* PowerStateMachine::GetPowerStateMessageQueue() NN_NOEXCEPT
    {
        return &m_PowerStateMessageQueue;
    }

    NotificationMessageQueue* PowerStateMachine::GetNotificationMessageQueue() NN_NOEXCEPT
    {
        return &m_NotificationMessageQueue;
    }

    PowerState PowerStateMachine::GetCurrentState() const NN_NOEXCEPT
    {
        return m_CurrentState;
    }

    void PowerStateMachine::AddStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateTransitionMutex)> lock(m_StateTransitionMutex);
        m_ObserverList.push_back(*pObserver);
    }

    void PowerStateMachine::RemoveStateChangeObserver(StateChangeObserver* pObserver) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateTransitionMutex)> lock(m_StateTransitionMutex);
        m_ObserverList.erase(m_ObserverList.iterator_to(*pObserver));
    }

    Result PowerStateMachine::EnterSleep(os::SystemEvent** ppOutAwakeEvent) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(ppOutAwakeEvent);

        {
            std::lock_guard<decltype(m_StateTransitionMutex)> lock(m_StateTransitionMutex);
            m_AwakeEvent.Clear();
        }

        m_PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_SleepRequested));

        *ppOutAwakeEvent = &m_AwakeEvent;
        NN_RESULT_SUCCESS;
    }

    Result PowerStateMachine::Shutdown(bool reboot) NN_NOEXCEPT
    {
        if ( reboot )
        {
            m_PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_RebootRequested));
        }
        else
        {
            m_PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_ShutdownRequested));
        }
        NN_RESULT_SUCCESS;
    }

    Result PowerStateMachine::PutErrorState() NN_NOEXCEPT
    {
        // PowerStateMessage_PutErrorRequested を投げて遷移をトリガーするが、ResetQueue などとかち合うとメッセージが捨てられる可能性があるので、フェイルセーフとしてフラグも立てておく
        m_IsOnError = true;
        AbortAllPowerStateHandling();
        m_PowerStateMessageQueue.Enqueue(AddTimeStamp(PowerStateMessage_PutErrorRequested));
        NN_RESULT_SUCCESS;
    }

    void PowerStateMachine::LoopStateMachine() NN_NOEXCEPT
    {
        // Note: シーケンスを比較的ベタに実装できるよう、ステートマシン内部のハンドリング処理は長時間ブロックしても良いことにする
        //       それによる予期せぬ影響を避けておくため、スレッドを他に使い回せない外部メソッドにしておく

        UpdateStateInfoOnReport(); // まず一回コンテキストを更新しておく
        for (;;)
        {
            auto message = m_PowerStateMessageQueue.Dequeue();
            ProcessMessage(message);
        }
    }

    void PowerStateMachine::ProcessMessage(PowerStateMessageWithMeta messageWithMeta) NN_NOEXCEPT
    {
        // メッセージハンドラの呼び出し
        auto handler = GetPowerStateHandlerImpl(m_CurrentState);
        auto newDestState = PowerState_None;

        // エラーステート遷移が求められている場合はハンドラを呼ばずに強制遷移
        if ( messageWithMeta == PowerStateMessage_PutErrorRequested ||
             (m_IsOnError && m_CurrentState != PowerState_Error) )
        {
            newDestState = PowerState_Error; // エラー時にはハンドラを呼ばずに強制遷移
        }
        else
        {
            newDestState = handler->OnMessage(messageWithMeta);
        }
        LogPowerStateHandler(m_CurrentState, newDestState, PowerStateHandlerType_OnMessage, messageWithMeta); // テスト・デバッグ用フットプリント

        // 新たな目的地を指定されたか？
        if (newDestState != NoPowerStatechange)
        {
            m_Context.destinationState = newDestState;
            UpdateStateInfoOnReport(); // destinationState が更新されたので
        }

        ProcessStateTransition();
    }

    void PowerStateMachine::ProcessStateTransition() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_StateTransitionMutex)> lock(m_StateTransitionMutex);

        if (m_Context.destinationState == m_CurrentState) // 遷移不要だった
        {
            m_Context.destinationState = PowerState_None;
        }

        while (m_Context.destinationState != PowerState_None)
        {
            // 前のステートから出る
            auto handler = GetPowerStateHandlerImpl(m_CurrentState);
            handler->Exit();
            LogPowerStateHandler(m_CurrentState, PowerState_None, PowerStateHandlerType_Exit); // テスト・デバッグ用フットプリント

            // オブザーバのステート遷移ハンドラを呼び出す
            for ( const auto& observer : m_ObserverList )
            {
                observer.OnStateExit(m_CurrentState);
            }

            // 現在の目的地に応じて次に入るステートを更新する
            auto nextState = DetermineNextState(m_CurrentState, m_Context.destinationState);
#if !defined(NN_BUILD_CONFIG_OS_WIN)
            NN_DETAIL_SPSM_INFO_V1("DetermineNextState: (curr %s, dest %s) => nextState %s\n",
                GetPowerStateNameString(m_CurrentState), GetPowerStateNameString(m_Context.destinationState), GetPowerStateNameString(nextState));
#endif
            NN_SDK_ASSERT_NOT_EQUAL(nextState, NoPowerStatechange);
            NN_SDK_ASSERT_NOT_EQUAL(nextState, m_CurrentState);
            m_Context.previousState = m_CurrentState;
            m_CurrentState = nextState;
            UpdateStateInfoOnReport(); // m_CurrentState, previousState が更新されたので
            handler = GetPowerStateHandlerImpl(m_CurrentState);

            // 次のステートに入る
            auto newDestState = handler->Entry();
            if ( m_IsOnError && m_CurrentState != PowerState_Error ) // エラーステート遷移が求められている場合はハンドラからの戻り値を無視して上書き
            {
                newDestState = PowerState_Error;
            }
            LogPowerStateHandler(m_CurrentState, newDestState, PowerStateHandlerType_Entry); // テスト・デバッグ用フットプリント

            // オブザーバのステート遷移ハンドラを呼び出す
            for ( const auto& observer : m_ObserverList )
            {
                observer.OnStateEntry(m_CurrentState);
            }

            // 新たな目的地を指定されたか？
            if (newDestState != NoPowerStatechange)
            {
                m_Context.destinationState = newDestState;
                UpdateStateInfoOnReport(); // destinationState が更新されたので
                // NOTE: newDestState に m_CurrentState を指定することですぐに OnDestination を発動させることができる
            }

            if (m_Context.destinationState == m_CurrentState) // 目的地到着
            {
                // 先に destinationState をクリアしてから OnDestination を呼ぶ
                m_Context.destinationState = PowerState_None;

                newDestState = handler->OnDestination();
                if ( m_IsOnError && m_CurrentState != PowerState_Error ) //エラーステート遷移が求められている場合はハンドラからの戻り値を無視して上書き
                {
                    newDestState = PowerState_Error;
                }
                LogPowerStateHandler(m_CurrentState, newDestState, PowerStateHandlerType_OnDestination); // テスト・デバッグ用フットプリント

                // 新たな目的地を指定されたか？
                if (newDestState != NoPowerStatechange && newDestState != m_CurrentState)
                {
                    m_Context.destinationState = newDestState;
                    UpdateStateInfoOnReport(); // destinationState が更新されたので
                }
            }

            NN_SDK_ASSERT_NOT_EQUAL(m_Context.destinationState, m_CurrentState);
        }

        NN_SDK_ASSERT_EQUAL(m_Context.destinationState, PowerState_None);
    }

    void PowerStateMachine::UpdateStateInfoOnReport() NN_NOEXCEPT
    {
        auto errorReportResult = UpdateErrorReportContext(m_CurrentState, m_Context.previousState, m_Context.destinationState);
        if ( errorReportResult.IsFailure() )
        {
            NN_DETAIL_SPSM_ERROR("Could not update error report context (result 0x%08x).\n", errorReportResult.GetInnerValueForDebug());
        }

        static auto ToSrepoState = [](nn::spsm::PowerState state) NN_NOEXCEPT -> nn::srepo::SystemPowerState
        {
            switch ( state )
            {
                case PowerState_None:
                case PowerState_Initial:
                case PowerState_Error:
                    return nn::srepo::SystemPowerState::FullAwake; // 起動途中などの特殊なステートを srepo で区別する意味はないので FullAwake にまとめておく
                case PowerState_FullAwake:
                    return nn::srepo::SystemPowerState::FullAwake;
                case PowerState_MinimumAwake:
                    return nn::srepo::SystemPowerState::MinimumAwake;
                case PowerState_MinimumAwakeForLowBatterySign:
                    return nn::srepo::SystemPowerState::MinimumAwakeForLowBatterySign;
                case PowerState_MinimumAwakeForBackgroundTask:
                    return nn::srepo::SystemPowerState::MinimumAwakeForBackgroundTask;
                case PowerState_SleepReady:
                    return nn::srepo::SystemPowerState::SleepReady;
                case PowerState_ShutdownReady:
                    return nn::srepo::SystemPowerState::ShutdownReady;
            } // NOLINT(style/switch_default)

            NN_UNEXPECTED_DEFAULT;
        };
        nn::srepo::NotifySystemPowerStateChanged(ToSrepoState(m_CurrentState));
    }

}}}

