﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/psc.h>
#include <nn/psc/psc_PmControl.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/psc/psc_RequiredPmModuleIdList.h>
#include <nn/spsm/spsm_ResultPrivate.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_Server.h>
#include <nn/spsm/observer/spsm_Observer.h>

#if defined(NN_DETAIL_SPSM_DUMP_TRANSITION_FAILURE)
#include <nn/svc/svc_Base.h>
#endif

#include "spsm_IPowerStateHandler.h"
#include "spsm_PowerStateHandlerImpl.h"
#include "spsm_ErrorReporter.h"
#include "spsm_WakeReason.h"
#include "spsm_Shutdown.h"
#include "spsm_SettingsHolder.h"
#include "spsm_Debug.h"
#include "spsm_EventLog.h"
#include "spsm_PerformanceLog.h"
#include "spsm_PmControl.h"

namespace nn { namespace spsm { namespace server {

    void EnterSc7();

namespace {

    bool g_IsAbortingAllHandling = false;

    nn::psc::PmTransitionOrder GetTransitionOrder(nn::psc::PmState current, nn::psc::PmState dest) NN_NOEXCEPT
    {
        switch (current)
        {
            case nn::psc::PmState_FullAwake:
            {
                // 最上位なのでどこに遷移するにも Lower 方向
                return nn::psc::PmTransitionOrder_ToLowerPowerState;
            }
            case nn::psc::PmState_MinimumAwake:
            {
                if ( dest == nn::psc::PmState_FullAwake )
                {
                    return nn::psc::PmTransitionOrder_ToHigherPowerState;
                }
                else // SleepReady, ShutdownReady
                {
                    NN_SDK_ASSERT(dest != nn::psc::PmState_ShutdownReady || dest != nn::psc::PmState_SleepReady);
                    return nn::psc::PmTransitionOrder_ToLowerPowerState;
                }
                break;
            }
            case nn::psc::PmState_SleepReady:
            {
                NN_SDK_ASSERT(dest != nn::psc::PmState_ShutdownReady);
                return nn::psc::PmTransitionOrder_ToHigherPowerState;
            }
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::psc::PmState ToPscPowerState(nn::spsm::PowerState state) NN_NOEXCEPT
    {
        switch (state)
        {
            case nn::spsm::PowerState_FullAwake: return nn::psc::PmState_FullAwake;
            case nn::spsm::PowerState_MinimumAwake: return nn::psc::PmState_MinimumAwake;
            case nn::spsm::PowerState_MinimumAwakeForLowBatterySign: return nn::psc::PmState_MinimumAwake; // PSC 上は MinimumAwake 扱い
            case nn::spsm::PowerState_MinimumAwakeForBackgroundTask: return nn::psc::PmState_MinimumAwake; // PSC 上は MinimumAwake 扱い
            case nn::spsm::PowerState_SleepReady: return nn::psc::PmState_SleepReady;
            case nn::spsm::PowerState_ShutdownReady: return nn::psc::PmState_ShutdownReady;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    Result GetPmControlResult(nn::psc::PmControl& control) NN_NOEXCEPT
    {
        nn::psc::PmControlState state;
        NN_RESULT_DO(control.GetState(&state));
        NN_SDK_ASSERT(state == nn::psc::PmControlState_Idle || nn::psc::PmControlState_Error); // これ以外での Result の取り出しは無意味
        return control.GetResult();
    }
#endif

    Result ExecutePmRequest(nn::psc::PmState pmState, nn::psc::PmTransitionOrder order, nn::psc::PmFlagSet flags = nn::psc::MakeNoPmFlags()) NN_NOEXCEPT
    {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::psc::PmControl*  pPmControl = GetPmControl();

        // PSC にリクエストを投げる
        NN_RESULT_DO(pPmControl->DispatchRequest(pmState, flags, order));

        // 完了通知を待つ
        // 長時間完了しない場合は解析用ログをダンプ（タイムアウトしてエラーに落とす必然性はない）
        nn::TimeSpan elapsedTime = 0;
        const nn::TimeSpan PollInterval = nn::TimeSpan::FromSeconds(3);
        while (!pPmControl->GetEventPointer()->TimedWait(PollInterval))
        {
            elapsedTime += PollInterval;
            if ( g_IsAbortingAllHandling )
            {
                // 上位から遷移停止指示が来ているので、（ベストエフォートで）処理を停止して上位にエラーを返す
                // 上位で PowerState_Error に遷移するはず
                auto cancelResult = pPmControl->Cancel();
                if ( cancelResult.IsFailure() )
                {
                    NN_DETAIL_SPSM_ERROR("Could not cancel transaction (result 0x%08x).\n", cancelResult.GetInnerValueForDebug());
                }
                NN_RESULT_THROW(nn::spsm::ResultTransitionAborted());
            }
            if ( elapsedTime <= nn::TimeSpan::FromSeconds(15) ) // ある程度ダンプしたらそれ以上表示しない
            {
                NN_DETAIL_SPSM_WARN("PmControl dispatched request is taking too long.\n");

                pPmControl->PrintModuleInformation(); // モジュール情報のダンプ
            }
            if ( 0 < SettingsHolder::GetInstance().GetTransitionTimeoutTime() && elapsedTime >= SettingsHolder::GetInstance().GetTransitionTimeoutTime() )
            {
#if defined(NN_DETAIL_SPSM_DUMP_TRANSITION_FAILURE)
                NN_DETAIL_SPSM_INFO("Dump started.\n");
                nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, -1ll, 0ll, 0ll); // All processes
                nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, -2ll, 0ll, 0ll); // Kernel threads
                nn::svc::KernelDebug(nn::svc::KernelDebugType_ThreadCallstack, -1ll, 0ll, 0ll);
                for(nn::Bit64 pid =  0; pid <   6; ++pid)
                {
                    nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, pid, 0ll, 0ll);
                }
                for(nn::Bit64 pid = 80; pid < 130; ++pid)
                {
                    nn::svc::KernelDebug(nn::svc::KernelDebugType_CpuUtilization, pid, 0ll, 0ll);
                }
#endif

                // 永久に待ち続けて電池を使いつくして落ちる事態を避けるため & テレメトリで情報収集するため、FATAL に飛ばす
                auto errorReportResult = ReportStateTransitionError(pPmControl, nn::spsm::ResultTransitionTimedOut());
                if ( errorReportResult.IsFailure() )
                {
                    NN_DETAIL_SPSM_ERROR("Could not send error report (result 0x%08x).\n", errorReportResult.GetInnerValueForDebug());
                }
                NN_ABORT("PmControl dispatched request timed out. Shutdown forcibly.\n");
            }
        }

        // 成功時にもモジュール情報をダンプしておく（所要時間解析で有用なので）
        pPmControl->PrintModuleInformation();

        // PSC のシーケンス実行の結果を取得
        auto result = GetPmControlResult(*pPmControl);
        if ( result.IsFailure() )
        {
            auto errorReportResult = ReportStateTransitionError(pPmControl, result);
            if ( errorReportResult.IsFailure() )
            {
                NN_DETAIL_SPSM_ERROR("Could not send error report (result 0x%08x).\n", errorReportResult.GetInnerValueForDebug());
            }
            NN_RESULT_THROW(result);
        }

        // PSC のモジュール情報をメモリにダンプ（リリースビルドにおける所要時間解析用）
        LogPscPerformance(pPmControl);
#endif

        LogPscControl(pmState, order, flags);
        NN_RESULT_SUCCESS;
    }

    Result ExecutePmRequestByPowerState(nn::spsm::PowerState previous, nn::spsm::PowerState current, nn::psc::PmFlagSet flags = nn::psc::MakeNoPmFlags()) NN_NOEXCEPT
    {
        // PSC にリクエストを投げる
        auto pmState = ToPscPowerState(current);
        auto order = GetTransitionOrder(ToPscPowerState(previous), pmState);
        return ExecutePmRequest(pmState, order, flags);
    }

    void WaitModuleRegistration(const InitializeMode mode) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // すべての PmModule の登録を待つために、
        // すべての PmModule に依存するダミーの PmModule を作成し、PowerState の遷移を行う。
        nn::Result result;
        nn::psc::PmModule module;

        switch (mode)
        {
        case InitializeMode_Normal:
            result = module.Initialize(nn::psc::PmModuleId_Spsm,
                nn::psc::RequiredPmModuleIdListForNormalMode,
                sizeof(nn::psc::RequiredPmModuleIdListForNormalMode) / sizeof(nn::psc::PmModuleId),
                nn::os::EventClearMode_ManualClear);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            break;
        case InitializeMode_Maintenance:
            result = module.Initialize(nn::psc::PmModuleId_Spsm,
                nn::psc::RequiredPmModuleIdListForMaintenanceMode,
                sizeof(nn::psc::RequiredPmModuleIdListForMaintenanceMode) / sizeof(nn::psc::PmModuleId),
                nn::os::EventClearMode_ManualClear);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            break;
        case InitializeMode_NoWait:
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        auto pControl = GetPmControl();
        auto pControlEvent = pControl->GetEventPointer();
        auto pModuleEvent = module.GetEventPointer();

        NN_DETAIL_SPSM_INFO("Start waiting PSC module registration.\n");

        result = pControl->DispatchRequest(nn::psc::PmState_FullAwake, nn::psc::MakeNoPmFlags(), nn::psc::PmTransitionOrder_ToHigherPowerState);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        bool done = false;
        do
        {
            switch (nn::os::TimedWaitAny(nn::TimeSpan::FromSeconds(10), pModuleEvent->GetBase(), pControlEvent->GetBase()))
            {
            case -1: // timeout
                // 未登録のモジュールを把握するために、定期的にモジュールの情報を表示する
                pControl->PrintModuleInformation();
                break;

            case 0: // pModuleEvent
                pModuleEvent->Clear();

                nn::psc::PmFlagSet flags;
                nn::psc::PmState state;

                result = module.GetRequest(&state, &flags);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                result = module.Acknowledge(state, ResultSuccess());
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                break;

            case 1: // pControlEvent
                pControlEvent->Clear();

                nn::psc::PmControlState controlState;

                result = pControl->GetState(&controlState);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                if (controlState == nn::psc::PmControlState_Idle)
                {
                    done = true;
                }
                else if (controlState == nn::psc::PmControlState_Error)
                {
                    result = pControl->GetResult();
                    NN_ABORT("[spsm] PmControlState_Error: %08x.\n", result.GetInnerValueForDebug());
                }
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }

        } while(!done);

        NN_DETAIL_SPSM_INFO("Finish waiting PSC module registration.\n");

        module.Finalize();
#else
        NN_UNUSED(mode);
#endif
    }

    bool IsBatteryEnough() NN_NOEXCEPT
    {
        if ( !nn::spsm::server::SettingsHolder::GetInstance().HasBattery() )
        {
            // 充電池非搭載ならチェック不要
            return true;
        }

        // 電池残量（未加工）が 3% 以上でなければ要充電画面を表示して FullAwake 遷移を中断
        auto rawBatteryCharge = nn::psm::GetRawBatteryChargePercentage();

        // %f を使うとスタックを大量消費するので自前で整数にする
        int inte = static_cast<int>(rawBatteryCharge);
        int frac = static_cast<int>((rawBatteryCharge - static_cast<double>(inte)) * 100);
        NN_UNUSED(frac);
        NN_DETAIL_SPSM_INFO("Raw battery charge: %d.%d %%\n", inte, frac);

        if ( rawBatteryCharge < 3.0f )
        {
            NN_DETAIL_SPSM_INFO("Battery is not enough to go FullAwake.\n");
            return false;
        }
        return true;
    }
} // anonymous namespace

    // 初期状態から最初に訪れるステート
    class PowerStateHandlerImplInitial final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplInitial() NN_NOEXCEPT :
            IPowerStateHandler(PowerState_Initial),
            m_InitializeMode(InitializeMode_Normal)
        {}
        ~PowerStateHandlerImplInitial() NN_NOEXCEPT {}

        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            switch (messageWithMeta)
            {
                case PowerStateMessage_StartStateMachine:
                {
                    SettingsHolder::GetInstance().LoadStaticConfiguration();
                    SettingsHolder::GetInstance().LoadTransitionBehaviorSettings();

                    // TORIAEZU: SleepReady 中に settings/fs アクセスはできないので、これは起動時に一度だけ読む形にしておく
                    SettingsHolder::GetInstance().LoadSleepReadyBehaviorSettings();

                    IntializeWakeReason();

                    WaitModuleRegistration(m_InitializeMode);

                    // XXX: 本来は MinimumAwake を初期状態として、起動時に FullAwake に移行するシーケンスを設けたかったが、現在のプラットフォームでは妥協した
                    return PowerState_FullAwake;
                }
                default:
                    return NoPowerStatechange;
            }
        }

        void SetInitializeMode(const InitializeMode mode)
        {
            m_InitializeMode = mode;
        }

    private:
        InitializeMode m_InitializeMode;
    };

    // 全起動状態
    class PowerStateHandlerImplFullAwake final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplFullAwake() NN_NOEXCEPT :
            IPowerStateHandler(PowerState_FullAwake),
            m_PowerButtonStartPressingDuringFullAwake(false),
            m_HomeButtonStartPressingDuringFullAwake(false),
            m_IsRealSleepRequired(true),
            m_StateEnterTime(nn::os::Tick(0)),
            m_LastReportedTimeWhenPowerButtonPressedBriefly(nn::os::Tick(0))
        {}
        ~PowerStateHandlerImplFullAwake() NN_NOEXCEPT {}

        virtual PowerState Entry() NN_NOEXCEPT
        {
            // 再起動せずとも設定変更が反映されるようにするためここで読む (所要時間は実測 100us 程度だったのでウェイク時間等にはほぼ影響なし)
            SettingsHolder::GetInstance().LoadFullAwakeBehaviorSettings();

            m_PowerButtonStartPressingDuringFullAwake = false;
            m_HomeButtonStartPressingDuringFullAwake  = false;
            m_IsRealSleepRequired = false;
            m_StateEnterTime = nn::os::GetSystemTick();

            if (m_pContext->previousState != PowerState_Initial) // 起動後最初に FullAwake に入る際は PSC のシーケンスは実行しない
            {
                NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                    ExecutePmRequestByPowerState(m_pContext->previousState, HandledState)
                    );
            }

            return NoPowerStatechange;
        }

        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            // Awake の通知
            SignalAwakeEvent();
            return NoPowerStatechange;
        }

        virtual void Exit() NN_NOEXCEPT
        {
            m_PowerButtonStartPressingDuringFullAwake = false;
            m_HomeButtonStartPressingDuringFullAwake  = false;
            m_IsRealSleepRequired = false;
        }

        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            switch (messageWithMeta)
            {
                case PowerStateMessage_SleepRequested:
                {
                    // 起動中の設定変更を反映するためここで読み込み
                    auto &settingsHolder = SettingsHolder::GetInstance();
                    settingsHolder.LoadSleepModeSettings();
                    bool isEnterRealSleepSettingEnabled
                        = settingsHolder.IsEnterRealSleepSettingEnabled();
                    // スリープが必要な要因が投げられた場合は必ずスリープする
                    if ( m_IsRealSleepRequired || isEnterRealSleepSettingEnabled )
                    {
                        m_IsRealSleepRequired = false;
                        return PowerState_SleepReady;
                    }
                    else
                    {
                        // Pseudo Sleep が有効
                        os::SleepThread(nn::TimeSpan::FromSeconds(1));
                        SignalAwakeEvent();
                        return NoPowerStatechange;
                    }
                }
                case PowerStateMessage_ShutdownRequested:
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_RebootRequested:
                {
                    SetRebootFlag(true);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_EventPowerButtonStartedPressing:
                {
                    // FullAwake に入ってから電源ボタンが押され始めたことをタイムスタンプにより判断する
                    // （メッセージ側のタイムスタンプはキューイング時に押されている）
                    if ( nn::os::Tick(0) < messageWithMeta.GetTimeStamp() - m_StateEnterTime )
                    {
                        m_PowerButtonStartPressingDuringFullAwake = true;
                    }
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventPowerButtonPressedBriefly:
                {
                    // FullAwake に入ってから押され始めた短押しにのみ反応する
                    // MinimumAwake 等から起床するために電源ボタンを押したのにそれを離したらまた寝てしまうので起きられない、といった不具合を避けるため
                    // XXX: また、実際にスリープに入る前に連打されて、スリープ直前に投げたメッセージが復帰後に処理されてしまうケースをなるべく防ぐため、
                    // 暫定的に最後に短押しイベントを投げてから一定時間は同じイベントを投げないようにする
                    if ( m_PowerButtonStartPressingDuringFullAwake &&
                         SettingsHolder::GetInstance().GetHushTimeAfterBriefPowerButtonPress() < nn::os::ConvertToTimeSpan(messageWithMeta.GetTimeStamp() - m_LastReportedTimeWhenPowerButtonPressedBriefly) )
                    {
                        // am/omm にスルーパス
                        SendNotificationMessage(NotificationMessage_PowerButtonPressedBriefly);
                        m_PowerButtonStartPressingDuringFullAwake = false;
                        m_LastReportedTimeWhenPowerButtonPressedBriefly = nn::os::GetSystemTick();
                    }
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventPowerButtonPressedFor3Sec:
                {
                    // am/omm にスルーパス
                    SendNotificationMessage(NotificationMessage_PowerButtonPressedFor3Sec);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventPowerButtonPressedFor7Sec:
                {
                    // am/omm にスルーパス
                    SendNotificationMessage(NotificationMessage_PowerButtonPressedFor7Sec);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventAutoPowerDownTimerExpired:
                {
                    // am/omm から応答された時に必ずスリープすることを覚えておく
                    m_IsRealSleepRequired = true;
                    SendNotificationMessage(NotificationMessage_AutoPowerDownTimerExpired);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventCecSystemStandbyReceived:
                {
                    // am/omm にスルーパス
                    SendNotificationMessage(NotificationMessage_CecSystemStandbyReceived);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventSleepRequiredByHighTemperature:
                {
                    // am/omm から応答された時に必ずスリープすることを覚えておく
                    m_IsRealSleepRequired = true;
                    SendNotificationMessage(NotificationMessage_SleepRequiredByHighTemperature);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventSleepRequiredByLowBattery:
                {
                    // am/omm から応答された時に必ずスリープすることを覚えておく
                    m_IsRealSleepRequired = true;
                    SendNotificationMessage(NotificationMessage_SleepRequiredByLowBattery);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventShutdownRequiredByVeryLowBattery:
                {
                    // ここに来るケースが実在するか不明なのでログを残しておく
                    NN_DETAIL_SPSM_WARN("ShutdownRequiredByVeryLowBattery message was processed during FullAwake\n");

                    // am/omm にスルーパス
                    // ここに来るケースの発生確率が高く 7 秒長押しとハンドリングを分ける必要がある場合は別に NotificationMessage を定義する。
                    SendNotificationMessage(NotificationMessage_PowerButtonPressedFor7Sec);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventHomeButtonStartedPressing:
                {
                    // FullAwake に入ってから HOME ボタンが押され始めたことをタイムスタンプにより判断する
                    // （メッセージ側のタイムスタンプはキューイング時に押されている）
                    if ( nn::os::Tick(0) < messageWithMeta.GetTimeStamp() - m_StateEnterTime )
                    {
                        m_HomeButtonStartPressingDuringFullAwake = true;
                    }
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventHomeButtonPressedBriefly:
                {
                    // FullAwake に入ってから押され始めた短押しにのみ反応する
                    // MinimumAwake 等から起床するために HOME ボタンを押したのにそれを離したらまた寝てしまうので起きられない、といった不具合を避けるため
                    if ( m_HomeButtonStartPressingDuringFullAwake )
                    {
                        // am/omm にスルーパス
                        SendNotificationMessage(NotificationMessage_HomeButtonShortPressed);
                        m_HomeButtonStartPressingDuringFullAwake = false;
                    }
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventHomeButtonPressedLong:
                {
                    // am/omm にスルーパス
                    SendNotificationMessage(NotificationMessage_HomeButtonLongPressed);
                    return NoPowerStatechange;
                }
                case PowerStateMessage_EventFullAwakeRequestedByController:
                default:
                    return NoPowerStatechange;
            }
        } // NOLINT(impl/function_size)

    private:
        // FullAwake ステート固有パラメータ
        bool m_PowerButtonStartPressingDuringFullAwake; //!< このステート中に電源ボタンが押され始めたことを記録するフラグ
        bool m_HomeButtonStartPressingDuringFullAwake; //!< このステート中に HOME ボタンが押され始めたことを記録するフラグ
        bool m_IsRealSleepRequired; //!< スリープが必要なシグナルを投げたことを記録するフラグ
        nn::os::Tick m_StateEnterTime; //!< このステートに入ったときのタイムスタンプ
        nn::os::Tick m_LastReportedTimeWhenPowerButtonPressedBriefly; //!< 最後に電源ボタン短押しを am 以上に投げた時刻
    };

    // 最小構成起動
    class PowerStateHandlerImplMinimumAwake final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplMinimumAwake() NN_NOEXCEPT : IPowerStateHandler(PowerState_MinimumAwake) {}
        ~PowerStateHandlerImplMinimumAwake() NN_NOEXCEPT {}

        virtual PowerState Entry() NN_NOEXCEPT
        {
            NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                ExecutePmRequestByPowerState(m_pContext->previousState, HandledState)
                );

            if ( m_pContext->previousState == PowerState_SleepReady )
            {
                // 復帰要因を見て、遷移先を判定
                auto newDest = DetermineNewDestinationStateByWakeEvent();
                if ( newDest == PowerState_FullAwake && !IsBatteryEnough() )
                {
                    return PowerState_MinimumAwakeForLowBatterySign;
                }
                return newDest;
            }
            else
            {
                return NoPowerStatechange;
            }
            // NEVER REACHED HERE
        }

        // MinimumAwake に留まるときにのみ呼ばれる
        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            // 現時点では存在しない想定の遷移
            NN_SDK_ASSERT(false, "Unexpected transition\n");
            if ( m_pContext->previousState == PowerState_SleepReady ) // スリープから戻ってきて到達した場合にのみシグナル
            {
                SignalAwakeEvent();
            }
            return NoPowerStatechange;
        }

        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            // 現時点では存在しない想定の遷移
            // MinimumAwake での滞留をサポートするときに MinimumAwakeForBackgroundTask に準じる形でここをアップデートすること
            NN_SDK_ASSERT(false, "Unexpected transition\n");
            switch ( messageWithMeta )
            {
                case PowerStateMessage_SleepRequested:
                {
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_ShutdownRequested:
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_RebootRequested:
                {
                    SetRebootFlag(true);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_EventPowerButtonStartedPressing:
                {
                    // MinimumAwake に滞留している状態では電源ボタンの押しはじめで起きる
                    return PowerState_FullAwake;
                }
                case PowerStateMessage_EventSleepRequiredByHighTemperature:
                {
                    // ここに来るケースが実在するか不明なのでログ残しておく
                    NN_DETAIL_SPSM_WARN("SleepRequiredByHighTemperature message was processed during MinimumAwake\n");
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_EventSleepRequiredByLowBattery:
                {
                    // ここに来るケースが実在するか不明なのでログ残しておく
                    NN_DETAIL_SPSM_WARN("SleepRequiredByLowBattery message was processed during MinimumAwake\n");
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_EventShutdownRequiredByVeryLowBattery:
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                default:
                    return NoPowerStatechange;
            }
        }

    private:
        PowerState DetermineNewDestinationStateByWakeEvent() NN_NOEXCEPT
        {
#if defined(NN_BUILD_CONFIG_SPEC_NX)
            const WakeReasonFlagSet ShutdownReasonSet =
                WakeReasonFlag::BatteryLevelTooLow::Mask;
            const WakeReasonFlagSet FullAwakeReasonSet =
                WakeReasonFlag::PowerButtonPressed::Mask |
                WakeReasonFlag::AcOk::Mask |
                WakeReasonFlag::RtcForFullWakeup::Mask |
                WakeReasonFlag::BluetoothActivityDetect::Mask |
                WakeReasonFlag::SdCardDetect::Mask |
                WakeReasonFlag::GameCardDetect::Mask |
                WakeReasonFlag::ControllerButtonL::Mask |
                WakeReasonFlag::ControllerButtonR::Mask;
            const WakeReasonFlagSet BackgroundTaskReasonSet =
                WakeReasonFlag::RtcForBackgroundTask::Mask |
                WakeReasonFlag::WiFiActivityDetect::Mask;

            UpdateWakeReasonFlagSet();
            auto wakeReason = GetWakeReasonFlagSet();
            if ((wakeReason & ShutdownReasonSet).IsAnyOn())
            {
                SetRebootFlag(false);
                return PowerState_ShutdownReady;
            }
            else if ((wakeReason & FullAwakeReasonSet).IsAnyOn())
            {
                return PowerState_FullAwake;
            }
            else if ( (wakeReason & BackgroundTaskReasonSet).IsAnyOn() )
            {
                return PowerState_MinimumAwakeForBackgroundTask;
            }
            else
            {
                // TORIAEZU: SC7 を使ってない環境では、何のフラグも立っていなければ暫定的に FullAwake 行きに
                if ( !SettingsHolder::GetInstance().IsEnterSc7DebugSettingEnabled() && wakeReason.IsAllOff() )
                {
                    return PowerState_FullAwake;
                }

                // 上記以外は SleepReady に戻る
                return PowerState_SleepReady;
            }
#else
            return PowerState_FullAwake;
#endif
        }
    };

    // 要充電画面表示状態
    class PowerStateHandlerImplMinimumAwakeForLowBatterySign final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplMinimumAwakeForLowBatterySign() NN_NOEXCEPT : IPowerStateHandler(PowerState_MinimumAwakeForLowBatterySign) {}
        ~PowerStateHandlerImplMinimumAwakeForLowBatterySign() NN_NOEXCEPT {}

        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            SignalAwakeEvent();
            return NoPowerStatechange;
        }

        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            switch ( messageWithMeta )
            {
                case PowerStateMessage_SleepRequested:
                {
                    // 上位が一定時間要充電画面を表示したらこれを投げてくる想定
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_ShutdownRequested: // あるのか不明だがハンドリングはしておく
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_RebootRequested: // あるのか不明だがハンドリングはしておく
                {
                    SetRebootFlag(true);
                    return PowerState_ShutdownReady;
                }
                // このステート中、電源ボタンイベントは無視する
                // また、ごく短時間で抜ける想定であるため、本体高温や電池僅少イベントなどもハンドリングしない
                default:
                    return NoPowerStatechange;
            }
        }
    };

    // バックグラウンドタスク実行中
    class PowerStateHandlerImplMinimumAwakeForBackgroundTask final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplMinimumAwakeForBackgroundTask() NN_NOEXCEPT : IPowerStateHandler(PowerState_MinimumAwakeForBackgroundTask) {}
        ~PowerStateHandlerImplMinimumAwakeForBackgroundTask() NN_NOEXCEPT {}

        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            // MinimumAwake からの遷移同様、FullAwake への遷移時にはバッテリチェックが必要
            auto newDest = OnMessageBody(messageWithMeta);
            if ( newDest == PowerState_FullAwake && !IsBatteryEnough() )
            {
                return PowerState_MinimumAwakeForLowBatterySign;
            }
            return newDest;
        }

    private:
        PowerState OnMessageBody(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            switch ( messageWithMeta )
            {
                case PowerStateMessage_SleepRequested:
                {
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_ShutdownRequested:
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_RebootRequested:
                {
                    SetRebootFlag(true);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_EventPowerButtonStartedPressing:
                case PowerStateMessage_EventPowerButtonPressedBriefly:
                {
                    // スリープに相当するユーザ体験のため、電源ボタンの押し始め or 短押し終わりで起きる
                    SetWakeReasonFlagSet(WakeReasonFlag::PowerButtonPressed::Mask);
                    return PowerState_FullAwake;
                }
                case PowerStateMessage_EventSleepRequiredByHighTemperature:
                {
                    // NN_DETAIL_SPSM_INFO("SleepRequiredByHighTemperature message was processed during PowerStateHandlerImplMinimumAwakeForBackgroundTask\n");
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_EventSleepRequiredByLowBattery:
                {
                    // NN_DETAIL_SPSM_INFO("SleepRequiredByLowBattery message was processed during PowerStateHandlerImplMinimumAwakeForBackgroundTask\n");
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_EventShutdownRequiredByVeryLowBattery:
                {
                    SetRebootFlag(false);
                    return PowerState_ShutdownReady;
                }
                case PowerStateMessage_EventBackgroundTaskDone:
                {
                    // BG タスクが終わったのでスリープする
                    return PowerState_SleepReady;
                }
                case PowerStateMessage_EventHomeButtonStartedPressing:
                case PowerStateMessage_EventHomeButtonPressedBriefly:
                case PowerStateMessage_EventFullAwakeRequestedByController:
                {
                    // スリープに相当するユーザ体験のため、HOME ボタンの押し始め or 短押し終わりで起きる
                    SetWakeReasonFlagSet(WakeReasonFlag::BluetoothActivityDetect::Mask);
                    return PowerState_FullAwake;
                }
                case PowerStateMessage_EventPowerSupplyChanged:
                {
                    // スリープに相当するユーザ体験のため、給電状態変化（含むドックイン・アウト）で起きる
                    SetWakeReasonFlagSet(WakeReasonFlag::AcOk::Mask);
                    return PowerState_FullAwake;
                }
                case PowerStateMessage_EventSdCardStateChanged:
                {
                    // スリープに相当するユーザ体験のため、SD カード挿抜で起きる
                    SetWakeReasonFlagSet(WakeReasonFlag::SdCardDetect::Mask);
                    return PowerState_FullAwake;
                }
                case PowerStateMessage_EventGameCardStateChanged:
                {
                    // スリープに相当するユーザ体験のため、GC 挿抜で起きる
                    SetWakeReasonFlagSet(WakeReasonFlag::GameCardDetect::Mask);
                    return PowerState_FullAwake;
                }
                default:
                    return NoPowerStatechange;
            }
        }
    };

    // スリープ準備完了状態
    class PowerStateHandlerImplSleepReady final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplSleepReady() NN_NOEXCEPT : IPowerStateHandler(PowerState_SleepReady) {}
        ~PowerStateHandlerImplSleepReady() NN_NOEXCEPT {}

        virtual PowerState Entry() NN_NOEXCEPT
        {
            NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                ExecutePmRequestByPowerState(m_pContext->previousState, HandledState)
                );
            return NoPowerStatechange;
        }

        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            // これまでに受信済のメッセージをすべて処理する
            auto wakeReason = HandleAllRemainingPowerStateMessage();
            auto enterSc7 = !wakeReason.IsAnyOn(); // 既にウェイクトリガー相当のメッセージが届いていたか？

            // SoC スリープ直前に実行が必要な処理群を実行させる
            auto pmFlagsForEssentialServicesSleepReady = nn::psc::MakeNoPmFlags();
            if (!enterSc7)
            {
                // Workaround for NSBG-6205, SIGLO-41037
                // SC7 をスキップする場合、PmState_EssentialServicesSleepReady 遷移時にフラグを渡す
                pmFlagsForEssentialServicesSleepReady |= nn::psc::PmFlag::SkippingHardwareSleepAfterEssentialServicesSleepReady::Mask;
            }
            NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                ExecutePmRequest(nn::psc::PmState_EssentialServicesSleepReady, nn::psc::PmTransitionOrder_ToLowerPowerState, pmFlagsForEssentialServicesSleepReady)
            );

            if ( enterSc7 )
            {
                //
                // SC7 に入る
                //

                // 次に起きて MinimumAwake まで行ったときにアグリゲートされたウェイク要因を取得する
                RequireUpdateWakeReasonFlagSet(true);

                // settings の設定が false の場合は実際の SC7 には入らない。
                // ※ ここの段階ではみんな寝ていて settings の値を読めないので値は PowerState::Initialize で読んでおく
                if ( SettingsHolder::GetInstance().IsEnterSc7DebugSettingEnabled() )
                {
                    // NN_DETAIL_SPSM_INFO("Spsm entering SC7\n");
                    EnterSc7();
                }
                else
                {
                    NN_DETAIL_SPSM_INFO("Spsm sleeping 1 sec instead of SC7\n");
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                }
            }
            else
            {
                NN_DETAIL_SPSM_INFO("\n=== Skipped SC7 ===\n\n");
                // すでにウェイク要因が入っているので、SC7 に入らずすぐ起きることにする
                SetWakeReasonFlagSet(wakeReason);
                RequireUpdateWakeReasonFlagSet(false);
            }

            // SoC スリープからの復帰直後に実行が必要な処理群を実行させる
            NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                ExecutePmRequest(nn::psc::PmState_EssentialServicesAwake, nn::psc::PmTransitionOrder_ToHigherPowerState)
            );

            // 一旦 MinimumAwake を目的地に設定
            // その後の遷移先の決定は PowerState_MinimumAwake の OnDestination に到着してから行う
            return PowerState_MinimumAwake;
        }

    private:
        // メッセージをすべて処理してキューを空にする
        // すでにウェイクトリガーに相当するイベントが発生していたら、スリープをキャンセルして起きるようにする
        WakeReasonFlagSet HandleAllRemainingPowerStateMessage() NN_NOEXCEPT
        {
            PowerStateMessageWithMeta messageWithMeta;
            auto wakeReason = nn::spsm::WakeReasonFlagSet();

            while ( TryReceivePowerStateMessage(&messageWithMeta) )
            {
                switch ( messageWithMeta )
                {
                    case PowerStateMessage_EventPowerButtonStartedPressing:
                    case PowerStateMessage_EventPowerButtonPressedBriefly:
                    {
                        wakeReason |= WakeReasonFlag::PowerButtonPressed::Mask;
                        NN_DETAIL_SPSM_INFO("EventPowerButtonStartedPressing message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventShutdownRequiredByVeryLowBattery:
                    {
                        // タイミングによってはここで受信する可能性があり、黙って捨ててはいけない
                        wakeReason |= WakeReasonFlag::BatteryLevelTooLow::Mask;
                        NN_DETAIL_SPSM_INFO("EventShutdownRequiredByVeryLowBattery message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventHomeButtonStartedPressing:
                    case PowerStateMessage_EventHomeButtonPressedBriefly:
                    {
                        wakeReason |= WakeReasonFlag::BluetoothActivityDetect::Mask;
                        NN_DETAIL_SPSM_INFO("EventHomeButtonStartedPressing message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventPowerSupplyChanged:
                    {
                        wakeReason |= WakeReasonFlag::AcOk::Mask;
                        NN_DETAIL_SPSM_INFO("EventPowerSupplyChanged message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventSdCardStateChanged:
                    {
                        wakeReason |= WakeReasonFlag::SdCardDetect::Mask;
                        NN_DETAIL_SPSM_INFO("EventSdCardStateChanged message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventGameCardStateChanged:
                    {
                        wakeReason |= WakeReasonFlag::GameCardDetect::Mask;
                        NN_DETAIL_SPSM_INFO("EventGameCardStateChanged message was processed during SleepReady\n");
                        break;
                    }
                    case PowerStateMessage_EventFullAwakeRequestedByController:
                    {
                        wakeReason |= WakeReasonFlag::BluetoothActivityDetect::Mask;
                        NN_DETAIL_SPSM_INFO("PowerStateMessage_EventFullAwakeRequestedByController message was processed during SleepReady\n");
                        break;
                    }
                    default:
                    {
                        break;
                    }
                }
            }

            return wakeReason;
        }
    };

    // シャットダウン準備完了状態
    class PowerStateHandlerImplShutdownReady final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplShutdownReady() NN_NOEXCEPT : IPowerStateHandler(PowerState_ShutdownReady) {}
        ~PowerStateHandlerImplShutdownReady() NN_NOEXCEPT {}

        virtual PowerState Entry() NN_NOEXCEPT
        {
            NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(
                ExecutePmRequestByPowerState(m_pContext->previousState, HandledState)
            );
            return NoPowerStatechange;
        }

        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            ShutdownSystem();

            // Horizon では NEVER REACHED HERE
            // Windows ではテスタビリティのため返る
            return NoPowerStatechange;
        }
    };

    // 致命的エラーのときに退避してくる行き止まりステート
    class PowerStateHandlerImplError final : public IPowerStateHandler
    {
    public:
        PowerStateHandlerImplError() NN_NOEXCEPT : IPowerStateHandler(PowerState_Error) {}
        ~PowerStateHandlerImplError() NN_NOEXCEPT {}
    };

namespace {
    static nn::spsm::server::PowerStateHandlerImplInitial g_InitialHandler;
    static nn::spsm::server::PowerStateHandlerImplFullAwake g_FullAwakeHandler;
    static nn::spsm::server::PowerStateHandlerImplMinimumAwake g_MinimumAwakeHandler;
    static nn::spsm::server::PowerStateHandlerImplMinimumAwakeForLowBatterySign g_MinimumAwakeForLowBatterySignHandler;
    static nn::spsm::server::PowerStateHandlerImplMinimumAwakeForBackgroundTask g_MinimumAwakeForBackgroundTaskHandler;
    static nn::spsm::server::PowerStateHandlerImplSleepReady g_SleepReadyHandler;
    static nn::spsm::server::PowerStateHandlerImplShutdownReady g_ShutdownReadyHandler;
    static nn::spsm::server::PowerStateHandlerImplError g_ErrorHandler;
    nn::spsm::server::IPowerStateHandler* StateHandlers[] = {
        &g_InitialHandler,
        &g_FullAwakeHandler,
        &g_MinimumAwakeHandler,
        &g_MinimumAwakeForLowBatterySignHandler,
        &g_MinimumAwakeForBackgroundTaskHandler,
        &g_SleepReadyHandler,
        &g_ShutdownReadyHandler,
        &g_ErrorHandler
    };
}

    void InitializePowerStateHandlerImpl(PowerStateMessageQueue* pPowerStateMessageQueue, NotificationMessageQueue* pNotificationMessageQueue, nn::os::SystemEvent* pAwakeEvent, const PowerStateHandlerContext* pContext, const InitializeMode mode) NN_NOEXCEPT
    {
        // すべてのハンドラを初期化
        for (auto* handler : StateHandlers)
        {
            handler->SetPowerStateMessageQueue(pPowerStateMessageQueue);
            handler->SetNotificationMessageQueue(pNotificationMessageQueue);
            handler->SetAwakeEvent(pAwakeEvent);
            handler->SetContext(pContext);
        }

        // InitialHandler に対して、登録待ちする PSC のモジュールを設定
        g_InitialHandler.SetInitializeMode(mode);
    }

    IPowerStateHandler* GetPowerStateHandlerImpl(PowerState state) NN_NOEXCEPT
    {
        for (auto* handler : StateHandlers)
        {
            if (handler->HandledState == state)
            {
                return handler;
            }
        }
        NN_ABORT("Handler for state %d is not found\n", state);
    }

    PowerState DetermineNextState(PowerState current, PowerState dest) NN_NOEXCEPT
    {
        if ( current == dest || dest == PowerState_None )
        {
            return NoPowerStatechange;
        }
        if ( dest == PowerState_Error )
        {
            return dest;
        }
        switch (current)
        {
            case PowerState_None:       // 初期状態で訪れる
            case PowerState_Initial:    // 初期状態で訪れる
            {
                return dest;
            }
            case PowerState_FullAwake:
            {
                if ( dest == PowerState_MinimumAwake ||
                    dest == PowerState_SleepReady )
                {
                    return PowerState_MinimumAwake;
                }
                else if ( dest == PowerState_ShutdownReady )
                {
                    return PowerState_ShutdownReady;
                }
                break;
            }
            case PowerState_MinimumAwake:
            {
                if ( dest == PowerState_MinimumAwakeForLowBatterySign )
                {
                    return PowerState_MinimumAwakeForLowBatterySign;
                }
                else if ( dest == PowerState_MinimumAwakeForBackgroundTask )
                {
                    return PowerState_MinimumAwakeForBackgroundTask;
                }
                else if ( dest == PowerState_FullAwake )
                {
                    return PowerState_FullAwake;
                }
                else if ( dest == PowerState_SleepReady )
                {
                    return PowerState_SleepReady;
                }
                else if ( dest == PowerState_ShutdownReady )
                {
                    return PowerState_ShutdownReady;
                }
                break;
            }
            case PowerState_MinimumAwakeForLowBatterySign:
            {
                if ( dest == PowerState_MinimumAwake )
                {
                    // 現時点では存在しない想定の遷移
                    NN_SDK_ASSERT(false, "Unexpected transition\n");
                    return PowerState_MinimumAwake;
                }
                else if ( dest == PowerState_FullAwake )
                {
                    // 現時点では存在しない想定の遷移
                    NN_SDK_ASSERT(false, "Unexpected transition\n");
                    return PowerState_FullAwake;
                }
                else if ( dest == PowerState_SleepReady )
                {
                    return PowerState_SleepReady;
                }
                else if ( dest == PowerState_ShutdownReady )
                {
                    return PowerState_ShutdownReady;
                }
                break;
            }
            case PowerState_MinimumAwakeForBackgroundTask:
            {
                if ( dest == PowerState_MinimumAwakeForLowBatterySign )
                {
                    return PowerState_MinimumAwakeForLowBatterySign;
                }
                else if ( dest == PowerState_FullAwake )
                {
                    return PowerState_FullAwake;
                }
                else if ( dest == PowerState_SleepReady )
                {
                    return PowerState_SleepReady;
                }
                else if ( dest == PowerState_ShutdownReady )
                {
                    return PowerState_ShutdownReady;
                }
                break;
            }
            case PowerState_SleepReady:
            {
                if ( dest == PowerState_FullAwake ||
                     dest == PowerState_MinimumAwake )
                {
                    return PowerState_MinimumAwake;
                }
                // これは想定しない
                // else if ( dest == PowerState_ShutdownReady )
                // {
                //     return PowerState_ShutdownReady;
                // }
                break;
            }
            case PowerState_Error:  // 渡されないはず
            default:
                NN_UNEXPECTED_DEFAULT;
        }
        NN_ABORT("current(%d) / dest(%d)", current, dest); // NEVER REACHED HERE
    } // NOLINT(impl/function_size)

    void AbortAllPowerStateHandling() NN_NOEXCEPT
    {
        g_IsAbortingAllHandling = true;
    }
}}}

