﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/spsm/spsm_PowerStateTypes.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessage.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>
#include "spsm_NotificationMessageQueue.h"
#include "spsm_PowerStateHandlerContext.h"

#define NN_DETAIL_NS_STATE_HANDLER_GO_TO_ERROR_STATE_UNLESS(condition) \
    if (condition) {} \
    else \
    { \
        return ::nn::spsm::PowerState_Error; \
    }

#define NN_DETAIL_NS_STATE_HANDLER_RESULT_DO(r) \
    do \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if (!_nn_result_do_temporary.IsSuccess()) \
        { \
            NN_DETAIL_SPSM_ERROR( \
                "Failed: %s\n" \
                "  Module: %d\n" \
                "  Description: %d\n" \
                "  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(r), _nn_result_do_temporary.GetModule(), _nn_result_do_temporary.GetDescription(), _nn_result_do_temporary.GetInnerValueForDebug()); \
        } \
        NN_DETAIL_NS_STATE_HANDLER_GO_TO_ERROR_STATE_UNLESS(_nn_result_do_temporary.IsSuccess()); \
    } while (NN_STATIC_CONDITION(0))

namespace nn { namespace spsm { namespace server {

    const PowerState NoPowerStatechange = PowerState_None;

    class IPowerStateHandler
    {
    public:
        const PowerState HandledState;

    public:
        explicit IPowerStateHandler(PowerState handledState) NN_NOEXCEPT :
            HandledState(handledState),
            m_pContext(nullptr),
            m_pPowerStateMessageQueue(nullptr),
            m_pNotificationMessageQueue(nullptr),
            m_pAwakeEvent(nullptr)
        {}
        virtual ~IPowerStateHandler() NN_NOEXCEPT {}

        void SetPowerStateMessageQueue(PowerStateMessageQueue* pPowerStateMessageQueue)
        {
            m_pPowerStateMessageQueue = pPowerStateMessageQueue;
        }
        void SetNotificationMessageQueue(NotificationMessageQueue* pNotificationMessageQueue)
        {
            m_pNotificationMessageQueue = pNotificationMessageQueue;
        }
        void SetAwakeEvent(nn::os::SystemEvent* pAwakeEvent)
        {
            m_pAwakeEvent = pAwakeEvent;
        }
        void SetContext(const PowerStateHandlerContext* pContext)
        {
            m_pContext = pContext;
        }

        /**
         * @brief
         *  ステートに入る際に必ず実行される関数
         *
         * @return
         *  次の目的ステート。 @n
         *  NoPowerStatechange を返すと現在の目的ステートを維持します。 @n
         *  また、ハンドラに対応する現在のステートを返して目的ステートを上書きすると、 OnDestination() が直後に呼ばれます。
         */
        virtual PowerState Entry() NN_NOEXCEPT
        {
            return NoPowerStatechange;
        }

        /**
         * @brief
         *  ステートを出る際に必ず実行される関数
         */
        virtual void Exit() NN_NOEXCEPT
        {
        }

        /**
         * @brief
         *  ハンドラに対応する現在のステートが目的ステートに一致していたときに、Entry() の直後に呼ばれる関数。 @n
         *  目的ステートに一致していたが、直前の Entry() の戻り値でさらに別のステートを目的地とした場合は、この関数は呼ばれません。 @n
         *  この関数の開始時点で、 m_pContext->destinationState はステートマシンにより PowerState_None にクリアされています。
         *
         * @return
         *  次の目的ステート。 @n
         *  NoPowerStatechange を返すと現在の目的ステートを維持します。
         */
        virtual PowerState OnDestination() NN_NOEXCEPT
        {
            return NoPowerStatechange;
        }

        /**
         * @brief
         *  何らかのステートマシン外からのメッセージを受信したときに実行される関数
         *
         * @return
         *  次の目的ステート。 @n
         *  NoPowerStatechange を返すと現在の目的ステートを維持します。
         */
        virtual PowerState OnMessage(const PowerStateMessageWithMeta& messageWithMeta) NN_NOEXCEPT
        {
            NN_UNUSED(messageWithMeta);
            return NoPowerStatechange;
        }

    protected:
        void SendNotificationMessage(NotificationMessage message) NN_NOEXCEPT
        {
            m_pNotificationMessageQueue->Enqueue(message);
        }

        bool TryReceivePowerStateMessage(PowerStateMessageWithMeta* pOutMessageWithMeta) NN_NOEXCEPT
        {
            return m_pPowerStateMessageQueue->TryDequeue(pOutMessageWithMeta);
        }

        void ClearPowerStateMessageQueue() NN_NOEXCEPT
        {
            NN_DETAIL_SPSM_INFO("Reseting PowerStateMessageQueue\n");
            m_pPowerStateMessageQueue->ResetQueue();
        }

        void SignalAwakeEvent() NN_NOEXCEPT
        {
            m_pAwakeEvent->Signal();
        }

    protected:
        const PowerStateHandlerContext* m_pContext;

    private:
        // 派生クラスから直接いじらせたくない（Enqueue() など潜在的に危険なメソッドがあるため）
        PowerStateMessageQueue* m_pPowerStateMessageQueue;
        NotificationMessageQueue* m_pNotificationMessageQueue;
        nn::os::SystemEvent* m_pAwakeEvent;
    };

}}}

