﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/os/os_Tick.h>
#include <nn/psc.h>
#include <nn/psc/psc_PmControl.h>
#include <nn/spsm/spsm_Debug.h>
#include <nn/spsm/spsm_PowerStateTypes.h>
#include <nn/spsm/server/spsm_PowerStateMessage.h>

namespace nn { namespace spsm { namespace server {

    enum LogType
    {
        LogType_None,
        LogType_PscControl,
        LogType_PowerStateHandler,
        LogType_Sc7Entry,
        LogType_Sc7Exit,
    };

    struct PscControlLogItemParam
    {
        nn::psc::PmState pmState;
        nn::psc::PmTransitionOrder order;
        nn::psc::PmFlagSet flags;
    };

    enum PowerStateHandlerType
    {
        PowerStateHandlerType_Entry,
        PowerStateHandlerType_Exit,
        PowerStateHandlerType_OnDestination,
        PowerStateHandlerType_OnMessage,
    };

    struct PowerStateHandlerLogItemParam
    {
        nn::spsm::PowerState state;
        nn::spsm::PowerState newDestState;
        PowerStateHandlerType handlerType;
        PowerStateMessage message;
    };

    struct LogItem
    {
        LogType type;
        nn::os::Tick tick;
        union
        {
            PscControlLogItemParam pscControl;
            PowerStateHandlerLogItemParam powerStateHandler;
        } param;

        // コンストラクタ
        LogItem() NN_NOEXCEPT : type(LogType_None), tick(0)
        {
        }
        LogItem(LogType type, nn::os::Tick tick = static_cast<nn::os::Tick>(0)) NN_NOEXCEPT : type(type), tick(tick)
        {
        }
        LogItem(nn::psc::PmState pmState, nn::psc::PmTransitionOrder order, nn::psc::PmFlagSet flags, nn::os::Tick tick = static_cast<nn::os::Tick>(0)) NN_NOEXCEPT :
            type(LogType_PscControl), tick(tick)
        {
            param.pscControl.pmState = pmState;
            param.pscControl.order = order;
            param.pscControl.flags = flags;
        }
        LogItem(nn::spsm::PowerState state, nn::spsm::PowerState newDestState, PowerStateHandlerType handlerType, PowerStateMessage message, nn::os::Tick tick = static_cast<nn::os::Tick>(0)) NN_NOEXCEPT :
            type(LogType_PowerStateHandler), tick(tick)
        {
            param.powerStateHandler.state = state;
            param.powerStateHandler.newDestState = newDestState;
            param.powerStateHandler.handlerType = handlerType;
            param.powerStateHandler.message = message;
        }

        // 等価演算子オーバーロード
        bool operator==(const LogItem& rhs) const NN_NOEXCEPT
        {
            // tick は比較しない

            if (type != rhs.type)
            {
                return false;
            }
            switch (type)
            {
            case LogType_PscControl:
                if (param.pscControl.pmState == rhs.param.pscControl.pmState &&
                    param.pscControl.order == rhs.param.pscControl.order &&
                    param.pscControl.flags == rhs.param.pscControl.flags)
                {
                    return true;
                }
                break;
            case LogType_PowerStateHandler:
                if (param.powerStateHandler.state == rhs.param.powerStateHandler.state &&
                    param.powerStateHandler.newDestState == rhs.param.powerStateHandler.newDestState &&
                    param.powerStateHandler.handlerType == rhs.param.powerStateHandler.handlerType &&
                    param.powerStateHandler.message == rhs.param.powerStateHandler.message)
                {
                    return true;
                }
                break;
            default:
                return false;
            }
            return false;
        }
        inline bool operator!=(const LogItem& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }
    };

#if defined(NN_BUILD_CONFIG_OS_WIN)
    typedef std::vector<server::LogItem> LogItemList;
#else
    // 動的確保なしでそれっぽく使える vector もどき
    // 確保済領域をはみ出すときは記録しない
    class LogItemList
    {
    public:
        LogItemList() NN_NOEXCEPT : m_List(), m_Count() {}

        template< class... Args >
        void emplace_back(Args&&... args) NN_NOEXCEPT
        {
            if ( m_Count < LogItemCountMax )
            {
                m_List[m_Count] = std::forward<LogItem>(LogItem(args...));
                ++m_Count;
            }
        }

        void clear() NN_NOEXCEPT
        {
            m_Count = 0;
        }

        std::size_t size() const NN_NOEXCEPT
        {
            return m_Count;
        }

        inline LogItem operator[](std::size_t index) const NN_NOEXCEPT
        {
            return m_List[index];
        }

        inline LogItem& operator[](std::size_t index) NN_NOEXCEPT
        {
            return m_List[index];
        }

        bool operator==(const LogItemList& rhs) const NN_NOEXCEPT
        {
            for ( int i = 0; i < m_Count; ++i )
            {
                if ( m_List[i] != rhs[i] )
                {
                    return false;
                }
            }
            return true;
        }

    private:
        static const int LogItemCountMax = 128; // sizeof(LogItem) == 20 のとき 2560 B

    private:
        std::array<LogItem, LogItemCountMax> m_List;
        int m_Count;
    };
#endif

    const LogItemList& GetLog() NN_NOEXCEPT;
    void ResetLog() NN_NOEXCEPT;

    void LogPscControl(nn::psc::PmState pmState, nn::psc::PmTransitionOrder order, nn::psc::PmFlagSet flags) NN_NOEXCEPT;
    void LogPowerStateHandler(nn::spsm::PowerState state, nn::spsm::PowerState newDestState, PowerStateHandlerType handlerType, PowerStateMessage message = PowerStateMessage_None) NN_NOEXCEPT;
    void LogSc7Entry() NN_NOEXCEPT;
    void LogSc7Exit() NN_NOEXCEPT;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    const char* GetPowerStateHandlerTypeNameString(nn::spsm::server::PowerStateHandlerType handlerType) NN_NOEXCEPT;
#endif

    void AnalyzeLogForLastSleepWakeSequence(SleepWakeSequenceAnalyzedData* pOutData) NN_NOEXCEPT;
}}}

