﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  SD カード等の挿抜可能なストレージデバイスの挿抜イベントを監視する
//
class StorageDetection final : public IEventHandler
{
public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT NN_OVERRIDE;
    void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT NN_OVERRIDE
    {
        os::LinkMultiWaitHolder(p, &m_SdCardDetectEventHolder);
        os::LinkMultiWaitHolder(p, &m_GameCardDetectEventHolder);
        os::LinkMultiWaitHolder(p, &m_IgnoreStorageDetectionStartEventHolder);
        os::LinkMultiWaitHolder(p, &m_IgnoreStorageDetectionEndEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_SdCardDetectEventHolder);
        os::UnlinkMultiWaitHolder(&m_GameCardDetectEventHolder);
        os::UnlinkMultiWaitHolder(&m_IgnoreStorageDetectionStartEventHolder);
        os::UnlinkMultiWaitHolder(&m_IgnoreStorageDetectionEndEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( holder == &m_SdCardDetectEventHolder )
        {
            ProcessSdCardDetection();
            return true;
        }
        if ( holder == &m_GameCardDetectEventHolder )
        {
            ProcessGameCardDetection();
            return true;
        }
        if ( holder == &m_IgnoreStorageDetectionStartEventHolder )
        {
            ProcessIgnoreStorageDetectionStart();
            return true;
        }
        if ( holder == &m_IgnoreStorageDetectionEndEventHolder )
        {
            ProcessIgnoreStorageDetectionEnd();
            return true;
        }
        return false;
    }
    void OnPowerStateEntry(PowerState enteredState) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_PreviousState == PowerState_SleepReady )
        {
            m_IgnoreStorageDetectionEndEvent.Signal();
        }

        switch (enteredState)
        {
        case PowerState_SleepReady:
            {
                m_IgnoreStorageDetectionStartEvent.Signal();
                break;
            }
        default:
            {
                break;
            }
        }
    }
    void OnPowerStateExit(PowerState exitState) NN_NOEXCEPT NN_OVERRIDE
    {
        m_PreviousState = exitState;
    }
private:
    void ProcessSdCardDetection() NN_NOEXCEPT;
    void ProcessGameCardDetection() NN_NOEXCEPT;
    void ProcessIgnoreStorageDetectionStart() NN_NOEXCEPT;
    void ProcessIgnoreStorageDetectionEnd() NN_NOEXCEPT;

    void NotifySdCardStateChange() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("sd card state changed\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventSdCardStateChanged));
    }
    void NotifyGameCardStateChange() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("game card state changed\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventGameCardStateChanged));
    }

private:
    bool m_IgnoreStorageDetection { false };
    bool m_IsGameCardInserted     { false };
    bool m_IsSdCardInserted       { false };
    PowerState m_PreviousState    { PowerState_FullAwake };

    nn::spsm::server::PowerStateMessageQueue*    m_pPowerStateMessageQueue;

    std::unique_ptr<fs::IEventNotifier> m_SdCardNotifier;
    os::SystemEventType     m_SdCardDetectEvent;
    os::MultiWaitHolderType m_SdCardDetectEventHolder;
    os::Event               m_IgnoreStorageDetectionStartEvent { os::EventClearMode_AutoClear };
    os::Event               m_IgnoreStorageDetectionEndEvent   { os::EventClearMode_AutoClear };
    std::unique_ptr<fs::IEventNotifier> m_GameCardNotifier;
    os::SystemEventType     m_GameCardDetectEvent;
    os::MultiWaitHolderType m_GameCardDetectEventHolder;
    os::MultiWaitHolderType m_IgnoreStorageDetectionStartEventHolder;
    os::MultiWaitHolderType m_IgnoreStorageDetectionEndEventHolder;
};

//-----------------------------------------------------------------------------

}}} // namespace nn::spsm::observer

