﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_GameCard.h>
#include "spsm_StorageDetection.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  初期化
//
void StorageDetection::Initialize(nn::spsm::server::PowerStateMessageQueue* pPowerStateMessageQueue) NN_NOEXCEPT
{
    m_pPowerStateMessageQueue = pPowerStateMessageQueue;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenSdCardDetectionEventNotifier(&m_SdCardNotifier));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_SdCardNotifier->BindEvent(&m_SdCardDetectEvent, os::EventClearMode_AutoClear));
#else
    os::CreateSystemEvent(&m_SdCardDetectEvent, os::EventClearMode_AutoClear, false);
#endif
    os::ClearSystemEvent(&m_SdCardDetectEvent);
    os::InitializeMultiWaitHolder(&m_SdCardDetectEventHolder, &m_SdCardDetectEvent);
    os::SetMultiWaitHolderUserData(&m_SdCardDetectEventHolder, reinterpret_cast<uintptr_t>(this));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenGameCardDetectionEventNotifier(&m_GameCardNotifier));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_GameCardNotifier->BindEvent(&m_GameCardDetectEvent, os::EventClearMode_AutoClear));
#else
    os::CreateSystemEvent(&m_GameCardDetectEvent, os::EventClearMode_AutoClear, false);
#endif
    os::ClearSystemEvent(&m_GameCardDetectEvent);
    os::InitializeMultiWaitHolder(&m_GameCardDetectEventHolder, &m_GameCardDetectEvent);
    os::SetMultiWaitHolderUserData(&m_GameCardDetectEventHolder, reinterpret_cast<uintptr_t>(this));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    m_IsGameCardInserted = nn::fs::IsGameCardInserted();
    m_IsSdCardInserted   = nn::fs::IsSdCardInserted();
#endif

    os::InitializeMultiWaitHolder(&m_IgnoreStorageDetectionStartEventHolder, m_IgnoreStorageDetectionStartEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_IgnoreStorageDetectionEndEventHolder, m_IgnoreStorageDetectionEndEvent.GetBase());
    os::SetMultiWaitHolderUserData(&m_IgnoreStorageDetectionStartEventHolder, reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_IgnoreStorageDetectionEndEventHolder,   reinterpret_cast<uintptr_t>(this));
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
void StorageDetection::Finalize() NN_NOEXCEPT
{
    auto holders = { &m_SdCardDetectEventHolder,
                     &m_GameCardDetectEventHolder,
                     &m_IgnoreStorageDetectionStartEventHolder,
                     &m_IgnoreStorageDetectionEndEventHolder };
    for(auto &pHolder : holders)
    {
        os::FinalizeMultiWaitHolder(pHolder);
    }

    nn::os::DestroySystemEvent(&m_SdCardDetectEvent);
    nn::os::DestroySystemEvent(&m_GameCardDetectEvent);
}

//-----------------------------------------------------------------------------
//  StorageDetection のハンドラ関数（多重待ち解除時に呼ばれる）
//
void StorageDetection::ProcessSdCardDetection() NN_NOEXCEPT
{
    // NN_DETAIL_SPSM_INFO("!!!!! SD Card Event Caught !!!!!\n");
    if ( !nn::os::TryWaitSystemEvent(&m_SdCardDetectEvent) )
    {
        return;
    }

    if ( m_IgnoreStorageDetection )
    {
        // NN_DETAIL_SPSM_INFO("!!!!! SD Card Event Ignored (in ignore span) !!!!!\n");
        return;
    }

    auto isSdCardInserted = nn::fs::IsSdCardInserted();
    if ( m_IsSdCardInserted == isSdCardInserted )
    {
        // NN_DETAIL_SPSM_INFO("!!!!! SD Card Event Ignored (no state change) !!!!!\n");
        return;
    }

    m_IsSdCardInserted = isSdCardInserted;
    NotifySdCardStateChange();
}

void StorageDetection::ProcessGameCardDetection() NN_NOEXCEPT
{
    if ( !nn::os::TryWaitSystemEvent(&m_GameCardDetectEvent) )
    {
        return;
    }

    if ( m_IgnoreStorageDetection )
    {
        return;
    }

    auto isGameCardInserted = nn::fs::IsGameCardInserted();
    if ( m_IsGameCardInserted == isGameCardInserted )
    {
        return;
    }

    m_IsGameCardInserted = isGameCardInserted;
    NotifyGameCardStateChange();
}

void StorageDetection::ProcessIgnoreStorageDetectionStart() NN_NOEXCEPT
{
    // NN_DETAIL_SPSM_INFO("!!! Ignore Detection Start !!!\n");
    if( !m_IgnoreStorageDetectionStartEvent.TryWait() )
    {
        return;
    }
    m_IgnoreStorageDetection = true;
}
void StorageDetection::ProcessIgnoreStorageDetectionEnd() NN_NOEXCEPT
{
    // NN_DETAIL_SPSM_INFO("!!! Ignore Detection End !!!\n");
    if ( !m_IgnoreStorageDetectionEndEvent.TryWait() )
    {
        return;
    }
    m_IgnoreStorageDetection = false;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    m_IsGameCardInserted = nn::fs::IsGameCardInserted();
    m_IsSdCardInserted   = nn::fs::IsSdCardInserted();
#endif
}
//-----------------------------------------------------------------------------

}}} // namespace nn::spsm::observer

