﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psm/psm_SystemProcessTypes.h>
#include <nn/psm/psm_System.h>
#include <nn/spsm/spsm_PowerStateTypes.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  LowBattery 監視用クラス
//  apm から送られてくるシステム電圧低下および
//  psm から送られてくる電池残量僅少のイベントを監視する
//
class LowBattery final : public IEventHandler
{
public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_PowerSupplyChangeHolder);
        os::LinkMultiWaitHolder(p, &m_LowBatteryVoltageHolder);
        os::LinkMultiWaitHolder(p, &m_LowSystemVoltageHolder);
        os::LinkMultiWaitHolder(p, &m_ExplicitBatteryCheckEventHolder);
        os::LinkMultiWaitHolder(p, &m_IgnorePowerSupplyChangeStartEventHolder);
        os::LinkMultiWaitHolder(p, &m_IgnorePowerSupplyChangeEndEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_PowerSupplyChangeHolder);
        os::UnlinkMultiWaitHolder(&m_LowBatteryVoltageHolder);
        os::UnlinkMultiWaitHolder(&m_LowSystemVoltageHolder);
        os::UnlinkMultiWaitHolder(&m_ExplicitBatteryCheckEventHolder);
        os::UnlinkMultiWaitHolder(&m_IgnorePowerSupplyChangeStartEventHolder);
        os::UnlinkMultiWaitHolder(&m_IgnorePowerSupplyChangeEndEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT
    {
        if (holder == &m_PowerSupplyChangeHolder)
        {
            ProcessPowerSupplyChange();
            return true;
        }
        if (holder == &m_LowBatteryVoltageHolder)
        {
            ProcessBatteryVoltageChange();
            return true;
        }
        if (holder == &m_LowSystemVoltageHolder)
        {
            ProcessLowVoltage();
            return true;
        }
        if ( holder == &m_ExplicitBatteryCheckEventHolder )
        {
            ProcessExplicitBatteryCheck();
            return true;
        }
        if ( holder == &m_IgnorePowerSupplyChangeEndEventHolder )
        {
            ProcessIgnorePowerSupplyChangeEnd();
        }
        if ( holder == &m_IgnorePowerSupplyChangeStartEventHolder )
        {
            ProcessIgnorePowerSupplyChangeStart();
        }
        return false;
    }
    void OnPowerStateEntry(PowerState enteredState) NN_NOEXCEPT
    {
        switch ( enteredState )
        {
            // Entry 前に電池残量チェックがある FullAwake 以外で、長時間滞留する可能性があるステートでは
            // 遷移時に明示的にバッテリ残量を確認して、必要ならすぐに LowBattery などのメッセージを飛ばす必要がある
            case PowerState_MinimumAwakeForBackgroundTask:
            {
                m_ExplicitBatteryCheckEvent.Signal();
                break;
            }
            case PowerState_MinimumAwake:
            {
                if ( m_PreviousState == PowerState_SleepReady )
                {
                    m_IgnorePowerSupplyChangeEndEvent.Signal();
                }
                break;
            }
            case PowerState_SleepReady:
            {
                m_IgnorePowerSupplyChangeStartEvent.Signal();
                break;
            }
            default:
            {
                break;
            }
        }
    }
    void OnPowerStateExit(PowerState exitState) NN_NOEXCEPT
    {
        m_PreviousState = exitState;
    }
private:
    void ProcessPowerSupplyChange() NN_NOEXCEPT;
    void ProcessBatteryVoltageChange() NN_NOEXCEPT;
    void ProcessExplicitBatteryCheck() NN_NOEXCEPT;
    void CheckBatteryVoltageState() NN_NOEXCEPT;
    void ProcessIgnorePowerSupplyChangeStart() NN_NOEXCEPT;
    void ProcessIgnorePowerSupplyChangeEnd() NN_NOEXCEPT;

    void ProcessLowVoltage() NN_NOEXCEPT;

    void NotifyLowBattery() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("sleep required by low battery\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventSleepRequiredByLowBattery));
    }

    void NotifyVeryLowBattery() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("shutdown required by very low battery\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventShutdownRequiredByVeryLowBattery));
    }

    void NotifyPowerSupplyChange() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("power supply state changed\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventPowerSupplyChanged));
    }

private:
    nn::psm::ChargerType m_CurrentChargerType;
    PowerState           m_PreviousState { PowerState_FullAwake };
    bool                 m_IgnorePowerSupplyChange{ false };

    nn::spsm::server::PowerStateMessageQueue* m_pPowerStateMessageQueue;

    nn::psm::Session        m_PsmPowerSupplyDetectionSession;
    nn::psm::Session        m_PsmVoltageStateDetectionSession;
    os::SystemEventType     m_PowerSupplyChangeEvent;
    os::SystemEventType     m_VoltageStateChangeEvent;
    os::SystemEventType     m_SleepRequiredByLowVoltage;
    os::Event               m_ExplicitBatteryCheckEvent{ os::EventClearMode_AutoClear };
    os::Event               m_IgnorePowerSupplyChangeStartEvent{ os::EventClearMode_AutoClear };
    os::Event               m_IgnorePowerSupplyChangeEndEvent{ os::EventClearMode_AutoClear };
    os::MultiWaitHolderType m_PowerSupplyChangeHolder;
    os::MultiWaitHolderType m_LowBatteryVoltageHolder;
    os::MultiWaitHolderType m_LowSystemVoltageHolder;
    os::MultiWaitHolderType m_ExplicitBatteryCheckEventHolder;
    os::MultiWaitHolderType m_IgnorePowerSupplyChangeStartEventHolder;
    os::MultiWaitHolderType m_IgnorePowerSupplyChangeEndEventHolder;
};

}}} // namespace nn::spsm::observer

