﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/apm/apm_System.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>

#include "spsm_LowBattery.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  初期化
//
void LowBattery::Initialize(nn::spsm::server::PowerStateMessageQueue* pPowerStateMessageQueue) NN_NOEXCEPT
{
    m_pPowerStateMessageQueue = pPowerStateMessageQueue;

    nn::apm::InitializeForSystem();
    nn::apm::GetPerformanceEvent(&m_SleepRequiredByLowVoltage, nn::apm::EventTarget_SleepRequiredByLowVoltage, os::EventClearMode_AutoClear);

    nn::psm::Initialize();

    nn::psm::OpenSession(&m_PsmVoltageStateDetectionSession);
    nn::psm::SetChargerTypeChangeEventEnabled(&m_PsmVoltageStateDetectionSession, false);
    nn::psm::SetPowerSupplyChangeEventEnabled(&m_PsmVoltageStateDetectionSession, false);
    nn::psm::SetBatteryVoltageStateChangeEventEnabled(&m_PsmVoltageStateDetectionSession, true); // 電池電圧低下イベント取得のため
    nn::psm::BindStateChangeEvent(&m_VoltageStateChangeEvent, &m_PsmVoltageStateDetectionSession);

    nn::psm::OpenSession(&m_PsmPowerSupplyDetectionSession);
    nn::psm::SetChargerTypeChangeEventEnabled(&m_PsmPowerSupplyDetectionSession, true); // 電力供給能力の足りない AC アダプタでも反応させるため
    nn::psm::SetPowerSupplyChangeEventEnabled(&m_PsmPowerSupplyDetectionSession, true); // Type-C 挿抜イベント取得のため
    nn::psm::SetBatteryVoltageStateChangeEventEnabled(&m_PsmPowerSupplyDetectionSession, false);
    nn::psm::BindStateChangeEvent(&m_PowerSupplyChangeEvent, &m_PsmPowerSupplyDetectionSession);

    m_CurrentChargerType = nn::psm::GetChargerType();

    os::InitializeMultiWaitHolder(&m_LowSystemVoltageHolder, &m_SleepRequiredByLowVoltage);
    os::InitializeMultiWaitHolder(&m_LowBatteryVoltageHolder, &m_VoltageStateChangeEvent);
    os::InitializeMultiWaitHolder(&m_PowerSupplyChangeHolder, &m_PowerSupplyChangeEvent);
    os::InitializeMultiWaitHolder(&m_ExplicitBatteryCheckEventHolder, m_ExplicitBatteryCheckEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_IgnorePowerSupplyChangeStartEventHolder, m_IgnorePowerSupplyChangeStartEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_IgnorePowerSupplyChangeEndEventHolder, m_IgnorePowerSupplyChangeEndEvent.GetBase());

    os::SetMultiWaitHolderUserData(&m_LowSystemVoltageHolder,  reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_LowBatteryVoltageHolder, reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_PowerSupplyChangeHolder, reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_ExplicitBatteryCheckEventHolder, reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_IgnorePowerSupplyChangeStartEventHolder, reinterpret_cast<uintptr_t>(this));
    os::SetMultiWaitHolderUserData(&m_IgnorePowerSupplyChangeEndEventHolder, reinterpret_cast<uintptr_t>(this));
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
void LowBattery::Finalize() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_PowerSupplyChangeHolder);
    os::FinalizeMultiWaitHolder(&m_LowBatteryVoltageHolder);
    os::FinalizeMultiWaitHolder(&m_LowSystemVoltageHolder);
    os::FinalizeMultiWaitHolder(&m_ExplicitBatteryCheckEventHolder);
    os::FinalizeMultiWaitHolder(&m_IgnorePowerSupplyChangeStartEventHolder);
    os::FinalizeMultiWaitHolder(&m_IgnorePowerSupplyChangeEndEventHolder);

    nn::psm::UnbindStateChangeEvent(&m_PsmPowerSupplyDetectionSession);
    nn::psm::CloseSession(&m_PsmPowerSupplyDetectionSession);
    // psm::UnbindStateChangeEvent() 内で自動破棄されるので不要
    // nn::os::DestroySystemEvent(&m_PowerSupplyChangeEvent);

    nn::psm::UnbindStateChangeEvent(&m_PsmVoltageStateDetectionSession);
    nn::psm::CloseSession(&m_PsmVoltageStateDetectionSession);
    // psm::UnbindStateChangeEvent() 内で自動破棄されるので不要
    // nn::os::DestroySystemEvent(&m_VoltageStateChangeEvent);
    nn::psm::Finalize();

    nn::apm::FinalizeForSystem();
    nn::os::DestroySystemEvent(&m_SleepRequiredByLowVoltage);
}


//-----------------------------------------------------------------------------
//  LowBattery のハンドラ関数（多重待ち解除時に呼ばれる）
//
void LowBattery::ProcessPowerSupplyChange() NN_NOEXCEPT
{
    if ( !nn::os::TryWaitSystemEvent(&m_PowerSupplyChangeEvent) )
    {
        return;
    }
    nn::os::ClearSystemEvent(&m_PowerSupplyChangeEvent);

    if ( m_IgnorePowerSupplyChange )
    {
        return;
    }

    auto currentChargerType = nn::psm::GetChargerType();
    if ( currentChargerType == m_CurrentChargerType )
    {
        return;
    }
    m_CurrentChargerType = currentChargerType;
    NotifyPowerSupplyChange();
}

//-----------------------------------------------------------------------------
//  LowBattery のハンドラ関数（多重待ち解除時に呼ばれる）
//
void LowBattery::ProcessBatteryVoltageChange() NN_NOEXCEPT
{
    if ( !nn::os::TryWaitSystemEvent(&m_VoltageStateChangeEvent) )
    {
        return;
    }
    nn::os::ClearSystemEvent(&m_VoltageStateChangeEvent);
    CheckBatteryVoltageState();
}


void LowBattery::ProcessExplicitBatteryCheck() NN_NOEXCEPT
{
    if ( !m_ExplicitBatteryCheckEvent.TryWait() )
    {
        return;
    }
    CheckBatteryVoltageState();
}

void LowBattery::CheckBatteryVoltageState() NN_NOEXCEPT
{
    switch ( nn::psm::GetBatteryVoltageState() )
    {
        case nn::psm::BatteryVoltageState_ShutdownRequired:
            NotifyVeryLowBattery();
            break;
        case nn::psm::BatteryVoltageState_SleepRequired:
            NotifyLowBattery();
            break;
        default:
            break;
    }
}

void LowBattery::ProcessIgnorePowerSupplyChangeStart() NN_NOEXCEPT
{
    if ( !m_IgnorePowerSupplyChangeStartEvent.TryWait() )
    {
        return;
    }
    m_IgnorePowerSupplyChange = true;
}

void LowBattery::ProcessIgnorePowerSupplyChangeEnd() NN_NOEXCEPT
{
    if ( !m_IgnorePowerSupplyChangeEndEvent.TryWait() )
    {
        return;
    }
    m_IgnorePowerSupplyChange = false;
    m_CurrentChargerType = nn::psm::GetChargerType();
}
//-----------------------------------------------------------------------------
//  LowBattery のハンドラ関数（多重待ち解除時に呼ばれる）
//
void LowBattery::ProcessLowVoltage() NN_NOEXCEPT
{
    if ( !nn::os::TryWaitSystemEvent(&m_SleepRequiredByLowVoltage) )
    {
        return;
    }

    // 電池僅少を示していると考え LowBattery のイベントに集約する。
    NotifyLowBattery();
}

}}} // namespace nn::spsm::observer

