﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  HomeButton 監視用クラス
//
class HomeButton final : public IEventHandler
{
public:
    static TimeSpan GetTimeSpanShortPressing() NN_NOEXCEPT
    {
        return TimeSpan::FromMilliSeconds(30);
    }
    static TimeSpan GetTimeSpanLongPressing() NN_NOEXCEPT
    {
        return TimeSpan::FromMilliSeconds(500);
    }

public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_HomeButtonEventHolder);
        os::LinkMultiWaitHolder(p, &m_TimerEventHolder);
        os::LinkMultiWaitHolder(p, &m_WakeupEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_HomeButtonEventHolder);
        os::UnlinkMultiWaitHolder(&m_TimerEventHolder);
        os::UnlinkMultiWaitHolder(&m_WakeupEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT
    {
        if (holder == &m_TimerEventHolder)
        {
            TimerEventHandler();
            return true;
        }
        else if (holder == &m_HomeButtonEventHolder)
        {
            HomeButtonEventHandler();
            return true;
        }
        else if (holder == &m_WakeupEventHolder)
        {
            WakeupEventHandler();
            return true;
        }
        return false;
    }
    void OnPowerStateEntry(PowerState enteredState) NN_NOEXCEPT
    {
        switch ( enteredState )
        {
            case PowerState_FullAwake:
            {
                // スリープシーケンス終了時点で HOME ボタンを押下している場合は、
                // それを無効化（次回、再押下時から再び HOME ボタンが有効になる）。
                InvalidateCurrentButtonPressing( TimeSpan::FromMilliSeconds(500) );
                break;
            }
            default:
            {
                break;
            }
        }
    }

    void ChangeLongPressingTime(TimeSpan timeLongPressing) NN_NOEXCEPT;

private:
    void TimerEventHandler() NN_NOEXCEPT;
    void HomeButtonEventHandler() NN_NOEXCEPT;
    void WakeupEventHandler() NN_NOEXCEPT;
    void InvalidateCurrentButtonPressing(TimeSpan extensionTime) NN_NOEXCEPT;

    void NotifyStartPressingHomeButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("home button start pressing\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventHomeButtonStartedPressing));
    }

    void NotifyShortPressingHomeButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("home button short pressed\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventHomeButtonPressedBriefly));
    }

    void NotifyLongPressingHomeButton() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("home button long pressed\n");
        m_pPowerStateMessageQueue->Cancel(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventHomeButtonPressedBriefly)); // 本来同時に成立しないので、まだ処理されていないならキャンセルしておく
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventHomeButtonPressedLong));
    }

    void NotifyFullAwakeRequestByController() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("wakeup event received\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventFullAwakeRequestedByController));
    }
private:
    os::Mutex m_Mutex{ false };

    nn::spsm::server::PowerStateMessageQueue* m_pPowerStateMessageQueue;

    os::TimerEvent              m_TimerEvent{ os::EventClearMode_AutoClear };
    os::SystemEvent             m_HomeButtonEvent{ os::EventClearMode_AutoClear, true };
    os::SystemEvent             m_WakeupEvent{ os::EventClearMode_AutoClear, true };

    os::MultiWaitHolderType     m_TimerEventHolder;
    os::MultiWaitHolderType     m_HomeButtonEventHolder;
    os::MultiWaitHolderType     m_WakeupEventHolder;

    TimeSpan    m_TimeShortPressing;
    TimeSpan    m_TimeLongPressing;
    os::Tick    m_StartTick;
    os::Tick    m_EndTickToInvalidate{0};

    bool        m_State{ false };
};

//-----------------------------------------------------------------------------

}}} // namespace nn::spsm::observer

