﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/spsm/detail/spsm_Log.h>
#include <nn/spsm/server/spsm_PowerStateMessageQueue.h>

#include "spsm_IEventHandler.h"

namespace nn { namespace spsm { namespace observer {

//-----------------------------------------------------------------------------
//  BackgroundTask 監視用クラス
//  BGNUP などバックグラウンドのタスクの完了通知を監視
//
class BackgroundTask final : public IEventHandler
{
public:
    void Initialize(nn::spsm::server::PowerStateMessageQueue*) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_StateChangeEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_StateChangeEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder) NN_NOEXCEPT
    {
        if ( holder == &m_StateChangeEventHolder )
        {
            ProcessStateChange();
            return true;
        }
        return false;
    }
    void OnPowerStateEntry(PowerState enteredState) NN_NOEXCEPT
    {
        switch ( enteredState )
        {
            case PowerState_MinimumAwakeForBackgroundTask:
            {
                NotifyEnteringBackgroundTask();
                break;
            }
            default:
            {
                break;
            }
        }
    }
    void OnPowerStateExit(PowerState exitedState) NN_NOEXCEPT
    {
        switch ( exitedState )
        {
            case PowerState_MinimumAwakeForBackgroundTask:
            {
                NotifyLeavingBackgroundTask();
                break;
            }
            default:
            {
                break;
            }
        }
    }

private:
    void ProcessStateChange() NN_NOEXCEPT;
    void NotifyEnteringBackgroundTask() NN_NOEXCEPT;
    void NotifyLeavingBackgroundTask() NN_NOEXCEPT;

    void NotifyBackgroundTaskDone() NN_NOEXCEPT
    {
        // NN_DETAIL_SPSM_INFO_V1("background task done\n");
        m_pPowerStateMessageQueue->Enqueue(AddTimeStamp(nn::spsm::server::PowerStateMessage_EventBackgroundTaskDone));
    }

private:
    nn::spsm::server::PowerStateMessageQueue* m_pPowerStateMessageQueue;

    os::MultiWaitHolderType m_StateChangeEventHolder;

};

}}} // namespace nn::spsm::observer

