﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Xpad.h>

#include "settings_LockableMutexType.h"
#include "settings_XpadImpl.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad 用の割り当てを表す構造体です。
struct BasicXpadMap final
{
    //!< 汎用コントローラ間の割り当てが初期化済みか否かを表す値
    bool isGenericPadMapInitialized;

    //!< 汎用コントローラ間の割り当て
    BasicXpadGenericPadMap genericPadMap;

    //!< キーボード間の割り当て
    BasicXpadKeyboardMap keyboardMap;
};

//!< グローバルリソースの操作を排他するミューテックス
LockableMutexType g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

//!< 基本的な構成を持つ Xpad 用の割り当て
BasicXpadMap g_BasicXpadMaps[::nn::hid::XpadIdCountMax];

//!< プレイヤー番号が有効範囲に収まっているか否かを表す値を返します。
bool IsPlayerNumberInRange(int playerNumber) NN_NOEXCEPT
{
    return (0 <= playerNumber && playerNumber < ::nn::hid::XpadIdCountMax);
}

//!< 汎用コントローラの割り当ての規定値を取得します。
void GetDefaultGenericPadMap(BasicXpadGenericPadMap* outValue) NN_NOEXCEPT;

} // namespace

::nn::Result GetXpadGenericPadMap(BasicXpadGenericPadMap* outValue,
                                  int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ::nn::hid::ResultBasicXpadNullGenericPadMapBuffer());

    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    BasicXpadMap& map = g_BasicXpadMaps[playerNumber];

    if (!map.isGenericPadMapInitialized)
    {
        GetDefaultGenericPadMap(&map.genericPadMap);

        map.isGenericPadMapInitialized = true;
    }

    *outValue = map.genericPadMap;

    NN_RESULT_SUCCESS;
}

::nn::Result SetXpadGenericPadMap(const BasicXpadGenericPadMap& value,
                                  int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    BasicXpadMap& map = g_BasicXpadMaps[playerNumber];

    map.genericPadMap = value;

    map.isGenericPadMapInitialized = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ResetXpadGenericPadMap(int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    BasicXpadMap& map = g_BasicXpadMaps[playerNumber];

    int32_t genericPadId = map.genericPadMap.genericPadId;

    GetDefaultGenericPadMap(&map.genericPadMap);

    map.genericPadMap.genericPadId = genericPadId;

    map.isGenericPadMapInitialized = true;

    NN_RESULT_SUCCESS;
}

::nn::Result GetXpadKeyboardMap(BasicXpadKeyboardMap* outValue, int playerNumber
                                ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ::nn::hid::ResultBasicXpadNullKeyboardMapBuffer());

    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    *outValue = g_BasicXpadMaps[playerNumber].keyboardMap;

    NN_RESULT_SUCCESS;
}

::nn::Result SetXpadKeyboardMap(const BasicXpadKeyboardMap& value,
                                int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    g_BasicXpadMaps[playerNumber].keyboardMap = value;

    NN_RESULT_SUCCESS;
}

::nn::Result ResetXpadKeyboardMap(int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(IsPlayerNumberInRange(playerNumber),
                           ::nn::hid::ResultBasicXpadPlayerNumberOutOfRange());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    g_BasicXpadMaps[playerNumber].keyboardMap = BasicXpadKeyboardMap();

    NN_RESULT_SUCCESS;
}

namespace {

void GetDefaultGenericPadMap(BasicXpadGenericPadMap* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    outValue->genericPadId = 0;
    outValue->buttonA = GenericPadButton_2;
    outValue->buttonB = GenericPadButton_1;
    outValue->buttonX = GenericPadButton_4;
    outValue->buttonY = GenericPadButton_3;
    outValue->buttonStickL = GenericPadButton_9;
    outValue->buttonStickR = GenericPadButton_10;
    outValue->buttonL = GenericPadButton_5;
    outValue->buttonR = GenericPadButton_6;
    outValue->buttonZL = GenericPadButton_TriggerL;
    outValue->buttonZR = GenericPadButton_TriggerR;
    outValue->buttonStart = GenericPadButton_8;
    outValue->buttonSelect = GenericPadButton_7;
    outValue->buttonLeft = GenericPadButton_Left;
    outValue->buttonUp = GenericPadButton_Up;
    outValue->buttonRight = GenericPadButton_Right;
    outValue->buttonDown = GenericPadButton_Down;
    outValue->axisAnalogStickRX = GenericPadAxis_U;
    outValue->axisAnalogStickRY = GenericPadAxis_R;
    outValue->axisAnalogStickLX = GenericPadAxis_X;
    outValue->axisAnalogStickLY = GenericPadAxis_Y;
    outValue->attributes.Reset();
    outValue->attributes.Set<GenericPadAxisAttribute::IsXAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsYAxisInverted>(true);
    outValue->attributes.Set<GenericPadAxisAttribute::IsZAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsRAxisInverted>(true);
    outValue->attributes.Set<GenericPadAxisAttribute::IsUAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsVAxisInverted>(false);
}

} // namespace

}}} // namespace nn::settings::detail
