﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/fs/fs_SystemDataUpdateEvent.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "settings_SystemData-os.horizon.h"
#include "settings_SystemVersionDatabase.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< システムデータの識別子
const ::nn::ncm::SystemDataId SystemVersionSystemDataId = { 0x0100000000000809 };
const ::nn::ncm::SystemDataId RebootlessVersionSystemDataId = { 0x0100000000000826 };

//!< システムデータのマウント名
const char SystemVersionSystemDataMountName[] = "SystemVersion";
const char RebootlessVersionSystemDataMountName[] = "Rebootless";

//!< システムバージョンを表す構造体です。
struct SystemVersion final
{
    ::nn::settings::system::FirmwareVersion firmwareVersion;
    ::nn::settings::system::FirmwareVersionDigest firmwareVersionDigest;
};

//!< システムバージョンを取得します。
::nn::Result GetSystemVersion(const SystemVersion** ppOutValue) NN_NOEXCEPT;

//!< 再起動不要 NUP バージョンを取得します。
::nn::Result GetRebootlessVersion(const ::nn::settings::system::RebootlessSystemUpdateVersion** ppOutValue) NN_NOEXCEPT;

} // namespace

::nn::Result GetSystemVersionFirmwareVersion(
    ::nn::settings::system::FirmwareVersion* pOutValue, int port) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullFirmwareVersionBuffer());

    const SystemVersion* pSystemVersion = nullptr;

    NN_RESULT_DO(GetSystemVersion(&pSystemVersion));

    NN_SDK_ASSERT_NOT_NULL(pSystemVersion);

    *pOutValue = pSystemVersion->firmwareVersion;

    if (port == 1)
    {
        // 旧仕様とのバイナリ互換性を維持
        pOutValue->minorRelstep = 0;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemVersionFirmwareVersionDigest(
    ::nn::settings::system::FirmwareVersionDigest* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullFirmwareVersionDigestBuffer());

    const SystemVersion* pSystemVersion = nullptr;

    NN_RESULT_DO(GetSystemVersion(&pSystemVersion));

    NN_SDK_ASSERT_NOT_NULL(pSystemVersion);

    *pOutValue = pSystemVersion->firmwareVersionDigest;

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemRebootlessSystemUpdateVersion(
    ::nn::settings::system::RebootlessSystemUpdateVersion* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultNullRebootlessSystemUpdateVersionBuffer());

    const ::nn::settings::system::RebootlessSystemUpdateVersion* pRebootlessVersion = nullptr;

    NN_RESULT_DO(GetRebootlessVersion(&pRebootlessVersion));

    NN_SDK_ASSERT_NOT_NULL(pRebootlessVersion);

    *pOutValue = *pRebootlessVersion;

    NN_RESULT_SUCCESS;
}

namespace {

template<typename T>
::nn::Result ReadFile(T* pOutValue, const char* mountName, const char* fileName) NN_NOEXCEPT
{
    ::nn::fs::FileHandle fileHandle = {};

    {
        char path[32] = {};

        ::nn::util::SNPrintf(
            path, NN_ARRAY_SIZE(path), "%s:/%s", mountName, fileName);

        NN_RESULT_DO(
            ::nn::fs::OpenFile(&fileHandle, path, ::nn::fs::OpenMode_Read));
    }

    NN_UTIL_SCOPE_EXIT
    {
        ::nn::fs::CloseFile(fileHandle);
    };

    NN_RESULT_DO(::nn::fs::ReadFile(fileHandle, 0LL, pOutValue, sizeof(T)));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadSystemVersion(SystemVersion* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(::nn::fs::MountSystemData(SystemVersionSystemDataMountName, SystemVersionSystemDataId));

    NN_UTIL_SCOPE_EXIT
    {
        ::nn::fs::Unmount(SystemVersionSystemDataMountName);
    };

    NN_RESULT_DO(ReadFile(&pOutValue->firmwareVersion, SystemVersionSystemDataMountName, "file"));

    NN_RESULT_DO(ReadFile(&pOutValue->firmwareVersionDigest, SystemVersionSystemDataMountName, "digest"));

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemVersion(const SystemVersion** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static SystemVersion s_SystemVersion = {};

    static bool s_IsCached = false;

    if (!s_IsCached)
    {
        NN_RESULT_DO(ReadSystemVersion(&s_SystemVersion));

        s_IsCached = true;
    }

    *ppOutValue = &s_SystemVersion;

    NN_RESULT_SUCCESS;
}

::nn::Result ReadRebootlessVersion(::nn::settings::system::RebootlessSystemUpdateVersion* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(::nn::fs::MountSystemData(RebootlessVersionSystemDataMountName, RebootlessVersionSystemDataId));

    NN_UTIL_SCOPE_EXIT
    {
        ::nn::fs::Unmount(RebootlessVersionSystemDataMountName);
    };

    NN_RESULT_DO(ReadFile(pOutValue, RebootlessVersionSystemDataMountName, "version"));

    NN_RESULT_SUCCESS;
}

::nn::Result GetRebootlessVersion(const ::nn::settings::system::RebootlessSystemUpdateVersion** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::settings::system::RebootlessSystemUpdateVersion s_RebootlessVersion = {};
    static ::std::unique_ptr<fs::IEventNotifier> s_Notifier;
    static ::nn::os::SystemEventType s_UpdateEvent;

    static bool s_IsCached = false;

    if (!s_IsCached)
    {
        NN_RESULT_DO(::nn::fs::OpenSystemDataUpdateEventNotifier(&s_Notifier));
        NN_RESULT_DO(s_Notifier->BindEvent(&s_UpdateEvent, os::EventClearMode_AutoClear));

        NN_RESULT_DO(ReadRebootlessVersion(&s_RebootlessVersion));

        s_IsCached = true;
    }
    else if(::nn::os::TryWaitSystemEvent(&s_UpdateEvent))
    {
        NN_RESULT_DO(ReadRebootlessVersion(&s_RebootlessVersion));
    }

    *ppOutValue = &s_RebootlessVersion;

    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
