﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace settings { namespace detail {

//!< システムセーブデータを扱うクラスです。
class SystemSaveData final
{
private:
    //!< システムセーブデータの識別子
    ::nn::fs::SystemSaveDataId m_SystemSaveDataId;

    //!< システムセーブデータの合計サイズ
    int64_t m_TotalSize;

    //!< システムセーブデータのジャーナルサイズ
    int64_t m_JournalSize;

    //!< システムセーブデータのフラグ集合
    uint32_t m_Flags;

    //!< システムセーブデータのマウント名
    char m_MountName[::nn::fs::MountNameLengthMax + 1];

public:
    SystemSaveData() NN_NOEXCEPT;

    //!< システムセーブデータの識別子を設定します。
    void SetSystemSaveDataId(::nn::fs::SystemSaveDataId value) NN_NOEXCEPT;

    //!< システムセーブデータの合計サイズを設定します。
    void SetTotalSize(int64_t value) NN_NOEXCEPT;

    //!< システムセーブデータのジャーナルサイズを設定します。
    void SetJournalSize(int64_t value) NN_NOEXCEPT;

    //!< システムセーブデータのフラグ集合を設定します。
    void SetFlags(uint32_t value) NN_NOEXCEPT;

    //!< システムセーブデータのマウント名を設定します。
    void SetMountName(const char* name) NN_NOEXCEPT;

    //!< システムセーブデータをインメモリモードを有効にします。（デバッグ用）
    ::nn::Result EnableInMemoryModeForDebug(
        void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータをマウントします。
    ::nn::Result Mount(bool creates) NN_NOEXCEPT;

    //!< システムセーブデータの更新をコミットします。
    ::nn::Result Commit(bool synchronizes) NN_NOEXCEPT;

    //!< システムセーブデータを作成します。
    ::nn::Result Create(int64_t size) NN_NOEXCEPT;

    //!< システムセーブデータを読み込み権限で開きます。
    ::nn::Result OpenToRead() NN_NOEXCEPT;

    //!< システムセーブデータを書き込み権限で開きます。
    ::nn::Result OpenToWrite() NN_NOEXCEPT;

    //!< システムセーブデータを閉じます。
    void Close() NN_NOEXCEPT;

    //!< システムセーブデータから読み込みます。
    ::nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータへ書き込みます。
    ::nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

    //!< システムセーブデータへの書き込みをフラッシュします。
    ::nn::Result Flush() NN_NOEXCEPT;
};

}}} // namespace nn::settings::detail
