﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "settings_StaticObject.h"
#include "settings_SystemSaveData-os.horizon.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< マウント名のセパレータ
const char MountNameSeparator[] = ":";

//!< ディレクトリ名のセパレータ
const char DirectoryNameSeparator[] = "/";

//!< システムセーブデータのファイル名
const char SystemSaveDataFileName[] = "file";

//!< 遅延書き込みの遅延時間 (ミリ秒)
const int64_t LazyWriterDelay = 500;

//!< 遅延ファイルアクセスを扱うクラスです。
class LazyFileAccessor final
{
    NN_DISALLOW_COPY(LazyFileAccessor);
    NN_DISALLOW_MOVE(LazyFileAccessor);

private:
    //!< 遅延アクセスの対象となるファイルの名前の最大長
    static const int FileNameLengthMax = 31;

    //!< アクティブ状態にあるかか否か
    bool m_IsActiveted;

    //!< 遅延アクセス中か否か
    bool m_IsBusy;

    //!< キャッシュが有効か否か
    bool m_IsCached;

    //!< バッファの内容が変更されたか否か
    bool m_IsModified;

    //!< 遅延書き込み対象のマウント名
    char m_MountName[::nn::fs::MountNameLengthMax + 1];

    //!< 遅延書き込み対象のファイルパス
    char m_FilePath[::nn::fs::MountNameLengthMax + 2 + FileNameLengthMax + 1];

    //!< ファイルオープン時のモード
    int m_OpenMode;

    //!< ファイルサイズ
    int64_t m_FileSize;

    //!< 遅延書き込みの開始位置
    int64_t m_Offset;

    //!< 遅延書き込みのバイトサイズ
    size_t m_Size;

    //!< バッファ
    ::nn::Bit8 m_Buffer[512 << 10];

    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< タイマーイベント
    ::nn::os::TimerEvent m_TimerEvent;

    //!< スレッドオブジェクト
    ::nn::os::ThreadType m_Thread;

    //!< スレッドのスタック領域
    NN_ALIGNAS(0x1000) char m_ThreadStack[0x1000];

public:
    LazyFileAccessor() NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< ファイルの更新をコミットします。
    ::nn::Result Commit(const char name[], bool synchronizes) NN_NOEXCEPT;

    //!< ファイルを作成します。
    ::nn::Result Create(const char name[], int64_t size) NN_NOEXCEPT;

    //!< ファイルをオープンします。
    ::nn::Result Open(const char name[], int mode) NN_NOEXCEPT;

    //!< ファイルをクローズします。
    void Close() NN_NOEXCEPT;

    //!< ファイルから読み込みます。
    ::nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    //!< ファイルへ書き込みます。
    ::nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

private:
    //!< ファイルパスを作成します。
    template<size_t N>
    static int CreateFilePath(char (&path)[N], const char name[]) NN_NOEXCEPT;

    //!< メモリ領域を比較します。
    static bool AreEqual(
        const void* lhs, const void* rhs, size_t size) NN_NOEXCEPT;

    //!< スレッドの関数エントリ
    static void ThreadFunc(void* value) NN_NOEXCEPT;

    //!< マウント名を設定します。
    void SetMountName(const char name[]) NN_NOEXCEPT;

    //!< マウント名を比較します。
    bool CompareMountName(const char name[]) const NN_NOEXCEPT;

    //!< 遅延書き込みの受付を開始します。
    void InvokeWriteBackLoop() NN_NOEXCEPT;

    //!< ファイルの更新を同期的にコミットします。
    ::nn::Result CommitSynchronously() NN_NOEXCEPT;
};

//!< 遅延ファイルアクセッサを返します。
LazyFileAccessor& GetLazyFileAccessor() NN_NOEXCEPT;

} // namespace

SystemSaveData::SystemSaveData() NN_NOEXCEPT
    : m_SystemSaveDataId()
    , m_TotalSize()
    , m_JournalSize()
    , m_Flags()
{
    // 何もしない
}

void SystemSaveData::SetSystemSaveDataId(::nn::fs::SystemSaveDataId value
                                         ) NN_NOEXCEPT
{
    m_SystemSaveDataId = value;
}

void SystemSaveData::SetTotalSize(int64_t value) NN_NOEXCEPT
{
    m_TotalSize = value;
}

void SystemSaveData::SetJournalSize(int64_t value) NN_NOEXCEPT
{
    m_JournalSize = value;
}

void SystemSaveData::SetFlags(uint32_t value) NN_NOEXCEPT
{
    m_Flags = value;
}

void SystemSaveData::SetMountName(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    ::nn::util::Strlcpy(
        m_MountName,
        name,
        static_cast<int>(::std::extent<decltype(m_MountName)>::value));
}

::nn::Result SystemSaveData::EnableInMemoryModeForDebug(
    void* buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(buffer);
    NN_UNUSED(size);
    NN_RESULT_THROW(ResultUnsupportedOperation());
}

::nn::Result SystemSaveData::Mount(bool creates) NN_NOEXCEPT
{
    NN_RESULT_DO(GetLazyFileAccessor().Activate());

    ::nn::fs::DisableAutoSaveDataCreation();

    const ::nn::Result result =
        ::nn::fs::MountSystemSaveData(m_MountName, m_SystemSaveDataId);

    NN_RESULT_THROW_UNLESS(creates || result.IsSuccess(), result);

    if (result.IsFailure())
    {
        NN_RESULT_DO(
            ::nn::fs::CreateSystemSaveData(
                m_SystemSaveDataId, m_TotalSize, m_JournalSize, m_Flags));

        NN_RESULT_DO(
            ::nn::fs::MountSystemSaveData(m_MountName, m_SystemSaveDataId));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Commit(bool synchronizes) NN_NOEXCEPT
{
    NN_RESULT_DO(GetLazyFileAccessor().Commit(m_MountName, synchronizes));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Create(int64_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(GetLazyFileAccessor().Create(m_MountName, size));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::OpenToRead() NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetLazyFileAccessor().Open(m_MountName, ::nn::fs::OpenMode_Read));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::OpenToWrite() NN_NOEXCEPT
{
    NN_RESULT_DO(
        GetLazyFileAccessor().Open(m_MountName, ::nn::fs::OpenMode_Write));
    NN_RESULT_SUCCESS;
}

void SystemSaveData::Close() NN_NOEXCEPT
{
    GetLazyFileAccessor().Close();
}

::nn::Result SystemSaveData::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_RESULT_DO(GetLazyFileAccessor().Read(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Write(
    int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_RESULT_DO(GetLazyFileAccessor().Write(offset, buffer, size));
    NN_RESULT_SUCCESS;
}

::nn::Result SystemSaveData::Flush() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

namespace {

LazyFileAccessor::LazyFileAccessor() NN_NOEXCEPT
    : m_IsActiveted(false)
    , m_IsBusy(false)
    , m_IsCached(false)
    , m_IsModified(false)
    , m_OpenMode()
    , m_FileSize()
    , m_Offset()
    , m_Size()
    , m_Mutex(false)
    , m_TimerEvent(::nn::os::EventClearMode_AutoClear)
    , m_Thread()
{
    for (char& c : m_MountName)
    {
        c = '\0';
    }

    for (char& c : m_FilePath)
    {
        c = '\0';
    }

    ::std::memset(m_Buffer, 0, sizeof(m_Buffer));

    ::std::memset(m_ThreadStack, 0, sizeof(m_ThreadStack));
}

::nn::Result LazyFileAccessor::Activate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsActiveted)
    {
        NN_RESULT_DO(
            ::nn::os::CreateThread(
                &m_Thread,
                LazyFileAccessor::ThreadFunc,
                this,
                m_ThreadStack,
                sizeof(m_ThreadStack),
                NN_SYSTEM_THREAD_PRIORITY(settings, LazyWriter)));

        ::nn::os::SetThreadNamePointer(
            &m_Thread,
            NN_SYSTEM_THREAD_NAME(settings, LazyWriter));

        ::nn::os::StartThread(&m_Thread);

        m_IsActiveted = true;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result LazyFileAccessor::Commit(
    const char name[], bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(!m_IsBusy);
    NN_SDK_REQUIRES(this->CompareMountName(name));

    NN_UNUSED(name);

    if (synchronizes)
    {
        m_TimerEvent.Stop();

        NN_RESULT_DO(this->CommitSynchronously());
    }
    else
    {
        m_TimerEvent.StartOneShot(
            ::nn::TimeSpan::FromMilliSeconds(LazyWriterDelay));
    }

    NN_RESULT_SUCCESS;
}

::nn::Result LazyFileAccessor::Create(
    const char name[], int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(!m_IsBusy);

    if (m_IsCached)
    {
        m_TimerEvent.Stop();

        NN_RESULT_DO(this->CommitSynchronously());

        m_IsCached = false;

        this->SetMountName("");

        m_OpenMode = int();

        m_FileSize = int64_t();

        ::std::memset(m_Buffer, 0, sizeof(m_Buffer));
    }

    CreateFilePath(m_FilePath, name);

    NN_RESULT_DO(::nn::fs::CreateFile(m_FilePath, size));

    m_IsCached = true;

    m_IsModified = true;

    this->SetMountName(name);

    m_OpenMode = ::nn::fs::OpenMode_Write;

    m_FileSize = size;

    m_Offset = 0;

    m_Size = size;

    m_TimerEvent.StartOneShot(
        ::nn::TimeSpan::FromMilliSeconds(LazyWriterDelay));

    NN_RESULT_SUCCESS;
}

::nn::Result LazyFileAccessor::Open(const char name[], int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(!m_IsBusy);

    bool caches = true;

    if (m_IsCached)
    {
        if (this->CompareMountName(name))
        {
            if (m_OpenMode == mode || m_OpenMode == ::nn::fs::OpenMode_Read)
            {
                m_IsBusy = true;

                m_OpenMode = mode;

                NN_RESULT_SUCCESS;
            }

            caches = false;
        }

        m_TimerEvent.Stop();

        NN_RESULT_DO(this->CommitSynchronously());
    }

    if (caches)
    {
        ::nn::fs::FileHandle fileHandle = {};

        CreateFilePath(m_FilePath, name);

        NN_RESULT_DO(
            ::nn::fs::OpenFile(
                &fileHandle, m_FilePath, ::nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            ::nn::fs::CloseFile(fileHandle);
        };

        auto fileSize = int64_t();

        NN_RESULT_DO(
            ::nn::fs::GetFileSize(&fileSize, fileHandle));

        NN_SDK_ASSERT_MINMAX(
            fileSize, 0LL, static_cast<int64_t>(sizeof(m_Buffer)));

        NN_RESULT_DO(
            ::nn::fs::ReadFile(
                fileHandle, 0LL, m_Buffer, static_cast<size_t>(fileSize)));

        m_IsCached = true;

        this->SetMountName(name);

        m_FileSize = fileSize;
    }

    m_IsBusy = true;

    m_OpenMode = mode;

    NN_RESULT_SUCCESS;
}

void LazyFileAccessor::Close() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(m_IsBusy);

    m_IsBusy = false;
}

::nn::Result LazyFileAccessor::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(m_IsBusy);
    NN_SDK_REQUIRES(m_IsCached);
    NN_SDK_REQUIRES((m_OpenMode & ::nn::fs::OpenMode_Read) != 0);
    NN_SDK_REQUIRES_LESS_EQUAL(
        offset + static_cast<int64_t>(size), m_FileSize);

    ::std::memcpy(buffer, &m_Buffer[offset], size);

    NN_RESULT_SUCCESS;
}

::nn::Result LazyFileAccessor::Write(
    int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsActiveted);
    NN_SDK_REQUIRES(m_IsBusy);
    NN_SDK_REQUIRES(m_IsCached);
    NN_SDK_REQUIRES((m_OpenMode & ::nn::fs::OpenMode_Write) != 0);

    int64_t tail = offset + static_cast<int64_t>(size);

    NN_SDK_REQUIRES_LESS_EQUAL(tail, m_FileSize);

    if (!AreEqual(&m_Buffer[offset], buffer, size))
    {
        ::std::memcpy(&m_Buffer[offset], buffer, size);

        if (!m_IsModified)
        {
            m_IsModified = true;

            m_Offset = offset;

            m_Size = size;
        }
        else
        {
            tail = ::std::max(tail, m_Offset + static_cast<int64_t>(m_Size));

            m_Offset = ::std::min(m_Offset, offset);

            m_Size = static_cast<size_t>(tail - m_Offset);
        }
    }

    NN_RESULT_SUCCESS;
}

template<size_t N>
int LazyFileAccessor::CreateFilePath(
    char (&path)[N], const char name[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    const auto length = static_cast<int>(N);

    int position = ::nn::util::Strlcpy(path, name, length);

    position += ::nn::util::Strlcpy(
        &path[position], MountNameSeparator, length - position);

    position += ::nn::util::Strlcpy(
        &path[position], DirectoryNameSeparator, length - position);

    ::nn::util::Strlcpy(
        &path[position], SystemSaveDataFileName, length - position);

    return position;
}

bool LazyFileAccessor::AreEqual(
    const void* lhs, const void* rhs, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(lhs);
    NN_SDK_REQUIRES_NOT_NULL(rhs);

    auto lhs1 = reinterpret_cast<const ::nn::Bit8*>(lhs);

    auto rhs1 = reinterpret_cast<const ::nn::Bit8*>(rhs);

    for (size_t i = 0; i < size; ++i)
    {
        if (*lhs1 != *rhs1)
        {
            return false;
        }

        ++lhs1;

        ++rhs1;
    }

    return true;
}

void LazyFileAccessor::ThreadFunc(void* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    reinterpret_cast<LazyFileAccessor*>(value)->InvokeWriteBackLoop();
}

void LazyFileAccessor::SetMountName(const char name[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    const auto count =
        static_cast<int>(::std::extent<decltype(m_MountName)>::value);

    ::nn::util::Strlcpy(m_MountName, name, count);
}

bool LazyFileAccessor::CompareMountName(const char name[]) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    const auto count =
        static_cast<int>(::std::extent<decltype(m_MountName)>::value);

    return (::nn::util::Strncmp(m_MountName, name, count) == 0);
}

void LazyFileAccessor::InvokeWriteBackLoop() NN_NOEXCEPT
{
    while (NN_STATIC_CONDITION(true))
    {
        m_TimerEvent.Wait();

        ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

        if (!m_IsBusy)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(this->CommitSynchronously());
        }
    }
}

::nn::Result LazyFileAccessor::CommitSynchronously() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsBusy);

    auto commits = false;

    if (m_IsCached && m_IsModified)
    {
        ::nn::fs::FileHandle fileHandle = {};

        CreateFilePath(m_FilePath, m_MountName);

        NN_RESULT_DO(
            ::nn::fs::OpenFile(
                &fileHandle, m_FilePath, ::nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            ::nn::fs::CloseFile(fileHandle);
        };

        const ::nn::fs::WriteOption option = {};

        NN_RESULT_DO(
            ::nn::fs::WriteFile(
                fileHandle, m_Offset, &m_Buffer[m_Offset], m_Size, option));

        NN_RESULT_DO(::nn::fs::FlushFile(fileHandle));

        m_IsModified = false;

        m_Offset = int64_t();

        m_Size = size_t();

        commits = true;
    }

    if (commits)
    {
        NN_RESULT_DO(::nn::fs::CommitSaveData(m_MountName));
    }

    NN_RESULT_SUCCESS;
}

LazyFileAccessor& GetLazyFileAccessor() NN_NOEXCEPT
{
    return StaticObject<LazyFileAccessor>::Get();
}

} // namespace

}}} // namespace nn::settings::detail
