﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <iterator>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Windows.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "settings_SystemDataFile-os.win.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< 既定のディレクトリパス
const char DefaultDirectoryPath[] = ".\\";

//!< 試行回数
const int TrialCount = 1000;

//!< 試行間隔
const ::nn::TimeSpan TrialInterval = ::nn::TimeSpan::FromMilliSeconds(10);

//!< ミューテックスを初期化します。
::nn::Result InitializeMutex(HANDLE* pOutHandle, const char name[]) NN_NOEXCEPT;

//!< ミューテックスを破棄します。
void FinalizeMutex(HANDLE handle) NN_NOEXCEPT;

//!< ミューテックスをロックします。
void LockMutex(HANDLE handle) NN_NOEXCEPT;

//!< ミューテックスを解放します。
void UnlockMutex(HANDLE handle) NN_NOEXCEPT;

//!< ファイルを作成します。
::nn::Result CreateFile(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT;

//!< ファイルを読み込み用に開きます。
::nn::Result OpenFileToRead(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT;

//!< ファイルを書き込み用に開きます。
::nn::Result OpenFileToWrite(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT;

//!< ファイルを閉じます。
void CloseFile(HANDLE handle) NN_NOEXCEPT;

//!< ファイルのポインタ位置を移動します。
::nn::Result SetFilePointer(HANDLE handle, int64_t offset) NN_NOEXCEPT;

//!< ファイルのポインタ位置に EOF を設定します。
::nn::Result SetFileEof(HANDLE handle) NN_NOEXCEPT;

//!< ファイルを読み込みます。
::nn::Result ReadFile(
    HANDLE handle, void* buffer, size_t bufferSize) NN_NOEXCEPT;

//!< ファイルを書き込みます。
::nn::Result WriteFile(
    HANDLE handle, const void* buffer, size_t bufferSize) NN_NOEXCEPT;

//!< ファイルをフラッシュします。
::nn::Result FlushFile(HANDLE handle) NN_NOEXCEPT;

} // namespace

SystemDataFile::SystemDataFile() NN_NOEXCEPT
    : m_MutexHandle()
    , m_FileHandle()
{
    ::std::fill(::std::begin(m_Prefix), ::std::end(m_Prefix), '\0');

    ::std::fill(::std::begin(m_MutexName), ::std::end(m_MutexName), '\0');

    ::std::fill(::std::begin(m_FilePath), ::std::end(m_FilePath), '\0');
}

SystemDataFile::SystemDataFile(const SystemDataFile& that) NN_NOEXCEPT
    : m_MutexHandle(that.m_MutexHandle)
    , m_FileHandle(that.m_FileHandle)
{
    ::std::copy(
        ::std::begin(that.m_Prefix), ::std::end(that.m_Prefix),
        ::std::begin(m_Prefix));

    ::std::copy(
        ::std::begin(that.m_MutexName), ::std::end(that.m_MutexName),
        ::std::begin(m_MutexName));

    ::std::copy(
        ::std::begin(that.m_FilePath), ::std::end(that.m_FilePath),
        ::std::begin(m_FilePath));
}

void SystemDataFile::SetPrefix(const char* prefix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(prefix);

    ::nn::util::Strlcpy(
        m_Prefix, prefix, static_cast<int>(NN_ARRAY_SIZE(m_Prefix)));
}

void SystemDataFile::SetMountName(const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(name);

    const auto MutexNameSize = static_cast<int>(NN_ARRAY_SIZE(m_MutexName));

    const auto PrefixSize = static_cast<int>(NN_ARRAY_SIZE(m_Prefix));

    int offset = ::nn::util::Strlcpy(
        m_MutexName, m_Prefix, ::std::min(MutexNameSize, PrefixSize));

    const int MountNameSize = ::nn::fs::MountNameLengthMax + 1;

    ::nn::util::Strlcpy(
        &m_MutexName[offset], name,
        ::std::min(MutexNameSize - offset, MountNameSize));

    const auto FilePathSize = static_cast<int>(NN_ARRAY_SIZE(m_FilePath));

    offset = ::GetTempPathA(static_cast<DWORD>(FilePathSize), m_FilePath);

    if (offset == 0)
    {
        offset = ::nn::util::Strlcpy(
            m_FilePath, DefaultDirectoryPath,
            static_cast<int>(NN_ARRAY_SIZE(DefaultDirectoryPath)));
    }

    ::nn::util::Strlcpy(
        &m_FilePath[offset], m_MutexName,
        ::std::min(FilePathSize - offset, MutexNameSize));
}

::nn::Result SystemDataFile::Mount(bool creates) NN_NOEXCEPT
{
    if (!creates)
    {
        HANDLE mutexHandle = INVALID_HANDLE_VALUE;

        NN_RESULT_DO(InitializeMutex(&mutexHandle, m_MutexName));

        LockMutex(mutexHandle);

        NN_UTIL_SCOPE_EXIT
        {
            UnlockMutex(mutexHandle);

            FinalizeMutex(mutexHandle);
        };

        const DWORD dwAttrib = ::GetFileAttributesA(m_FilePath);

        NN_RESULT_THROW_UNLESS(
            (dwAttrib != INVALID_FILE_ATTRIBUTES) &&
            (dwAttrib & FILE_ATTRIBUTE_DIRECTORY) == 0,
            ::nn::fs::ResultTargetNotFound());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SystemDataFile::Create(int64_t size) NN_NOEXCEPT
{
    HANDLE mutexHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_DO(InitializeMutex(&mutexHandle, m_MutexName));

    LockMutex(mutexHandle);

    NN_UTIL_SCOPE_EXIT
    {
        UnlockMutex(mutexHandle);

        FinalizeMutex(mutexHandle);
    };

    HANDLE fileHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_DO(CreateFile(&fileHandle, m_FilePath));

    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(fileHandle);
    };

    NN_RESULT_THROW_UNLESS(
        SetFilePointer(fileHandle, size).IsSuccess(),
        ::nn::fs::ResultUsableSpaceNotEnough());

    NN_RESULT_THROW_UNLESS(
        SetFileEof(fileHandle).IsSuccess(),
        ::nn::fs::ResultUsableSpaceNotEnough());

    NN_RESULT_SUCCESS;
}

::nn::Result SystemDataFile::OpenToRead() NN_NOEXCEPT
{
    HANDLE mutexHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_DO(InitializeMutex(&mutexHandle, m_MutexName));

    LockMutex(mutexHandle);

    NN_UTIL_SCOPE_EXIT
    {
        if (mutexHandle != INVALID_HANDLE_VALUE)
        {
            UnlockMutex(mutexHandle);

            FinalizeMutex(mutexHandle);
        }
    };

    NN_RESULT_DO(OpenFileToRead(&m_FileHandle, m_FilePath));

    m_MutexHandle = mutexHandle;

    mutexHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_SUCCESS;
}

::nn::Result SystemDataFile::OpenToWrite() NN_NOEXCEPT
{
    HANDLE mutexHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_DO(InitializeMutex(&mutexHandle, m_MutexName));

    LockMutex(mutexHandle);

    NN_UTIL_SCOPE_EXIT
    {
        if (mutexHandle != INVALID_HANDLE_VALUE)
        {
            UnlockMutex(mutexHandle);

            FinalizeMutex(mutexHandle);
        }
    };

    NN_RESULT_DO(OpenFileToWrite(&m_FileHandle, m_FilePath));

    m_MutexHandle = mutexHandle;

    mutexHandle = INVALID_HANDLE_VALUE;

    NN_RESULT_SUCCESS;
}

void SystemDataFile::Close() NN_NOEXCEPT
{
    CloseFile(m_FileHandle);

    m_FileHandle = INVALID_HANDLE_VALUE;

    UnlockMutex(m_MutexHandle);

    FinalizeMutex(m_MutexHandle);

    m_MutexHandle = INVALID_HANDLE_VALUE;
}

::nn::Result SystemDataFile::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    NN_RESULT_DO(SetFilePointer(m_FileHandle, offset));

    NN_RESULT_DO(ReadFile(m_FileHandle, buffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result SystemDataFile::Write(
    int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    NN_RESULT_DO(SetFilePointer(m_FileHandle, offset));

    NN_RESULT_DO(WriteFile(m_FileHandle, buffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result SystemDataFile::Flush() NN_NOEXCEPT
{
    NN_RESULT_DO(FlushFile(m_FileHandle));

    NN_RESULT_SUCCESS;
}

namespace {

::nn::Result InitializeMutex(HANDLE* pOutHandle, const char name[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(name);

    HANDLE handle = ::CreateMutexA(NULL, FALSE, name);

    NN_RESULT_THROW_UNLESS(
        handle != nullptr, ResultWin32ApiFailedToCreateMutex());

    *pOutHandle = handle;

    NN_RESULT_SUCCESS;
}

void FinalizeMutex(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    ::CloseHandle(handle);
}

void LockMutex(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    ::WaitForSingleObject(handle, INFINITE);
}

void UnlockMutex(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    ::ReleaseMutex(handle);
}

::nn::Result CreateFile(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(path);

    HANDLE handle = INVALID_HANDLE_VALUE;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        handle = ::CreateFileA(
            path,
            GENERIC_WRITE,
            0,
            NULL,
            CREATE_NEW,
            FILE_ATTRIBUTE_NORMAL,
            NULL);

        if (handle != INVALID_HANDLE_VALUE)
        {
            *pOutHandle = handle;

            NN_RESULT_SUCCESS;
        }
        else
        {
            const DWORD lastError = ::GetLastError();

            if (lastError == ERROR_FILE_EXISTS ||
                lastError == ERROR_ALREADY_EXISTS)
            {
                NN_RESULT_THROW(::nn::fs::ResultPathAlreadyExists());
            }
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostEntryCorrupted());
}

::nn::Result OpenFileToRead(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(path);

    HANDLE handle = INVALID_HANDLE_VALUE;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        handle = ::CreateFileA(
            path,
            GENERIC_READ,
            FILE_SHARE_READ,
            NULL,
            OPEN_EXISTING,
            FILE_ATTRIBUTE_NORMAL,
            NULL);

        if (handle != INVALID_HANDLE_VALUE)
        {
            *pOutHandle = handle;

            NN_RESULT_SUCCESS;
        }
        else if (::GetLastError() == ERROR_PATH_NOT_FOUND)
        {
            NN_RESULT_THROW(::nn::fs::ResultPathNotFound());
        }
        else if (::GetLastError() == ERROR_FILE_NOT_FOUND)
        {
            NN_RESULT_THROW(::nn::fs::ResultPathNotFound());
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileCorrupted());
}

::nn::Result OpenFileToWrite(HANDLE* pOutHandle, const char path[]) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    NN_SDK_REQUIRES_NOT_NULL(path);

    HANDLE handle = INVALID_HANDLE_VALUE;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        handle = ::CreateFileA(
            path,
            GENERIC_WRITE,
            0,
            NULL,
            OPEN_EXISTING,
            FILE_ATTRIBUTE_NORMAL,
            NULL);

        if (handle != INVALID_HANDLE_VALUE)
        {
            *pOutHandle = handle;

            NN_RESULT_SUCCESS;
        }
        else if (::GetLastError() == ERROR_PATH_NOT_FOUND)
        {
            NN_RESULT_THROW(::nn::fs::ResultPathNotFound());
        }
        else if (::GetLastError() == ERROR_FILE_NOT_FOUND)
        {
            NN_RESULT_THROW(::nn::fs::ResultPathNotFound());
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileCorrupted());
}

void CloseFile(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    ::CloseHandle(handle);
}

::nn::Result SetFilePointer(HANDLE handle, int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0LL);

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        if (::SetFilePointer(
                handle, static_cast<LONG>(offset), NULL, FILE_BEGIN) !=
            INVALID_SET_FILE_POINTER)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileDataCorrupted());
}

::nn::Result SetFileEof(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        if (::SetEndOfFile(handle) == TRUE)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileDataCorrupted());
}

::nn::Result ReadFile(
    HANDLE handle, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const auto size = static_cast<DWORD>(bufferSize);

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        auto dataSize = DWORD();

        if (::ReadFile(handle, buffer, size, &dataSize, NULL) == TRUE)
        {
            NN_RESULT_THROW_UNLESS(
                dataSize == size, ::nn::fs::ResultOutOfRange());

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileDataCorrupted());
}

::nn::Result WriteFile(
    HANDLE handle, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    const auto size = static_cast<DWORD>(bufferSize);

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        auto dataSize = DWORD();

        if (::WriteFile(handle, buffer, size, &dataSize, NULL) == TRUE)
        {
            NN_RESULT_THROW_UNLESS(
                dataSize == size, ::nn::fs::ResultUsableSpaceNotEnough());

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileCorrupted());
}

::nn::Result FlushFile(HANDLE handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(handle);

    for (int i = 0; i < TrialCount; ++i)
    {
        if (0 < i)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        if (::FlushFileBuffers(handle) == TRUE)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(::nn::fs::ResultHostFileCorrupted());
}

} // namespace

}}} // namespace nn::settings::detail
