﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_IFirmwareDebugSettingsServer.sfdl.h>
#include <nn/settings/settings_ISystemSettingsServer.sfdl.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/settings_ServiceTypes.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_Out.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_TypedStorage.h>

#include "settings_FirmwareDebugSettingsServer.h"
#include "settings_SettingsGetterApiImpl.h"
#include "settings_SystemSettingsServer.h"

namespace nn { namespace settings { namespace fwdbg { namespace detail {

struct SettingsItemKeyIteratorImpl final
{
    ISettingsItemKeyIterator* handle;
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    NN_PADDING4;
#endif
    NN_PADDING8;
};

}}}} // namespace nn::settings::fwdbg::detail

namespace nn { namespace settings { namespace detail {

void CopySettingsName(SettingsName* pOutValue, const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(name);
    const size_t length = sizeof(pOutValue->value) - 1;
    ::strncpy(pOutValue->value, name, length);
    pOutValue->value[length] = '\0';
}

void CopySettingsItemKey(SettingsItemKey* pOutValue, const char* key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(key);
    const size_t length = sizeof(pOutValue->value) - 1;
    ::strncpy(pOutValue->value, key, length);
    pOutValue->value[length] = '\0';
}

::nn::Result GetDebugModeFlag(bool* pOutValue) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->GetDebugModeFlag(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSettingsItemValueSize(
    size_t* pOutValue, const char* name, const char* key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(name != nullptr, ResultNullSettingsName());
    NN_RESULT_THROW_UNLESS(key != nullptr, ResultNullSettingsItemKey());

    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));

    SettingsName settingsName = {};
    CopySettingsName(&settingsName, name);

    SettingsItemKey settingsItemKey = {};
    CopySettingsItemKey(&settingsItemKey, key);

    uint64_t value = 0;
    NN_RESULT_DO(
        pProxy->GetSettingsItemValueSize(&value, settingsName, settingsItemKey));
    *pOutValue = static_cast<size_t>(value);

    NN_RESULT_SUCCESS;
}

::nn::Result GetSettingsItemValue(
    size_t* pOutValue, void* buffer, size_t bufferSize,
    const char* name, const char* key) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(name != nullptr, ResultNullSettingsName());
    NN_RESULT_THROW_UNLESS(key != nullptr, ResultNullSettingsItemKey());
    NN_RESULT_THROW_UNLESS(
        buffer != nullptr, ResultNullSettingsItemValueBuffer());

    ::nn::sf::SharedPointer<ISystemSettingsServer> pProxy;
    NN_RESULT_DO(CreateSystemSettingsServerProxy(&pProxy));

    SettingsName settingsName = {};
    CopySettingsName(&settingsName, name);

    SettingsItemKey settingsItemKey = {};
    CopySettingsItemKey(&settingsItemKey, key);

    uint64_t value = 0;
    ::nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(buffer), bufferSize);
    NN_RESULT_DO(
        pProxy->GetSettingsItemValue(
            &value, outBuffer, settingsName, settingsItemKey));
    *pOutValue = static_cast<size_t>(value);

    NN_RESULT_SUCCESS;
}

::nn::Result CreateSettingsItemKeyIterator(
    ::nn::settings::fwdbg::SettingsItemKeyIterator* pOutValue,
    const char* name) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(name != nullptr, ResultNullSettingsName());

    ::nn::sf::SharedPointer<IFirmwareDebugSettingsServer> pProxy;
    NN_RESULT_DO(CreateFirmwareDebugSettingsServerProxy(&pProxy));

    SettingsName settingsName = {};
    CopySettingsName(&settingsName, name);

    ::nn::sf::SharedPointer<ISettingsItemKeyIterator> pointer;
    NN_RESULT_DO(pProxy->CreateSettingsItemKeyIterator(&pointer, settingsName));

    ::nn::util::Get(pOutValue->_impl).handle = pointer.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result DestroySettingsItemKeyIterator(
    ::nn::settings::fwdbg::SettingsItemKeyIterator* pIterator) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pIterator != nullptr, ResultNullSettingsItemKeyIterator());

    ::nn::sf::SharedPointer<ISettingsItemKeyIterator> pointer(
        ::nn::util::Get(pIterator->_impl).handle, false);

    pointer.Reset();

    NN_RESULT_SUCCESS;
}

::nn::Result AdvanceSettingsItemKeyIterator(
    ::nn::settings::fwdbg::SettingsItemKeyIterator* pIterator) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        pIterator != nullptr, ResultNullSettingsItemKeyIterator());

    ::nn::sf::SharedPointer<ISettingsItemKeyIterator> pointer(
        ::nn::util::Get(pIterator->_impl).handle, true);

    NN_RESULT_DO(pointer->GoNext());

    NN_RESULT_SUCCESS;
}

::nn::Result GetSettingsItemKeySize(
    size_t* pOutValue,
    const ::nn::settings::fwdbg::SettingsItemKeyIterator& iterator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nn::sf::SharedPointer<ISettingsItemKeyIterator> pointer(
        ::nn::util::Get(iterator._impl).handle, true);

    auto value = uint64_t();
    NN_RESULT_DO(pointer->GetKeySize(&value));
    *pOutValue = static_cast<size_t>(value);

    NN_RESULT_SUCCESS;
}

::nn::Result GetSettingsItemKey(
    size_t* pOutValue, void* buffer, size_t bufferSize,
    const ::nn::settings::fwdbg::SettingsItemKeyIterator& iterator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_THROW_UNLESS(buffer != nullptr, ResultNullSettingsItemKeyBuffer());

    ::nn::sf::SharedPointer<ISettingsItemKeyIterator> pointer(
        ::nn::util::Get(iterator._impl).handle, true);

    auto value = uint64_t();
    ::nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(buffer), bufferSize);
    NN_RESULT_DO(pointer->GetKey(&value, outBuffer));
    *pOutValue = static_cast<size_t>(value);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::settings::detail
