﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <utility>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_IFirmwareDebugSettingsServer.sfdl.h>
#include <nn/settings/settings_ServiceTypes.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/system/settings_Web.h>

namespace nn { namespace settings { namespace detail {

//!< ファームウェアデバッグ設定サービスのキーイテレータを扱うクラスです。
template<typename T>
class SettingsItemKeyIterator
{
    NN_DISALLOW_COPY(SettingsItemKeyIterator);
    NN_DISALLOW_MOVE(SettingsItemKeyIterator);

private:
    T m_Impl;

public:
    explicit SettingsItemKeyIterator(T&& impl) NN_NOEXCEPT
        : m_Impl(::std::move(impl))
    {
        // 何もしない
    }

    virtual ~SettingsItemKeyIterator() NN_NOEXCEPT
    {
        // 何もしない
    }

    //!< キーイテレータを次へ進めます。
    ::nn::Result GoNext() NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Impl.GoNext());
        NN_RESULT_SUCCESS;
    }

    //!< キーイテレータの指すキーのサイズを取得します。
    ::nn::Result GetKeySize(::nn::sf::Out<uint64_t> outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_Impl.GetKeySize(outValue.GetPointer()));
        NN_RESULT_SUCCESS;
    }

    //!< キーイテレータの指すキーを取得します。
    ::nn::Result GetKey(
        ::nn::sf::Out<uint64_t> outValue, const ::nn::sf::OutBuffer& buffer
        ) NN_NOEXCEPT
    {
        NN_RESULT_DO(
            m_Impl.GetKey(
                outValue.GetPointer(),
                buffer.GetPointerUnsafe(), buffer.GetSize()));
        NN_RESULT_SUCCESS;
    }
};

//!< ファームウェアデバッグ設定サービスを扱うクラスです。
class FirmwareDebugSettingsServer final
{
    NN_DISALLOW_COPY(FirmwareDebugSettingsServer);
    NN_DISALLOW_MOVE(FirmwareDebugSettingsServer);

public:
    FirmwareDebugSettingsServer() NN_NOEXCEPT;

    ::nn::Result SetSettingsItemValue(
        const SettingsName& name, const SettingsItemKey& key,
        const ::nn::sf::InBuffer& value) NN_NOEXCEPT;

    ::nn::Result ResetSettingsItemValue(
        const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT;

    ::nn::Result CreateSettingsItemKeyIterator(
        ::nn::sf::Out<
            ::nn::sf::SharedPointer<ISettingsItemKeyIterator>> outValue,
        const SettingsName& name) NN_NOEXCEPT;

    ::nn::Result ReadSettings(
        ::nn::sf::Out<uint64_t> outCount, const ::nn::sf::OutBuffer& buffer,
        int32_t target) NN_NOEXCEPT;

    ::nn::Result ResetSettings(int32_t target) NN_NOEXCEPT;

    ::nn::Result SetWebInspectorFlag(bool value) NN_NOEXCEPT;

    ::nn::Result SetAllowedSslHosts(
        const ::nn::sf::InArray<::nn::settings::system::AllowedSslHost>&
            values) NN_NOEXCEPT;

    ::nn::Result SetHostFsMountPoint(
        const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT;
};

//!< ファームウェアデバッグ設定サービスのプロクシを作成します。
::nn::Result CreateFirmwareDebugSettingsServerProxy(
    ::nn::sf::SharedPointer<IFirmwareDebugSettingsServer>* pOutValue
    ) NN_NOEXCEPT;

}}} // namespace nn::settings::detail
