﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/settings/settings_ServiceTypes.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nn/TargetConfigs/build_Base.h>

#include "settings_KeyValueStore.h"

namespace nn { namespace settings { namespace detail {

//!< ファームウェアデバッグ設定の設定項目のキーイテレータを扱うクラスです。
class FwdbgSettingsItemKeyIterator final
{
    NN_DISALLOW_COPY(FwdbgSettingsItemKeyIterator);

private:
    FwdbgSettingsItemKeyIterator& operator=(
        FwdbgSettingsItemKeyIterator&&) NN_NOEXCEPT;

private:
    bool m_IsInitialized;               //!< 初期化状態にあるか否か

    KeyValueStoreKeyIterator m_Impl;    //!< 内部実装

public:
    FwdbgSettingsItemKeyIterator() NN_NOEXCEPT;

    FwdbgSettingsItemKeyIterator(
        FwdbgSettingsItemKeyIterator&& other) NN_NOEXCEPT;

    ~FwdbgSettingsItemKeyIterator() NN_NOEXCEPT;

    //!< イテレータを初期化します。
    ::nn::Result Initialize(const SettingsName& name) NN_NOEXCEPT;

    //!< イテレータを次へ進めます。
    ::nn::Result GoNext() NN_NOEXCEPT;

    //!< イテレータの指すキーのサイズを取得します。
    ::nn::Result GetKeySize(uint64_t* pOutValue) NN_NOEXCEPT;

    //!< イテレータの指すキーを取得します。
    ::nn::Result GetKey(
        uint64_t* pOutCount, char* outBuffer, size_t count) NN_NOEXCEPT;

private:
    //!< イテレータを開放します。
    void Finalize() NN_NOEXCEPT;
};

//!< ファームウェアデバッグ設定からデバッグモードフラグを取得します。
::nn::Result GetFwdbgDebugModeFlag(bool* pOutValue) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定から HDMI-CEC 抑止フラグを内部向けに取得します。
::nn::Result GetFwdbgHdmiCecSuppressionFlagForInternal(
    bool* pOutValue) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定の設定項目の値のバイト数を取得します。
::nn::Result GetFwdbgSettingsItemValueSize(
    uint64_t* pOutCount, const SettingsName& name, const SettingsItemKey& key
    ) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定の設定項目の値を取得します。
::nn::Result GetFwdbgSettingsItemValue(
    uint64_t* pOutCount, char* outBuffer, size_t count,
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定の設定項目へ値を設定します。
::nn::Result SetFwdbgSettingsItemValue(
    const SettingsName& name, const SettingsItemKey& key,
    const char* buffer, size_t count) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定の設定項目の値をリセットします。
::nn::Result ResetFwdbgSettingsItemValue(
    const SettingsName& name, const SettingsItemKey& key) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定を読み出します。
::nn::Result ReadFwdbgSettings(
    uint64_t* pOutCount, char outBuffer[], size_t count,
    ::nn::settings::fwdbg::SettingsTarget target) NN_NOEXCEPT;

//!< ファームウェアデバッグ設定をリセットします。
::nn::Result ResetFwdbgSettings(
    ::nn::settings::fwdbg::SettingsTarget target) NN_NOEXCEPT;

}}} // namespace nn::settings::detail
