﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <functional>
#include <mutex>
#include <new>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/settings/system/settings_Ptm.h>
#include <nn/util/util_TypedStorage.h>

#include "settings_DeviceSettingsDatabase.h"
#include "settings_LockableMutexType.h"
#include "settings_Platform.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemSaveData-os.win.h"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2017) && defined(NN_SDK_BUILD_DEBUG)
#undef offsetof
#define offsetof(type, member) \
    (reinterpret_cast<size_t>(&(reinterpret_cast<type*>(0)->member)))
#endif

//!< デバイス設定の警告ログを出力します。
#define NN_SETTINGS_DEVICE_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[device settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< システムセーブデータの識別子
const ::nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000053;

//!< システムセーブデータの合計サイズ
const int64_t SystemSaveDataTotalSize = (256 + 32) << 10;

//!< システムセーブデータのジャーナルサイズ
const int64_t SystemSaveDataJournalSize = (256 + 32) << 10;

//!< システムセーブデータのフラグ集合
const uint32_t SystemSaveDataFlags =
    ::nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveData |
    ::nn::fs::SaveDataFlags_KeepAfterRefurbishment;

//!< マウント名
const char MountName[] = "DeviceSettings";

//!< PTM の電池 LOT のデフォルト値
const ::nn::settings::system::PtmBatteryLot DefaultPtmBatteryLot = {};

//!< PTM の電池残量パラメータのデフォルト値
const ::nn::settings::system::PtmFuelGaugeParameter
    DefaultPtmFuelGaugeParameter = {};

//!< システムセーブデータ操作排他用ミューテックス
LockableMutexType g_SystemSaveDataMutex =
{
    NN_OS_MUTEX_INITIALIZER(false)
};

//!< システムセーブデータから値を取得します。
template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータの値を変更します。
template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT;

//!< システム設定から値を取得します。
template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< システム設定の値を変更します。
template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT;

//!< システムセーブデータから PTM の電池 LOT を取得します。
::nn::Result GetSystemSaveDataPtmBatteryLot(
    ::nn::settings::system::PtmBatteryLot* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの PTM の電池 LOT を変更します。
::nn::Result SetSystemSaveDataPtmBatteryLot(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::PtmBatteryLot& value) NN_NOEXCEPT;

//!< システムセーブデータから PTM の電池残量パラメータを取得します。
::nn::Result GetSystemSaveDataPtmFuelGaugeParameter(
    ::nn::settings::system::PtmFuelGaugeParameter* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの PTM の電池残量パラメータを変更します。
::nn::Result SetSystemSaveDataPtmFuelGaugeParameter(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::PtmFuelGaugeParameter& value) NN_NOEXCEPT;

//!< システムセーブデータ上のデバイス設定をリセットします。
::nn::Result ResetSystemSaveDataDeviceSettings() NN_NOEXCEPT;

} // namespace

::nn::Result GetDeviceSettingsPtmBatteryLot(
    ::nn::settings::system::PtmBatteryLot* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::PtmBatteryLot Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetPtmBatteryLot()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullPtmBatteryLotBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataPtmBatteryLot));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultPtmBatteryLot;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetDeviceSettingsPtmBatteryLot(
    const ::nn::settings::system::PtmBatteryLot& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::PtmBatteryLot Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetPtmBatteryLot()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataPtmBatteryLot, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetDeviceSettingsPtmFuelGaugeParameter(
    ::nn::settings::system::PtmFuelGaugeParameter* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::PtmFuelGaugeParameter Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetPtmFuelGaugeParameter()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullPtmFuelGaugeParameterBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataPtmFuelGaugeParameter));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultPtmFuelGaugeParameter;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetDeviceSettingsPtmFuelGaugeParameter(
    const ::nn::settings::system::PtmFuelGaugeParameter& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::PtmFuelGaugeParameter Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetPtmFuelGaugeParameter()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataPtmFuelGaugeParameter, false));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

namespace {

//!< PTM 設定のデータベースを表す構造体です。
struct PtmSettingsDatabase final
{
    ::nn::settings::system::PtmBatteryLot batteryLot;
    ::nn::settings::system::PtmFuelGaugeParameter fuelGaugeParameter;

    ::nn::Bit8 margin[80];
};

NN_STATIC_ASSERT(sizeof(PtmSettingsDatabase) == 128);

//!< デバイス設定データベースのヘッダを表す構造体です。
struct DeviceSettingsDatabaseHeader final
{
    ::nn::Bit8 margin[16];
};

NN_STATIC_ASSERT(sizeof(DeviceSettingsDatabaseHeader) == 16);

//!< デバイス設定データベースを表す構造体です。
struct DeviceSettingsDatabase final
{
    DeviceSettingsDatabaseHeader header;
    PtmSettingsDatabase ptmSettingsDatabase;

    ::nn::Bit8 margin[262000];
};

NN_STATIC_ASSERT(sizeof(DeviceSettingsDatabase) == 262144);

//!< システムセーブデータを返します。
::nn::Result GetSystemSaveData(SystemSaveData** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemSaveData,
                             sizeof(SystemSaveData),
                         NN_ALIGNOF(SystemSaveData)> s_Storage;

    SystemSaveData& systemSaveData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemSaveData) SystemSaveData();
        systemSaveData.SetSystemSaveDataId(SystemSaveDataId);
        systemSaveData.SetTotalSize(SystemSaveDataTotalSize);
        systemSaveData.SetJournalSize(SystemSaveDataJournalSize);
        systemSaveData.SetFlags(SystemSaveDataFlags);
        systemSaveData.SetMountName(MountName);

        NN_RESULT_DO(systemSaveData.Mount(true));

        s_IsInitialized = true;
    }

    *ppOutValue = &systemSaveData;

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result = getter(pOutValue, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(setter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    const ::nn::Result result = setter(pSystemSaveData, value);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(synchronizes));
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = getter(pOutValue);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataDeviceSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutValue);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_DEVICE_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutValue);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(setter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = setter(value);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataDeviceSettings();

        if (result.IsSuccess())
        {
            result = setter(value);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_DEVICE_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataPtmBatteryLot(
    ::nn::settings::system::PtmBatteryLot* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(DeviceSettingsDatabase, ptmSettingsDatabase) +
            offsetof(PtmSettingsDatabase, batteryLot)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataPtmBatteryLot(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::PtmBatteryLot& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(DeviceSettingsDatabase, ptmSettingsDatabase) +
            offsetof(PtmSettingsDatabase, batteryLot)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataPtmFuelGaugeParameter(
    ::nn::settings::system::PtmFuelGaugeParameter* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(DeviceSettingsDatabase, ptmSettingsDatabase) +
            offsetof(PtmSettingsDatabase, fuelGaugeParameter)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataPtmFuelGaugeParameter(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::PtmFuelGaugeParameter& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(DeviceSettingsDatabase, ptmSettingsDatabase) +
            offsetof(PtmSettingsDatabase, fuelGaugeParameter)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のデバイス設定をゼロ初期化します。
::nn::Result ClearSystemSaveDataDeviceSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    const auto fileSize = static_cast<int32_t>(sizeof(DeviceSettingsDatabase));
    const ::nn::Bit8 zeros[512] = {};
    for (int32_t restSize = fileSize; 0 < restSize; )
    {
        const auto size = ::std::min(
            static_cast<int32_t>(NN_ARRAY_SIZE(zeros)), restSize);
        NN_RESULT_DO(pSystemSaveData->Write(fileSize - restSize, zeros, size));
        restSize -= size;
    }

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のデバイス設定をリセットします。
::nn::Result ResetSystemSaveDataDeviceSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(ClearSystemSaveDataDeviceSettings(pSystemSaveData));
    NN_RESULT_DO(SetSystemSaveDataPtmBatteryLot(
                     pSystemSaveData, DefaultPtmBatteryLot));
    NN_RESULT_DO(SetSystemSaveDataPtmFuelGaugeParameter(
                     pSystemSaveData, DefaultPtmFuelGaugeParameter));

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSaveDataDeviceSettings() NN_NOEXCEPT
{
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    ::nn::Result result = pSystemSaveData->Create(
        static_cast<int64_t>(sizeof(DeviceSettingsDatabase)));
    NN_RESULT_THROW_UNLESS(result.IsSuccess() ||
                           ::nn::fs::ResultPathAlreadyExists::Includes(result),
                           result);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    result = ResetSystemSaveDataDeviceSettings(pSystemSaveData);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(true));
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
