﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_DebugPad.h>

#include "settings_DebugPadImpl.h"
#include "settings_LockableMutexType.h"

namespace nn { namespace settings { namespace detail {

namespace {

//!< DebugPad 用の割り当てを表す構造体です。
struct DebugPadMap final
{
    //!< 汎用コントローラ間の割り当てが初期化済みか否かを表す値
    bool isGenericPadMapInitialized;

    //!< 汎用コントローラ間の割り当て
    DebugPadGenericPadMap genericPadMap;

    //!< キーボード間の割り当て
    DebugPadKeyboardMap keyboardMap;
};

//!< グローバルリソースの操作を排他するミューテックス
LockableMutexType g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

//!< DebugPad 用の割り当て
DebugPadMap g_DebugPadMap;

//!< 汎用コントローラの割り当ての規定値を取得します。
void GetDefaultGenericPadMap(DebugPadGenericPadMap* outValue) NN_NOEXCEPT;

} // namespace

::nn::Result GetDebugPadGenericPadMap(DebugPadGenericPadMap* outValue
                                      ) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ::nn::hid::ResultDebugPadNullGenericPadMapBuffer());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    if (!g_DebugPadMap.isGenericPadMapInitialized)
    {
        GetDefaultGenericPadMap(&g_DebugPadMap.genericPadMap);

        g_DebugPadMap.isGenericPadMapInitialized = true;
    }

    *outValue = g_DebugPadMap.genericPadMap;

    NN_RESULT_SUCCESS;
}

::nn::Result SetDebugPadGenericPadMap(const DebugPadGenericPadMap& value
                                      ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    g_DebugPadMap.genericPadMap = value;

    g_DebugPadMap.isGenericPadMapInitialized = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ResetDebugPadGenericPadMap() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    int32_t genericPadId = g_DebugPadMap.genericPadMap.genericPadId;

    GetDefaultGenericPadMap(&g_DebugPadMap.genericPadMap);

    g_DebugPadMap.genericPadMap.genericPadId = genericPadId;

    g_DebugPadMap.isGenericPadMapInitialized = true;

    NN_RESULT_SUCCESS;
}

::nn::Result GetDebugPadKeyboardMap(DebugPadKeyboardMap* outValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(outValue != nullptr,
                           ::nn::hid::ResultDebugPadNullKeyboardMapBuffer());

    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    *outValue = g_DebugPadMap.keyboardMap;

    NN_RESULT_SUCCESS;
}

::nn::Result SetDebugPadKeyboardMap(const DebugPadKeyboardMap& value
                                    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    g_DebugPadMap.keyboardMap = value;

    NN_RESULT_SUCCESS;
}

::nn::Result ResetDebugPadKeyboardMap() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(g_Mutex)> locker(g_Mutex);

    g_DebugPadMap.keyboardMap = DebugPadKeyboardMap();

    NN_RESULT_SUCCESS;
}

namespace {

void GetDefaultGenericPadMap(DebugPadGenericPadMap* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    outValue->genericPadId = 0;
    outValue->buttonA = GenericPadButton_4;
    outValue->buttonB = GenericPadButton_3;
    outValue->buttonX = GenericPadButton_2;
    outValue->buttonY = GenericPadButton_1;
    outValue->buttonL = GenericPadButton_5;
    outValue->buttonR = GenericPadButton_6;
    outValue->buttonZL = GenericPadButton_7;
    outValue->buttonZR = GenericPadButton_8;
    outValue->buttonStart = GenericPadButton_10;
    outValue->buttonSelect = GenericPadButton_9;
    outValue->buttonLeft = GenericPadButton_Left;
    outValue->buttonUp = GenericPadButton_Up;
    outValue->buttonRight = GenericPadButton_Right;
    outValue->buttonDown = GenericPadButton_Down;
    outValue->axisAnalogStickRX = GenericPadAxis_Z;
    outValue->axisAnalogStickRY = GenericPadAxis_R;
    outValue->axisAnalogStickLX = GenericPadAxis_X;
    outValue->axisAnalogStickLY = GenericPadAxis_Y;
    outValue->attributes.Reset();
    outValue->attributes.Set<GenericPadAxisAttribute::IsXAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsYAxisInverted>(true);
    outValue->attributes.Set<GenericPadAxisAttribute::IsZAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsRAxisInverted>(true);
    outValue->attributes.Set<GenericPadAxisAttribute::IsUAxisInverted>(false);
    outValue->attributes.Set<GenericPadAxisAttribute::IsVAxisInverted>(false);
}

} // namespace

}}} // namespace nn::settings::detail
