﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <new>
#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_ResultPrivate.h>
#include <nn/settings/detail/settings_Log.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_Uuid.h>

#include "settings_ApplicationSettingsDatabase.h"
#include "settings_LockableMutexType.h"
#include "settings_Platform.h"
#include "settings_SystemSettingsDatabase.h"
#include "settings_Uuid.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "settings_SystemSaveData-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "settings_SystemSaveData-os.win.h"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2017) && defined(NN_SDK_BUILD_DEBUG)
#undef offsetof
#define offsetof(type, member) \
    (reinterpret_cast<size_t>(&(reinterpret_cast<type*>(0)->member)))
#endif

//!< アプリケーション設定の警告ログを出力します。
#define NN_SETTINGS_APPLN_WARN(...) \
    NN_DETAIL_SETTINGS_WARN("[application settings] Warning: " __VA_ARGS__)

namespace nn { namespace settings { namespace detail {

namespace {

//!< システムセーブデータの識別子
const ::nn::fs::SystemSaveDataId SystemSaveDataId = 0x8000000000000054;

//!< システムセーブデータの合計サイズ
const int64_t SystemSaveDataTotalSize = (128 + 32) << 10;

//!< システムセーブデータのジャーナルサイズ
const int64_t SystemSaveDataJournalSize = (128 + 32) << 10;

//!< システムセーブデータのフラグ集合
const uint32_t SystemSaveDataFlags =
    ::nn::fs::SaveDataFlags_KeepAfterResettingSystemSaveDataWithoutUserSaveData;

//!< マウント名
const char MountName[] = "ApplnSettings";

//!< サービスディスカバリの制御設定のデフォルト値
const ::nn::settings::system::ServiceDiscoveryControlSettings
    DefaultServiceDiscoveryControlSettings = {};

//!< 修理中か否かを表す値のデフォルト値
const bool DefaultInRepairProcessFlag = false;

//!< システムセーブデータ操作排他用ミューテックス
LockableMutexType g_SystemSaveDataMutex =
{
    NN_OS_MUTEX_INITIALIZER(false)
};

//!< システムセーブデータから値を取得します。
template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT;

//!< システムセーブデータの値を変更します。
template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT;

//!< システム設定から値を取得します。
template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT;

//!< システム設定の値を変更します。
template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT;

//!< システムセーブデータから Mii 作者 ID を取得します。
::nn::Result GetSystemSaveDataMiiAuthorId(
    ::nn::util::Uuid* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの Mii 作者 ID を変更します。
::nn::Result SetSystemSaveDataMiiAuthorId(
    SystemSaveData* pSystemSaveData,
    const ::nn::util::Uuid& value) NN_NOEXCEPT;

//!< システムセーブデータのサービスディスカバリの制御設定を取得します。
::nn::Result GetSystemSaveDataServiceDiscoveryControlSettings(
    ::nn::settings::system::ServiceDiscoveryControlSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータのサービスディスカバリの制御設定を変更します。
::nn::Result SetSystemSaveDataServiceDiscoveryControlSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ServiceDiscoveryControlSettings& value) NN_NOEXCEPT;

//!< システムセーブデータから修理中か否かを表す値を取得します。
::nn::Result GetSystemSaveDataInRepairProcessFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT;

//!< システムセーブデータの修理中か否かを表す値を変更します。
::nn::Result SetSystemSaveDataInRepairProcessFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT;

//!< システムセーブデータ上のアプリケーション設定をリセットします。
::nn::Result ResetSystemSaveDataApplicationSettings() NN_NOEXCEPT;

} // namespace

::nn::Result GetApplicationSettingsMiiAuthorId(
    ::nn::util::Uuid* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::util::Uuid Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetMiiAuthorId()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullMiiAuthorIdBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataMiiAuthorId));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                *pOutValue = GenerateUuid();
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetApplicationSettingsServiceDiscoveryControlSettings(
    ::nn::settings::system::ServiceDiscoveryControlSettings* pOutValue) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ServiceDiscoveryControlSettings Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "GetServiceDiscoveryControlSettings()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(
                    pOutValue != nullptr,
                    ResultNullServiceDiscoveryControlSettingsBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue,
                        GetSystemSaveDataServiceDiscoveryControlSettings));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                NN_SDK_REQUIRES_NOT_NULL(pOutValue);
                *pOutValue = DefaultServiceDiscoveryControlSettings;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetApplicationSettingsServiceDiscoveryControlSettings(
    const ::nn::settings::system::ServiceDiscoveryControlSettings& value) NN_NOEXCEPT
{
    typedef ::nn::settings::system::ServiceDiscoveryControlSettings Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetServiceDiscoveryControlSettings()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value,
                        SetSystemSaveDataServiceDiscoveryControlSettings,
                        true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result GetApplicationSettingsInRepairProcessFlag(bool* pOutValue) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        GetSystemSettingsValue<Type>(
            pOutValue,
            "IsInRepairProcess()",
            [] (Type* pOutValue) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                                       ResultNullIsInRepairBuffer());
                NN_RESULT_DO(
                    GetSystemSaveDataValue<Type>(
                        pOutValue, GetSystemSaveDataInRepairProcessFlag));
                NN_RESULT_SUCCESS;
            },
            [] (Type* pOutValue) NN_NOEXCEPT
            {
                if (pOutValue != nullptr)
                {
                    *pOutValue = DefaultInRepairProcessFlag;
                }
            }));
    NN_RESULT_SUCCESS;
}

::nn::Result SetApplicationSettingsInRepairProcessFlag(bool value) NN_NOEXCEPT
{
    typedef bool Type;
    NN_RESULT_DO(
        SetSystemSettingsValue<Type>(
            value,
            "SetInRepairProcessEnabled()",
            [] (const Type& value) NN_NOEXCEPT -> ::nn::Result
            {
                NN_RESULT_DO(
                    SetSystemSaveDataValue<Type>(
                        value, SetSystemSaveDataInRepairProcessFlag, true));
                NN_RESULT_SUCCESS;
            }));
    NN_RESULT_SUCCESS;
}

namespace {

//!< Mii 設定のデータベースを表す構造体です。
struct MiiSettingsDatabase final
{
    ::nn::util::Uuid miiAuthorId;

    ::nn::Bit8 margin[48];
};

NN_STATIC_ASSERT(sizeof(MiiSettingsDatabase) == 64);

//!< アプリケーション設定データベースのヘッダを表す構造体です。
struct ApplicationSettingsDatabaseHeader final
{
    ::nn::Bit8 margin[16];
};

NN_STATIC_ASSERT(sizeof(ApplicationSettingsDatabaseHeader) == 16);

//!< サービスディスカバリ設定を表す構造体です。
struct ServiceDiscoverySettings final
{
    ::nn::settings::system::ServiceDiscoveryControlSettings controlSettings;

    ::nn::Bit8 margin[28];
};

NN_STATIC_ASSERT(sizeof(ServiceDiscoverySettings) == 32);

//!< 修理工程で設定するフラグのデータベースを表す構造体です。
struct RepairSettingsDatabase final
{
    uint32_t requiresRunRepairTimeReviser;
    uint32_t inRepairProcessFlag;

    ::nn::Bit8 margin[56];
};

NN_STATIC_ASSERT(sizeof(RepairSettingsDatabase) == 64);

//!< アプリケーション設定データベースを表す構造体です。
struct ApplicationSettingsDatabase final
{
    ApplicationSettingsDatabaseHeader header;
    MiiSettingsDatabase miiSettingsDatabase;
    ServiceDiscoverySettings serviceDiscoverySettings;
    RepairSettingsDatabase repairSettingsDatabase;

    ::nn::Bit8 margin[130896];
};

NN_STATIC_ASSERT(sizeof(ApplicationSettingsDatabase) == 131072);

//!< システムセーブデータを返します。
::nn::Result GetSystemSaveData(SystemSaveData** ppOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(ppOutValue);

    static ::nn::util::TypedStorage<SystemSaveData,
                             sizeof(SystemSaveData),
                         NN_ALIGNOF(SystemSaveData)> s_Storage;

    SystemSaveData& systemSaveData = ::nn::util::Get(s_Storage);

    static bool s_IsInitialized = false;

    if (!s_IsInitialized)
    {
        new(&systemSaveData) SystemSaveData();
        systemSaveData.SetSystemSaveDataId(SystemSaveDataId);
        systemSaveData.SetTotalSize(SystemSaveDataTotalSize);
        systemSaveData.SetJournalSize(SystemSaveDataJournalSize);
        systemSaveData.SetFlags(SystemSaveDataFlags);
        systemSaveData.SetMountName(MountName);

        NN_RESULT_DO(systemSaveData.Mount(true));

        s_IsInitialized = true;
    }

    *ppOutValue = &systemSaveData;

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSaveDataValue(
    T* pOutValue,
    ::std::function<::nn::Result(T*, SystemSaveData*) NN_NOEXCEPT
        > getter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToRead());
    const ::nn::Result result = getter(pOutValue, pSystemSaveData);
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSaveDataValue(
    const T& value,
    ::std::function<::nn::Result(SystemSaveData*, const T&) NN_NOEXCEPT
        > setter,
    bool synchronizes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(setter);
    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    const ::nn::Result result = setter(pSystemSaveData, value);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(synchronizes));
    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result GetSystemSettingsValue(
    T* pOutValue,
    const char* const functionName,
    ::std::function<::nn::Result(T*) NN_NOEXCEPT> getter,
    ::std::function<void(T*) NN_NOEXCEPT> initializer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(getter);
    NN_SDK_REQUIRES_NOT_NULL(initializer);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = getter(pOutValue);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataApplicationSettings();

        if (result.IsSuccess())
        {
            result = getter(pOutValue);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_APPLN_WARN(
            "%s failed. (%08x, %d%03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            ErrorCodePlatformId,
            result.GetModule(), result.GetDescription());

        initializer(pOutValue);
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

template<typename T>
::nn::Result SetSystemSettingsValue(
    const T& value,
    const char* const functionName,
    ::std::function<::nn::Result(const T&) NN_NOEXCEPT> setter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(functionName);
    NN_SDK_REQUIRES_NOT_NULL(setter);

    ::std::lock_guard<decltype(g_SystemSaveDataMutex)
                      > locker(g_SystemSaveDataMutex);

    ::nn::Result result = setter(value);

    if (::nn::fs::ResultPathNotFound::Includes(result))
    {
        result = ResetSystemSaveDataApplicationSettings();

        if (result.IsSuccess())
        {
            result = setter(value);
        }
    }

    if (result.IsFailure())
    {
        NN_SETTINGS_APPLN_WARN(
            "%s failed. (%08x, %03d-%04d)\n",
            functionName,
            result.GetInnerValueForDebug(),
            result.GetModule(), result.GetDescription());
    }

    NN_UNUSED(functionName);

    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataMiiAuthorId(
    ::nn::util::Uuid* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, miiSettingsDatabase) +
            offsetof(MiiSettingsDatabase, miiAuthorId)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataMiiAuthorId(
    SystemSaveData* pSystemSaveData,
    const ::nn::util::Uuid& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, miiSettingsDatabase) +
            offsetof(MiiSettingsDatabase, miiAuthorId)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataServiceDiscoveryControlSettings(
    ::nn::settings::system::ServiceDiscoveryControlSettings* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, serviceDiscoverySettings)),
        pOutValue, sizeof(*pOutValue)));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataServiceDiscoveryControlSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::settings::system::ServiceDiscoveryControlSettings& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, serviceDiscoverySettings)),
        &value, sizeof(value)));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSystemSaveDataInRepairProcessFlag(
    bool* pOutValue,
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    auto inRepairFlag = int32_t();
    NN_RESULT_DO(pSystemSaveData->Read(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, repairSettingsDatabase) +
            offsetof(RepairSettingsDatabase, inRepairProcessFlag)),
        &inRepairFlag, sizeof(inRepairFlag)));
    *pOutValue = (inRepairFlag == 1);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSystemSaveDataInRepairProcessFlag(
    SystemSaveData* pSystemSaveData,
    const bool& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);
    const auto inRepairFlag = static_cast<int32_t>(value);
    NN_RESULT_DO(pSystemSaveData->Write(
        static_cast<int64_t>(
            offsetof(ApplicationSettingsDatabase, repairSettingsDatabase) +
            offsetof(RepairSettingsDatabase, inRepairProcessFlag)),
        &inRepairFlag, sizeof(inRepairFlag)));
    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のアプリケーション設定をゼロ初期化します。
::nn::Result ClearSystemSaveDataApplicationSettings(
    SystemSaveData* pSystemSaveData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    const auto fileSize =
        static_cast<int32_t>(sizeof(ApplicationSettingsDatabase));
    const ::nn::Bit8 zeros[512] = {};
    for (int32_t restSize = fileSize; 0 < restSize; )
    {
        const int32_t size =
            ::std::min(static_cast<int32_t>(NN_ARRAY_SIZE(zeros)), restSize);
        NN_RESULT_DO(pSystemSaveData->Write(fileSize - restSize, zeros, size));
        restSize -= size;
    }

    NN_RESULT_SUCCESS;
}

//!< システムセーブデータ上のアプリケーション設定をリセットします。
::nn::Result ResetSystemSaveDataApplicationSettings(
    SystemSaveData* pSystemSaveData,
    const ::nn::util::Uuid& miiAuthorId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemSaveData);

    NN_RESULT_DO(ClearSystemSaveDataApplicationSettings(pSystemSaveData));
    NN_RESULT_DO(SetSystemSaveDataMiiAuthorId(pSystemSaveData, miiAuthorId));
    NN_RESULT_DO(SetSystemSaveDataServiceDiscoveryControlSettings(pSystemSaveData, DefaultServiceDiscoveryControlSettings));

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSystemSaveDataApplicationSettings() NN_NOEXCEPT
{
    ::nn::util::Uuid miiAuthorId = {};
    NN_RESULT_DO(GetSystemSettingsMiiAuthorId(&miiAuthorId));

    SystemSaveData* pSystemSaveData = nullptr;
    NN_RESULT_DO(GetSystemSaveData(&pSystemSaveData));
    NN_SDK_ASSERT_NOT_NULL(pSystemSaveData);
    ::nn::Result result = pSystemSaveData->Create(
        static_cast<int64_t>(sizeof(ApplicationSettingsDatabase)));
    NN_RESULT_THROW_UNLESS(result.IsSuccess() ||
                           ::nn::fs::ResultPathAlreadyExists::Includes(result),
                           result);
    NN_RESULT_DO(pSystemSaveData->OpenToWrite());
    result = ResetSystemSaveDataApplicationSettings(
        pSystemSaveData, miiAuthorId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pSystemSaveData->Flush());
    pSystemSaveData->Close();
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), result);
    NN_RESULT_DO(pSystemSaveData->Commit(true));
    NN_RESULT_SUCCESS;
}

} // namespace

}}} // namespace nn::settings::detail
