﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/pwm/server/pwm_ManagerImpl.h>

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/pwm/driver/pwm.h>
#include <nn/pwm/driver/pwm_ChannelDev.h> // OpenSessionForDev

#include <nn/pwm/pwm_IManager.sfdl.h>
#include <nn/pwm/pwm_IChannelSession.sfdl.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace pwm { namespace server {

// IChannelSession 実装クラス
class ChannelSessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、値によって、OpenSession / ForDev を呼び分ける
    ChannelSessionImpl(ManagerImpl* pParent, int number, bool isForDev) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        if(isForDev)
        {
            nn::pwm::driver::OpenSessionForDev(&m_ChannelSession, number);
        }
        else
        {
            nn::pwm::driver::OpenSession(&m_ChannelSession, static_cast<nn::pwm::ChannelName>(number));
        }
    }

    // Session のデストラクタで Session のクローズを行う
    ~ChannelSessionImpl()
    {
        nn::pwm::driver::CloseSession(&m_ChannelSession);
    }

    // PWM ドライバライブラリの API
    nn::Result SetPeriod(nn::TimeSpanType period) NN_NOEXCEPT;
    nn::Result GetPeriod(nn::sf::Out<nn::TimeSpanType> outPeriod) NN_NOEXCEPT;
    nn::Result SetDuty(int duty) NN_NOEXCEPT;
    nn::Result GetDuty(nn::sf::Out<int> outDuty) NN_NOEXCEPT;
    nn::Result SetEnabled(bool enable) NN_NOEXCEPT;
    nn::Result GetEnabled(nn::sf::Out<bool> outEnable) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl>  m_Parent;

    // PWM ドライバライブラリ向けのセッション構造体
    nn::pwm::driver::ChannelSession     m_ChannelSession;
};



// ここから各 PWM API の内部実装
nn::Result ChannelSessionImpl::SetPeriod(nn::TimeSpanType period) NN_NOEXCEPT
{
    nn::TimeSpan periodClass(period);
    nn::pwm::driver::SetPeriod(&m_ChannelSession, period);
    return nn::ResultSuccess();
}

nn::Result ChannelSessionImpl::GetPeriod(nn::sf::Out<nn::TimeSpanType> outPeriod) NN_NOEXCEPT
{
    nn::TimeSpan periodClass = nn::pwm::driver::GetPeriod(&m_ChannelSession);
    *outPeriod = periodClass;
    return nn::ResultSuccess();
}

nn::Result ChannelSessionImpl::SetDuty(int duty) NN_NOEXCEPT
{
    nn::pwm::driver::SetDuty(&m_ChannelSession, duty);
    return nn::ResultSuccess();
}

nn::Result ChannelSessionImpl::GetDuty(nn::sf::Out<int> outDuty) NN_NOEXCEPT
{
    *outDuty = nn::pwm::driver::GetDuty(&m_ChannelSession);
    return nn::ResultSuccess();
}

nn::Result ChannelSessionImpl::SetEnabled(bool enable) NN_NOEXCEPT
{
    nn::pwm::driver::SetEnabled(&m_ChannelSession, enable);
    return nn::ResultSuccess();
}

nn::Result ChannelSessionImpl::GetEnabled(nn::sf::Out<bool> outEnable) NN_NOEXCEPT
{
    *outEnable = nn::pwm::driver::GetEnabled(&m_ChannelSession);
    return nn::ResultSuccess();
}

// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::OpenSessionForDev(nn::sf::Out<nn::sf::SharedPointer<nn::pwm::IChannelSession>> outSession, std::int32_t channel) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<nn::pwm::IChannelSession, ChannelSessionImpl>(&m_Allocator, this, static_cast<int>(channel), true);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::pwm::IChannelSession>> outSession, std::int32_t channel) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<nn::pwm::IChannelSession, ChannelSessionImpl>(&m_Allocator, this, static_cast<int>(channel), false);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}
