﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pwm/pwm_Lib.h>
#include <nn/pwm/pwm_Channel.h>
#include <nn/pwm/pwm_ChannelDev.h>

#include <nn/pwm/pwm_IManager.sfdl.h> // for IManager
#include "pwm_GetManagerByHipc.h"

namespace nn {
namespace pwm {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
// DFC と HIPC とで共通でよい。
nn::sf::SharedPointer<IManager> g_Manager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

inline IChannelSession* GetInterface(ChannelSession* pSession) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSession->_sessionHandle);
    return static_cast<nn::pwm::IChannelSession*>(pSession->_sessionHandle);
}

}

void Initialize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if (g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(!g_Manager);

        g_Manager = nn::pwm::GetManagerByHipc();
    }
    g_InitializeCount++;
}

void InitializeWith(nn::sf::SharedPointer<nn::pwm::IManager> manager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_Manager = manager;

    g_InitializeCount++;
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if(g_InitializeCount == 0)
    {
        NN_SDK_ASSERT(g_Manager);
        // 共有ポインタへの nullptr の代入で解放できる。
        g_Manager = nullptr;
    }
}

// SF を外に出さないために外部にはハンドルとして渡すようにラップする。
void OpenSessionForDev(ChannelSession* pOutSession, int channelIndex) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);
    nn::sf::SharedPointer<nn::pwm::IChannelSession> session;
    auto result = g_Manager->OpenSessionForDev(&session, static_cast<std::int32_t>(channelIndex));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
}
void OpenSession(ChannelSession* pOutSession, ChannelName name) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_Manager);
    nn::sf::SharedPointer<nn::pwm::IChannelSession> session;
    auto result = g_Manager->OpenSession(&session, static_cast<std::int32_t>(name));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutSession->_sessionHandle = session.Detach();
}
void CloseSession(ChannelSession* pSession) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(GetInterface(pSession));
}
void SetPeriod(ChannelSession* pSession, nn::TimeSpan period) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetPeriod(period);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
nn::TimeSpan GetPeriod(ChannelSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    nn::TimeSpanType outValue;

    auto result = GetInterface(pSession)->GetPeriod(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}
void SetDuty(ChannelSession* pSession, int duty) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetDuty(duty);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
int GetDuty(ChannelSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    int outValue;

    auto result = GetInterface(pSession)->GetDuty(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}
void SetEnabled(ChannelSession* pSession, bool enable) NN_NOEXCEPT
{
    auto result = GetInterface(pSession)->SetEnabled(enable);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}
bool GetEnabled(ChannelSession* pSession) NN_NOEXCEPT
{
    // 出力用の変数
    bool outValue;

    auto result = GetInterface(pSession)->GetEnabled(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}

} // pwm
} // nn
