﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   PMC レジスタ操作ユーティリティ関数
 * @detail  セキュリティの観点から、PMC は nn::dd::QueryIoMappingAddress() によるマッピングが使えないので、
 *          nn::dd::ReadIoRegister() / nn::dd::WriteIoRegister() を使うユーティリティ関数を提供する。
 */

#pragma once

#include <nn/dd.h>

namespace nn {
namespace pwm {
namespace driver {
namespace detail {

// address に指定したレジスタの読み出しを実行します。
// TegraX1 では レジスタの書き込み完了に使用します。
NN_FORCEINLINE void DummyRead(nn::dd::PhysicalAddress address)
{
    nn::dd::ReadIoRegister(address);
}

NN_FORCEINLINE void WriteMasked32(nn::dd::PhysicalAddress address, Bit32 value, Bit32 mask)
{
    auto temp = nn::dd::ReadIoRegister(address);
    nn::dd::WriteIoRegister(address, (temp & ~mask) | (value & mask));
}

NN_FORCEINLINE void Write32(nn::dd::PhysicalAddress address, uint32_t val) NN_NOEXCEPT
{
    nn::dd::WriteIoRegister(address, val);
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOn32(nn::dd::PhysicalAddress address, uint32_t flags) NN_NOEXCEPT
{
    auto temp = nn::dd::ReadIoRegister(address);
    temp |= flags;
    nn::dd::WriteIoRegister(address, temp);
}

// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void SetBitOff32(nn::dd::PhysicalAddress address, uint32_t flags) NN_NOEXCEPT
{
    auto temp = nn::dd::ReadIoRegister(address);
    temp &= ~flags;
    nn::dd::WriteIoRegister(address, temp);
}

// Overwrite partly on specified register
// Note: Both RW happens on specified register. Don't use on RO/WO register!
NN_FORCEINLINE void Modify32(nn::dd::PhysicalAddress address, uint32_t value, uint32_t mask) NN_NOEXCEPT
{
    auto temp = nn::dd::ReadIoRegister(address);
    temp &= ~mask;
    temp |= value;
    nn::dd::WriteIoRegister(address, temp);
}

NN_FORCEINLINE bool GetBit32(nn::dd::PhysicalAddress address, uint32_t flags) NN_NOEXCEPT
{
    auto temp = nn::dd::ReadIoRegister(address);
    temp &= flags;
    return (temp != 0);
}

NN_FORCEINLINE uint32_t Read32(nn::dd::PhysicalAddress address) NN_NOEXCEPT
{
    return nn::dd::ReadIoRegister(address);
}

} // detail
} // driver
} // pwm
} // nn
