﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/psm/driver/psm.h>
#include <nn/psm/psm_PsmServer.h>
#include <nn/psm/psm_ResultPrivate.h>
#include <nn/psm/psm_SystemTypes.h>
#include <nn/psm/psm_SystemProcessTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>

namespace nn { namespace psm {

class PsmServer::PsmSession
{
    NN_DISALLOW_COPY(PsmSession);
    NN_DISALLOW_MOVE(PsmSession);

public:
    PsmSession(
        PsmServer* pParent,
        ::nn::psm::driver::Session internalSession) NN_NOEXCEPT
        : m_Parent(pParent, true)
        , m_InternalSession(internalSession)
    {
        // 何もしない
    }

    ~PsmSession() NN_NOEXCEPT
    {
        ::nn::psm::driver::CloseSession(m_InternalSession);
    }

    ::nn::Result BindStateChangeEvent(
        ::nn::sf::Out<::nn::sf::NativeHandle> outHandle) NN_NOEXCEPT;

    ::nn::Result UnbindStateChangeEvent() NN_NOEXCEPT;

    ::nn::Result SetChargerTypeChangeEventEnabled(
        bool isEnabled) NN_NOEXCEPT;

    ::nn::Result SetPowerSupplyChangeEventEnabled(
        bool isEnabled) NN_NOEXCEPT;

    ::nn::Result SetBatteryVoltageStateChangeEventEnabled(
        bool isEnabled) NN_NOEXCEPT;

private:
    ::nn::sf::SharedPointer<PsmServer> m_Parent;

    ::nn::psm::driver::Session m_InternalSession;

    ::nn::os::SystemEvent m_SystemEvent;
};

::nn::Result PsmServer::PsmSession::BindStateChangeEvent(
    ::nn::sf::Out<::nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::psm::driver::BindStateChangeEvent(
            m_SystemEvent.GetBase(),
            m_InternalSession));

    ::nn::os::NativeHandle internalHandle =
          ::nn::os::GetReadableHandleOfSystemEvent(m_SystemEvent.GetBase());
    // 寿命管理しない
    *outHandle = ::nn::sf::NativeHandle(internalHandle, false);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::PsmSession::UnbindStateChangeEvent() NN_NOEXCEPT
{
    return ::nn::psm::driver::UnbindStateChangeEvent(m_InternalSession);
}

::nn::Result PsmServer::PsmSession::SetChargerTypeChangeEventEnabled(
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::psm::driver::SetChargerTypeChangeEventEnabled(
            m_InternalSession,
            isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::PsmSession::SetPowerSupplyChangeEventEnabled(
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::psm::driver::SetPowerSupplyChangeEventEnabled(
            m_InternalSession,
            isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::PsmSession::SetBatteryVoltageStateChangeEventEnabled(
    bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(
        ::nn::psm::driver::SetBatteryVoltageStateChangeEventEnabled(
            m_InternalSession,
            isEnabled));

    NN_RESULT_SUCCESS;
}

PsmServer::PsmServer() NN_NOEXCEPT
{
    m_HeapHandle = ::nn::lmem::CreateExpHeap(
        &m_HeapBuffer,
        sizeof(m_HeapBuffer),
        ::nn::lmem::CreationOption_NoOption);

    m_Allocator.Attach(m_HeapHandle);
}

PsmServer::~PsmServer() NN_NOEXCEPT
{
    ::nn::lmem::DestroyExpHeap(m_HeapHandle);
}

::nn::Result PsmServer::GetBatteryChargePercentage(
    ::nn::sf::Out<int> outBatteryCharge) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outBatteryCharge.GetPointer() != nullptr,
        ResultPsmNullPointer());

    int batteryCharge;
    NN_RESULT_DO(::nn::psm::driver::GetBatteryChargePercentage(&batteryCharge));
    outBatteryCharge.Set(batteryCharge);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetRawBatteryChargePercentage(
    ::nn::sf::Out<double> outRawBatteryCharge) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outRawBatteryCharge.GetPointer() != nullptr,
        ResultPsmNullPointer());

    double rawBatteryCharge;
    NN_RESULT_DO(::nn::psm::driver::GetRawBatteryChargePercentage(&rawBatteryCharge));
    outRawBatteryCharge.Set(rawBatteryCharge);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetBatteryAgePercentage(
    ::nn::sf::Out<double> outAge) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outAge.GetPointer() != nullptr,
        ResultPsmNullPointer());

    double age;
    NN_RESULT_DO(::nn::psm::driver::GetBatteryAgePercentage(&age));
    outAge.Set(age);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetChargerType(
    ::nn::sf::Out<int> outChargerType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outChargerType.GetPointer() != nullptr,
        ResultPsmNullPointer());

    ChargerType chargerType;
    NN_RESULT_DO(::nn::psm::driver::GetChargerType(&chargerType));
    outChargerType.Set(static_cast<int>(chargerType));

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetBatteryVoltageState(
    ::nn::sf::Out<int> outBatteryVoltageState) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outBatteryVoltageState.GetPointer() != nullptr,
        ResultPsmNullPointer());

    BatteryVoltageState batteryVoltageState;
    NN_RESULT_DO(::nn::psm::driver::GetBatteryVoltageState(&batteryVoltageState));
    outBatteryVoltageState.Set(static_cast<int>(batteryVoltageState));

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::EnableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::EnableBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::DisableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::DisableBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::IsBatteryChargingEnabled(
    ::nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outIsEnabled.GetPointer() != nullptr,
        ResultPsmNullPointer());

    bool isEnabled;
    NN_RESULT_DO(::nn::psm::driver::IsBatteryChargingEnabled(&isEnabled));
    outIsEnabled.Set(isEnabled);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::AcquireControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::AcquireControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::ReleaseControllerPowerSupply() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::ReleaseControllerPowerSupply());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::OpenSession(
    ::nn::sf::Out<nn::sf::SharedPointer<nn::psm::IPsmSession>> outSession) NN_NOEXCEPT
{
    ::nn::psm::driver::Session internalSession;
    NN_RESULT_DO(::nn::psm::driver::OpenSession(&internalSession));

    typedef ::nn::sf::ObjectFactory<Allocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<IPsmSession, PsmSession>(
        &m_Allocator,
        this,
        internalSession);

    *outSession = ::std::move(p);
    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::EnableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::EnableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::DisableEnoughPowerChargeEmulation() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::DisableEnoughPowerChargeEmulation());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::EnableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::EnableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::DisableFastBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::psm::driver::DisableFastBatteryCharging());

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::IsEnoughPowerSupplied(
    ::nn::sf::Out<bool> outIsEnoughPowerSupplied) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        outIsEnoughPowerSupplied.GetPointer() != nullptr,
        ResultPsmNullPointer());

    bool isEnoughPowerSupplied;
    NN_RESULT_DO(
        ::nn::psm::driver::IsEnoughPowerSupplied(&isEnoughPowerSupplied));
    outIsEnoughPowerSupplied.Set(isEnoughPowerSupplied);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetBatteryChargeInfoEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    ::nn::os::SystemEventType* pSystemEvent = nullptr;

    NN_RESULT_DO(::nn::psm::driver::GetBatteryChargeInfoEvent(&pSystemEvent));

    ::nn::os::NativeHandle internalHandle = ::nn::os::GetReadableHandleOfSystemEvent(pSystemEvent);

    // 寿命管理しない
    outHandle.Set(::nn::sf::NativeHandle(internalHandle, false));

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetBatteryChargeInfoFields(nn::sf::Out<nn::psm::BatteryChargeInfoFields> outBatteryChargeInfoFields) NN_NOEXCEPT
{
    BatteryChargeInfoFields fields;

    NN_RESULT_DO(::nn::psm::driver::GetBatteryChargeInfoFields(&fields));

    outBatteryChargeInfoFields.Set(fields);

    NN_RESULT_SUCCESS;
}

::nn::Result PsmServer::GetBatteryChargeCalibratedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    ::nn::os::SystemEventType* pSystemEvent = nullptr;

    NN_RESULT_DO(::nn::psm::driver::GetBatteryChargeCalibratedEvent(&pSystemEvent));

    ::nn::os::NativeHandle internalHandle = ::nn::os::GetReadableHandleOfSystemEvent(pSystemEvent);

    // 寿命管理しない
    outHandle.Set(::nn::sf::NativeHandle(internalHandle, false));

    NN_RESULT_SUCCESS;
}

}} // namespace nn::psm
